//
//  TCGBAuthAdapterDelegate.h
//  Gamebase
//
//  Created by NHN on 2016. 11. 16..
//  © NHN Corp. All rights reserved.
//

#import <UIKit/UIKit.h>

#ifndef TCGBAuthAdapterDelegate_h
#define TCGBAuthAdapterDelegate_h

NS_ASSUME_NONNULL_BEGIN

@class TCGBProviderAuthCredential;
@class TCGBAuthProviderProfile;
@class TCGBError;
@class TCGBAuthAdapterAdditionalInfo;

/** The protocol TCGBAuthAdapterDelegate is for developing Auth Adapter.<br/>
 This protocol contains several methods such as to login, logout and so on.
 */
@protocol TCGBAuthAdapterDelegate <NSObject>

@optional
@property (nonatomic, strong, nullable) TCGBError *initializeResult;

@required
/**---------------------------------------------------------------------------------------
 * @name Authentication API
 *  ---------------------------------------------------------------------------------------
 */

- (instancetype)initWithAdditionalInfo:(TCGBAuthAdapterAdditionalInfo *)additionalInfo;

/** Login to identity service provider.
 
 @param sender sender instance
 @param target target instance
 @param additionalInfo If identity provider needs to set something when it is initialized, this additionanlInfo is set.
 @param viewController presenting view controller
 @param completion completion handler when auth process(login, logout, withdraw) is completed.
 */
- (void)loginWithSender:(NSObject *)sender
                 target:(id<TCGBAuthAdapterDelegate>)target
         additionalInfo:(nullable NSDictionary<NSString *, id> *)additionalInfo
         viewController:(UIViewController *)viewController
             completion:(void(^)(TCGBProviderAuthCredential * _Nullable  credential, TCGBError * _Nullable error))completion;

/** Logout to identity service provider.
 
 @param sender sender instance
 @param target target instance
 @param additionalInfo If identity provider needs to set something when it is initialized, this additionanlInfo is set.
 @param completion completion handler when auth process(login, logout, withdraw) is completed.
 */
- (void)logoutWithSender:(NSObject *)sender
                  target:(id<TCGBAuthAdapterDelegate>)target
          additionalInfo:(nullable NSDictionary<NSString *, id> *)additionalInfo
              completion:(void(^)(TCGBError * _Nullable error))completion;

/** Withdrwal to identity service provider.
 
 @param sender sender instance
 @param target target instance
 @param additionalInfo If identity provider needs to set something when it is initialized, this additionanlInfo is set.
 @param completion completion handler when auth process(login, logout, withdraw) is completed.
 */
- (void)withdrawWithSender:(NSObject *)sender
                    target:(id<TCGBAuthAdapterDelegate>)target
            additionalInfo:(nullable NSDictionary<NSString *, id> *)additionalInfo
                completion:(void(^)(TCGBError * _Nullable error))completion;


@optional // when IDP do not support mapping
/** Mapping to identity service provider.
 
 @param sender sender instance
 @param target target instance
 @param additionalInfo If identity provider needs to set something when it is initialized, this additionanlInfo is set.
 @param completion completion handler when auth process(login, logout, withdraw) is completed.
 */
- (void)addMappingWithSender:(id)sender
                      target:(id<TCGBAuthAdapterDelegate>)target
              additionalInfo:(nullable NSDictionary<NSString *, id> *)additionalInfo
              viewController:(nullable UIViewController *)viewController
                  completion:(void(^)(id  _Nullable credential, TCGBError * _Nullable error))completion;

/**---------------------------------------------------------------------------------------
 * @name Provider information
 *  ---------------------------------------------------------------------------------------
 */
@optional // gettingIDP auth infos after logging in
/** 
 @return Provider's authenticated credential
 */
- (nullable TCGBProviderAuthCredential *)authInfosFromProvider;

@optional
- (nullable NSString *)accessToken;
- (nullable NSString *)userID;
- (void)setProfileWithUserInfo:(NSDictionary *)userInfo;
- (nullable TCGBAuthProviderProfile *)profile;

/**---------------------------------------------------------------------------------------
 * @name Adapter information
 *  ---------------------------------------------------------------------------------------
 */
@required
/**
 @return semantic version string of Auth Adapter
 */
+ (NSString *)versionString;

/**
 @return Provider's name
 */
- (NSString *)providerName;

#pragma mark - System Events
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdeprecated-declarations"
/**---------------------------------------------------------------------------------------
 * @name Application life cycle for identify provider's sdk setting
 *  ---------------------------------------------------------------------------------------
 */

@optional
/// Managing State Transitions
// launch time
- (BOOL)application:(UIApplication *)application willFinishLaunchingWithOptions:(NSDictionary *)launchOptions;
- (BOOL)application:(UIApplication *)application didFinishLaunchingWithOptions:(nullable NSDictionary<UIApplicationLaunchOptionsKey, id> *)launchOptions;
// transitioning to foreground
- (void)applicationDidBecomeActive:(UIApplication *)application;
// transitioning to background
- (void)applicationDidEnterBackground:(UIApplication *)application;
// transitioning  to the inactive state
- (void)applicationWillResignActive:(UIApplication *)application;
- (void)applicationWillEnterForeground:(UIApplication *)application;
// termination
- (void)applicationWillTerminate:(UIApplication *)application;

/// Responding to Notifications and Events
// When a remote notification arrives, the system calls the
- (void)application:(UIApplication *)application didReceiveRemoteNotification:(NSDictionary *)userInfo fetchCompletionHandler:(void (^)(UIBackgroundFetchResult result))completionHandler;
// When a local notification fires, the system calls the
- (void)application:(UIApplication *)application didReceiveLocalNotification:(UILocalNotification *)notification;
// When the user TCGBs a custom action in the alert for a remote or local notification’s, the system calls the
- (void)application:(UIApplication *)application handleActionWithIdentifier:(nullable NSString *)identifier forRemoteNotification:(NSDictionary *)userInfo completionHandler:(void (^)(void))completionHandler;
- (void)application:(UIApplication *)application handleActionWithIdentifier:(nullable NSString *)identifier forLocalNotification:(UILocalNotification *)notification completionHandler:(void (^)(void))completionHandler;
// For apps that want to initiate background downloads, the system calls the
- (void)application:(UIApplication *)application performFetchWithCompletionHandler:(void (^)(UIBackgroundFetchResult result))completionHandler;
// For apps that use the NSURLSession class to perform background downloads, the system calls the
- (void)application:(UIApplication *)application handleEventsForBackgroundURLSession:(NSString *)identifier completionHandler:(void (^)(void))completionHandler;
// When a low-memory condition occurs, the system notifies the app delegate by calling its
- (void)applicationDidReceiveMemoryWarning:(UIApplication *)application;
//When a significant change in time occurs, the system notifies the app delegate by calling its
- (void)applicationSignificantTimeChange:(UIApplication *)application;
// When the user locks the device, the system calls the app delegate’s
- (void)applicationProtectedDataWillBecomeUnavailable:(UIApplication *)application;
// you can reestablish your references to the data in the app delegate’s
- (void)applicationProtectedDataDidBecomeAvailable:(UIApplication *)application;


/// handle OpenURL
- (BOOL)application:(UIApplication *)app openURL:(NSURL *)url options:(NSDictionary<UIApplicationOpenURLOptionsKey,id> *)options; // available iOS9 and later
- (BOOL)application:(UIApplication *)application openURL:(NSURL *)url sourceApplication:(nullable NSString *)sourceApplication annotation:(id)annotation; // available iOS 4.2 ~ iOS8(deprecated at 9.0)
- (BOOL)application:(UIApplication *)application handleOpenURL:(NSURL *)url; // available iOS2 ~ iOS8(deprecated at 9.0)

- (BOOL)application:(UIApplication *)application continueUserActivity:(NSUserActivity *)userActivity restorationHandler:(void (^)(NSArray<id<UIUserActivityRestoring>> * _Nullable))restorationHandler;

- (void)application:(UIApplication *)application didDiscardSceneSessions:(NSSet<UISceneSession *> *)sceneSessions API_AVAILABLE(ios(13.0));
- (void)scene:(UIScene *)scene willConnectToSession:(UISceneSession *)session options:(UISceneConnectionOptions *)connectionOptions API_AVAILABLE(ios(13.0));
- (void)sceneDidDisconnect:(UIScene *)scene API_AVAILABLE(ios(13.0));
- (void)sceneDidBecomeActive:(UIScene *)scene API_AVAILABLE(ios(13.0));
- (void)sceneWillResignActive:(UIScene *)scene API_AVAILABLE(ios(13.0));
- (void)sceneWillEnterForeground:(UIScene *)scene API_AVAILABLE(ios(13.0));
- (void)sceneDidEnterBackground:(UIScene *)scene API_AVAILABLE(ios(13.0));

/// handle OpenURL
- (void)scene:(UIScene *)scene openURLContexts:(NSSet<UIOpenURLContext *> *)URLContexts API_AVAILABLE(ios(13.0)); // available iOS 13 and later

#pragma clang diagnostic pop

@end

NS_ASSUME_NONNULL_END

#endif /* TCGBAuthAdapterDelegate_h */
