#include "NhnWebViewModule.h"

#include "NhnWebViewDefines.h"
#include "NhnWebViewErrorCode.h"
#include "Windows/NhnCefWindowsInvoker.h"

void FNhnWebViewModule::StartupModule()
{
    Caller = MakeShared<FNhnCefWindowsInvoker, ESPMode::ThreadSafe>();
}

void FNhnWebViewModule::ShutdownModule()
{
    if (Caller.IsValid())
    {
        Caller = nullptr;
    }
}

void FNhnWebViewModule::Initialize(const FString& Locale, const FCefWebViewResultDelegate& Callback)
{
    if (bIsInitialized)
    {
        if (Callback.ExecuteIfBound(FNhnWebViewError(NhnWebViewErrorCode::Success)) == false)
        {
            NHNWEBVIEW_LOG_DEBUG("WebView callback is null. Already initialize.");
        }
        
        return;
    }

    bIsInitialized = Caller->Initialize(Locale);

    const FNhnWebViewError Result = FNhnWebViewError(bIsInitialized ? NhnWebViewErrorCode::Success : NhnWebViewErrorCode::InternalError);
    if (Callback.ExecuteIfBound(Result) == false)
    {
        NHNWEBVIEW_LOG_WARNING("WebView callback is null (%s)", *Result.ToString());
    }
}

bool FNhnWebViewModule::IsInitialized() const
{
    return bIsInitialized;
}

void FNhnWebViewModule::CreateWebView(const FCreateParams& Configuration, const FCefWebViewDelegateWebViewInfo& Callback)
{
    if (bIsInitialized == false)
    {
        NHNWEBVIEW_LOG_ERROR("Not initialized.");
        Callback.ExecuteIfBound(nullptr, FNhnWebViewError(NhnWebViewErrorCode::NotInitialized));
        return;
    }

    const int32 WebIndex = AddWebView();
    const TSharedPtr<FNhnCefWebView> WebViewPtr = GetWebView(WebIndex);
    if (WebViewPtr.IsValid())
    {
        WebViewPtr->CreateWebView(WebIndex, Configuration, Callback);
    }
}

void FNhnWebViewModule::ShowWebView(const FShowParams& Params)
{
    if (CanInvoke(Params.OpenDelegate, Params.Index) == false)
    {
        return;
    }

    const TSharedPtr<FNhnCefWebView> WebViewPtr = GetWebView(Params.Index);
    if (WebViewPtr.IsValid())
    {
        WebViewPtr->ShowWebView(Params);
    }
}

void FNhnWebViewModule::HideWebView(const int32 WebIndex, const FCefWebViewResultDelegate& Callback)
{
    if (CanInvoke(Callback, WebIndex) == false)
    {
        return;
    }
    
    const TSharedPtr<FNhnCefWebView> WebViewPtr = GetWebView(WebIndex);
    if (WebViewPtr.IsValid())
    {
        WebViewPtr->HideWebView();
        Callback.ExecuteIfBound(FNhnWebViewError(NhnWebViewErrorCode::Success));
    }
}

void FNhnWebViewModule::RemoveWebView(const int32 WebIndex, const FCefWebViewResultDelegate& Callback)
{
    if (CanInvoke(Callback, WebIndex) == true)
    {
        RemoveWebView(WebIndex);
        Callback.ExecuteIfBound(FNhnWebViewError(NhnWebViewErrorCode::Success));
    }
}

void FNhnWebViewModule::ResizeWebView(const int32 WebIndex, const int32 Width, const int32 Height, const FCefWebViewResultDelegate& Callback)
{
    if (CanInvoke(Callback, WebIndex) == true)
    {
        RemoveWebView(WebIndex);
        Callback.ExecuteIfBound(FNhnWebViewError(NhnWebViewErrorCode::Success));
    }
}

void FNhnWebViewModule::ShowScrollBar(const int32 WebIndex, bool bIsShow, const FCefWebViewResultDelegate& Callback)
{
    if (CanInvoke(Callback, WebIndex) == true)
    {
        const TSharedPtr<FNhnCefWebView> WebViewPtr = GetWebView(WebIndex);
        if (WebViewPtr.IsValid())
        {
            WebViewPtr->ShowScrollBar(bIsShow);
        Callback.ExecuteIfBound(FNhnWebViewError(NhnWebViewErrorCode::Success));
        }
    }
}

void FNhnWebViewModule::GoHome(const int32 WebIndex, const FCefWebViewResultDelegate& Callback)
{
    if (CanInvoke(Callback, WebIndex) == true)
    {
        const TSharedPtr<FNhnCefWebView> WebViewPtr = GetWebView(WebIndex);
        if (WebViewPtr.IsValid())
        {
            WebViewPtr->GoHome();
            Callback.ExecuteIfBound(FNhnWebViewError(NhnWebViewErrorCode::Success));
        }
    }
}

void FNhnWebViewModule::GoBack(const int32 WebIndex, const FCefWebViewResultDelegate& Callback)
{
    if (CanInvoke(Callback, WebIndex) == true)
    {
        const TSharedPtr<FNhnCefWebView> WebViewPtr = GetWebView(WebIndex);
        if (WebViewPtr.IsValid())
        {
            WebViewPtr->GoBack();
            Callback.ExecuteIfBound(FNhnWebViewError(NhnWebViewErrorCode::Success));
        }
    }
}

bool FNhnWebViewModule::CanGoBack(const int32 WebIndex)
{
    if (IsInitialized())
    {
        const TSharedPtr<FNhnCefWebView> WebViewPtr = GetWebView(WebIndex);
        if (WebViewPtr.IsValid())
        {
            return WebViewPtr->CanGoBack();
        }
    }

    return false;
}

void FNhnWebViewModule::GoForward(const int32 WebIndex, const FCefWebViewResultDelegate& Callback)
{
    if (CanInvoke(Callback, WebIndex) == true)
    {
        const TSharedPtr<FNhnCefWebView> WebViewPtr = GetWebView(WebIndex);
        if (WebViewPtr.IsValid())
        {
            WebViewPtr->GoForward();
            Callback.ExecuteIfBound(FNhnWebViewError(NhnWebViewErrorCode::Success));
        }
    }
}

bool FNhnWebViewModule::CanGoForward(const int32 WebIndex)
{
    if (IsInitialized())
    {
        const TSharedPtr<FNhnCefWebView> WebViewPtr = GetWebView(WebIndex);
        if (WebViewPtr.IsValid())
        {
            return WebViewPtr->CanGoForward();
        }
    }

    return false;
}

void FNhnWebViewModule::ExecuteJavaScript(const int32 WebIndex, const FString& JavaScript, const FCefWebViewResultDelegate& Callback)
{
    if (CanInvoke(Callback, WebIndex) == true)
    {
        const TSharedPtr<FNhnCefWebView> WebViewPtr = GetWebView(WebIndex);
        if (WebViewPtr.IsValid())
        {
            WebViewPtr->ExecuteJavaScript(JavaScript);
            Callback.ExecuteIfBound(FNhnWebViewError(NhnWebViewErrorCode::Success));
        }
    }
}

void FNhnWebViewModule::SetInvalidRedirectUrlScheme(TArray<FString> SchemeList, const FCefWebViewResultDelegate& Callback)
{
    if (IsInitialize(Callback) == true)
    {
        FString SchemeString = TEXT("");
        const int Count = SchemeList.Num();
        for (int32 i = 0; i < Count; ++i)
        {
            SchemeString += SchemeList[i];
            if (i < Count - 1)
            {
                SchemeString += TEXT(";");
            }
        }
        FNhnCefWebView::ReserveInvalidRedirectUrlSchemes(SchemeString);
        Callback.ExecuteIfBound(FNhnWebViewError(NhnWebViewErrorCode::Success));
    }
}

void FNhnWebViewModule::SetDownloadCompleteOption(const int Option)
{
    FNhnCefWebView::SetDownloadCompleteOption(Option);
}

void FNhnWebViewModule::SetDebugEnable(const bool bEnable)
{
    FNhnCefWebView::SetDebugEnable(bEnable);
}

void FNhnWebViewModule::InputWeb(const int32 WebIndex, const int32 Flags, const int32 X, const int32 Y)
{
    FNhnCefWebView::InputWeb(WebIndex, Flags, X, Y);
}

int32 FNhnWebViewModule::AddWebView()
{
    const TSharedPtr<FNhnCefWebView> WebViewPtr = MakeShared<FNhnCefWebView>();
    int32 WebIndex = GetWebViewIndex(WebViewPtr);
    if (WebIndex < 0)
    {
        Index++;
        WebViewDictionary.Add(Index, WebViewPtr);
        WebIndex = Index;
    }
    
    return WebIndex;
}

void FNhnWebViewModule::RemoveWebView(const int32 WebIndex)
{
    if (WebViewDictionary.Contains(WebIndex) == true)
    {
        TSharedPtr<FNhnCefWebView> WebViewPtr = GetWebView(WebIndex);
        WebViewPtr.Reset();
        WebViewDictionary.Remove(WebIndex);
    }
}

TSharedPtr<FNhnCefWebView> FNhnWebViewModule::GetWebView(const int32 WebIndex) const
{
    return WebViewDictionary.FindRef(WebIndex);
}

int32 FNhnWebViewModule::GetWebViewIndex(const TSharedPtr<FNhnCefWebView>& FindWebView) const
{
    for (const auto& WebView : WebViewDictionary)
    {
        if (WebView.Value == FindWebView)
        {
            return WebView.Key;
        }
    }
    
    return -1;
}

int32 FNhnWebViewModule::GetWebViewCount() const
{
    return WebViewDictionary.Num();
}

bool FNhnWebViewModule::CanInvoke(const FCefWebViewResultDelegate& Callback, const int32 WebIndex) const
{
    if (IsInitialize(Callback) == false)
    {
        NHNWEBVIEW_LOG_ERROR("Not initialized.");
        return false;
    }

    if (HasWebView(WebIndex, Callback) == false)
    {
        NHNWEBVIEW_LOG_ERROR("There is no callback.");
        return false;
    }

    return true;
}

bool FNhnWebViewModule::IsInitialize(const FCefWebViewResultDelegate& Callback) const
{
    if (bIsInitialized == false)
    {
        NHNWEBVIEW_LOG_WARNING("CEF is not initialized.");
        Callback.ExecuteIfBound(FNhnWebViewError(NhnWebViewErrorCode::NotInitialized));
    }

    return bIsInitialized;
}

bool FNhnWebViewModule::HasWebView(const int32 WebIndex, const FCefWebViewResultDelegate& Callback) const
{
    if (GetWebView(WebIndex) == nullptr)
    {
        NHNWEBVIEW_LOG_WARNING("The webview could not be found. (WebIndex: %d)", WebIndex);
        Callback.ExecuteIfBound(FNhnWebViewError(NhnWebViewErrorCode::NotFound));
        
        return false;
    }

    return true;
}

IMPLEMENT_MODULE(FNhnWebViewModule, NHNWebView);
