﻿#include "NhnWebViewManager.h"

#include "NhnWebViewDefines.h"
#include "NhnWebViewConfiguration.h"
#include "NhnWebViewError.h"
#include "NhnWebViewModule.h"
#include "NhnWebViewSubsystem.h"
#include "NhnWebViewErrorCode.h"
#include "Internal/NhnCefRequestVo.h"
#include "Internal/NhnCefResponseVo.h"
#include "Internal/NhnCefTypes.h"
#include "Engine/UserInterfaceSettings.h"
#include "Widgets/SNhnWebView.h"
#include "Widgets/SNhnWebViewDialog.h"
#include "Widgets/SNhnWebViewPopup.h"

#define CEFWEBVIEW_DEBUG_ENABLE

namespace 
{
    constexpr int32 DefaultNavigationBarHeight = 60;
    
    void GetWebViewRect(const FNhnWebViewConfiguration& Configuration, const FVector2D& ViewportSize, FVector2D& GetViewportSize, FVector2D& GetPosition)
    {
        int NavigationBarHeight = DefaultNavigationBarHeight;
        if (Configuration.NavigationBarHeight > -1)
        {
            NavigationBarHeight = Configuration.NavigationBarHeight;
        }

        FVector2D CalculateViewportSize(ViewportSize);
    
        const float DPIValue = GetDefault<UUserInterfaceSettings>(UUserInterfaceSettings::StaticClass())->GetDPIScaleBasedOnSize(FIntPoint(CalculateViewportSize.X, CalculateViewportSize.Y));
        CalculateViewportSize.X /= DPIValue;
        CalculateViewportSize.Y /= DPIValue;
        
        if (Configuration.ViewportSize == FVector2D::ZeroVector)
        {
            GetViewportSize = FVector2D(CalculateViewportSize.X, CalculateViewportSize.Y - NavigationBarHeight);
            GetPosition = FVector2D(0, NavigationBarHeight);
        }
        else
        {
            GetViewportSize = FVector2D(Configuration.ViewportSize.X, Configuration.ViewportSize.Y - NavigationBarHeight);
            GetPosition = FVector2D(CalculateViewportSize.X * 0.5f - Configuration.ViewportSize.X * 0.5f,
                CalculateViewportSize.Y * 0.5f - Configuration.ViewportSize.Y * 0.5f + NavigationBarHeight);
        }
    }
    
    FCreateParams GetCefConfiguration(const FNhnWebViewConfiguration& InConfiguration, const FVector2D& ViewportSize)
    {
        FCreateParams CefConfiguration;
        CefConfiguration.BgType = CefBgType::Opaque;
        GetWebViewRect(InConfiguration, ViewportSize, CefConfiguration.Size, CefConfiguration.Position);
        CefConfiguration.PopupOption.Type = CefPopupType::Redirect;
        CefConfiguration.PopupOption.BlockMessage = TEXT("팝업을 지원하지 않습니다.");
        CefConfiguration.PopupIcon = InConfiguration.PopupIcon;
        return CefConfiguration;
    }
}

FNhnWebViewManager::FNhnWebViewManager(const TWeakObjectPtr<UNhnWebViewSubsystem>& Subsystem)
    : Subsystem(Subsystem), ZOrder(10)
{
}

FNhnWebViewManager::~FNhnWebViewManager()
{
}

void FNhnWebViewManager::ShowWebView(const FString& Url, const FNhnWebViewConfiguration& Configuration, const FNhnWebViewEventListener& EventParams)
{
    using namespace NhnCef;
    
    CloseCallback = EventParams.CloseCallback;
    SchemeList = EventParams.SchemeList;
    SchemeEvent = EventParams.SchemeCallback;

    const auto ProcessShow = [this, Configuration, Url, EventParams]()
    {
        if (SchemeList.Num() > 0)
        {
            FNhnWebViewModule::Get().SetInvalidRedirectUrlScheme(SchemeList, FCefWebViewResultDelegate::CreateLambda([](const FNhnWebViewError& Error)
                {
                    if (Error.Code != NhnWebViewErrorCode::Success)
                    {
                        NHNWEBVIEW_LOG_ERROR("SetInvalidRedirectUrlScheme %s", *Error.ToString());
                    }
                }));
        }

        FVector2D CurrentViewportSize;
        if (const UGameViewportClient* ViewportClient = Subsystem->GetGameInstance()->GetGameViewportClient())
        {
            ViewportClient->GetViewportSize(CurrentViewportSize);
        }

        const FCreateParams CefConfiguration = GetCefConfiguration(Configuration, CurrentViewportSize);

        FNhnWebViewModule::Get().CreateWebView(CefConfiguration, FCefWebViewDelegateWebViewInfo::CreateLambda([&](const FWebViewInfo* Info, const FNhnWebViewError&)
        {
            if (Info == nullptr)
            {
                NHNWEBVIEW_LOG_ERROR("info is nullptr.");
                return;
            }

            const auto Webview = CreateWebView(Info->WebIndex, Configuration);
            if (Webview == nullptr)
            {
                NHNWEBVIEW_LOG_ERROR("unable to create WebView UI.");
                return;
            }

            FShowParams ShowParams;
            ShowParams.Index = Webview->WebIndex;
            ShowParams.Url = Url;
            ShowParams.bShowScrollBar = true;
            
            ShowParams.OpenDelegate = FCefWebViewResultDelegate::CreateLambda([&EventParams](const FNhnWebViewError& Error)
            {
                EventParams.OpenCallback.ExecuteIfBound(Error);
            });
            
            ShowParams.TitleDelegate = FNhnCefWebViewTitleDelegate::CreateLambda([Webview](const FString& Title)
            {
                if (Webview->WebViewSlate != nullptr)
                {
                    Webview->WebViewSlate->ChangeTitle(Title);    
                }
            });
            
            ShowParams.UrlDelegate = FNhnCefWebViewUrlDelegate::CreateLambda([&](const FString& CurrentUrl)
            {
                for (auto& Scheme : SchemeList)
                {
                    if (CurrentUrl.Contains(Scheme))
                    {
                        SchemeEvent.ExecuteIfBound(CurrentUrl, FNhnWebViewError(NhnWebViewErrorCode::Success));
                        break;
                    }
                }
            });

            ShowParams.InputFocusDelegate = FNhnCefWebViewInputFocusDelegate::CreateLambda([](const int32 InputFocus)
            {
                NHNWEBVIEW_LOG_DEBUG("callback inputFocus : %d", InputFocus);
            });

            ShowParams.StatusDelegate = FCefWebViewStatusDelegate::CreateLambda([this](FWebViewStatus& Status)
            {
                switch (Status.Status)
                {
                    case CefWebUpdateStatus::JsDialog:
                        {
                            OpenDialog(Status.WebIndex, Status.JSDialog);
                            break;
                        }
                    case CefWebUpdateStatus::PopupBlock:
                        {
                            Status.JSDialog.Type = CefJsDialogType::Alert;
                            Status.JSDialog.Message = Status.PopupBlock.Message;
                            OpenDialog(Status.WebIndex, Status.JSDialog);
                            break;
                        }
                    default: ;
                }
            });
            
            FNhnWebViewModule::Get().ShowWebView(ShowParams);
        }));
    };

    if (FNhnWebViewModule::Get().IsInitialized())
    {
        ProcessShow();
    }
    else
    {
        FNhnWebViewModule::Get().Initialize(Configuration.Locale, FCefWebViewResultDelegate::CreateLambda([&ProcessShow](const FNhnWebViewError& Error)
        {
            if (Error.Code == NhnWebViewErrorCode::InternalError)
            {
                NHNWEBVIEW_LOG_ERROR("Initialize error.");
                return;
            }

#ifdef CEFWEBVIEW_DEBUG_ENABLE
            FNhnWebViewModule::Get().SetDebugEnable(true);
#endif
        
            FNhnWebViewModule::Get().SetDownloadCompleteOption(CefDownloadCompleteOption::OpenFile);

            ProcessShow();
        }));
    }
}

void FNhnWebViewManager::CloseWebView(const bool bIsExit)
{
    TArray<int32> WebViewIndices;
    WebViewContainer.GetKeys(WebViewIndices);
    for (const auto Index : WebViewIndices)
    {
        RemoveWebView(Index, bIsExit);
    }
    
    CloseCallback.ExecuteIfBound(FNhnWebViewError(NhnWebViewErrorCode::Success));
}

bool FNhnWebViewManager::ExistWebView() const
{
    return WebViewContainer.Num() > 0;
}

void FNhnWebViewManager::OpenDialog(const int32 Index, const NhnCef::FJsDialog& Dialog)
{
    if (WebViewContainer.Contains(Index) == false)
    {
        return;
    }

    const auto Info = WebViewContainer.Find(Index);
    if (Info->JsDialogSlate.IsValid() == true)
    {
        return;
    }
    
    Info->JsDialogSlate = SNew(SNhnWebViewDialog)
        .Index(Index)
        .JsDialogType(Dialog.Type)
        .JsDialogMessage(Dialog.Message)
        .OnCloseDialog(this, &FNhnWebViewManager::OnCloseDialog);
    
    check(Subsystem.IsValid() && Subsystem->GetGameInstance());

    UGameViewportClient* ViewportClient = Subsystem->GetGameInstance()->GetGameViewportClient();
    if (ViewportClient == nullptr)
    {
        NHNWEBVIEW_LOG_ERROR("Not found viewport. %d", Index);
        return;
    }
    
    ViewportClient->AddViewportWidgetContent(Info->JsDialogSlate.ToSharedRef(), GetNextOrder());
}

int32 FNhnWebViewManager::GetNextOrder()
{
    return ZOrder++;
}

void FNhnWebViewManager::OnCloseWebView(const int32 Index)
{
    CloseWebView(false);
}

void FNhnWebViewManager::OnCloseDialog(const int32 Index, const bool bIsOkay)
{
    if (WebViewContainer.Contains(Index) == false)
    {
        return;
    }
    
    const auto Info = WebViewContainer.Find(Index);
    if (Info->JsDialogSlate.IsValid() == false)
    {
        return;
    }

    FNhnWebViewModule::Get().InputWeb(Index, CefWebInput::JsDialog, bIsOkay, 0);
    
    check(Subsystem.IsValid() && Subsystem->GetGameInstance());

    UGameViewportClient* ViewportClient = Subsystem->GetGameInstance()->GetGameViewportClient();
    if (ViewportClient == nullptr)
    {
        NHNWEBVIEW_LOG_ERROR("Not found viewport. %d", Index);
        return;
    }
    
    ViewportClient->RemoveViewportWidgetContent(Info->JsDialogSlate.ToSharedRef());
    Info->JsDialogSlate = nullptr;
}

const FNhnWebViewInformation* FNhnWebViewManager::CreateWebView(const int32 WebIndex, const FNhnWebViewConfiguration& InConfiguration)
{
    if (WebViewContainer.Contains(WebIndex))
    {
        NHNWEBVIEW_LOG_ERROR("Contains webview index : %d", WebIndex);
        return nullptr;
    }
    
    check(Subsystem.IsValid() && Subsystem->GetGameInstance());

    UGameViewportClient* ViewportClient = Subsystem->GetGameInstance()->GetGameViewportClient();
    if (ViewportClient == nullptr)
    {
        NHNWEBVIEW_LOG_ERROR("Not found viewport. %d", WebIndex);
        return nullptr;
    }

    FNhnWebViewInformation Info(WebIndex);

    switch (InConfiguration.Type)
    {
        case Window:
        default:
            Info.WebViewSlate = SNew(SNhnWebView)
                .GameViewport(Subsystem->GetGameInstance()->GetGameViewportClient())
                .WebIndex(WebIndex)
                .Title(InConfiguration.Title)
                .ViewportSize(InConfiguration.ViewportSize)
                .ShowNavigationBar(InConfiguration.NavigationBarVisible)
                .TitleColor(InConfiguration.TitleColor)
                .NavigationBarColor(InConfiguration.NavigationBarColor)
                .NavigationBarIconTintColor(InConfiguration.NavigationBarIconTintColor)
                .NavigationBarHeight(InConfiguration.NavigationBarHeight < 0 ? DefaultNavigationBarHeight : InConfiguration.NavigationBarHeight)
                .ShowBackButton(InConfiguration.BackButtonVisible)
                .ShowForwardButton(InConfiguration.ForwardButtonVisible)
                .ShowAlwaysNavigationButtons(InConfiguration.NavigationButtonsAlwaysVisible)
                .BackgroundColor(InConfiguration.BackgroundColor);

            ViewportClient->AddViewportWidgetContent(Info.WebViewSlate.ToSharedRef(), GetNextOrder());
            break;
        case FloatingPopup:
            Info.WebViewPopupSlate = SNew(SNhnWebViewPopup)
                .GameViewport(Subsystem->GetGameInstance()->GetGameViewportClient())
                .WebIndex(WebIndex)
                .ViewportSize(InConfiguration.ViewportSize)
                .BackgroundColor(InConfiguration.BackgroundColor);

            ViewportClient->AddViewportWidgetContent(Info.WebViewPopupSlate.ToSharedRef(), GetNextOrder());
            break;
    }
    
    WebViewContainer.Add(WebIndex, Info);
    
    return WebViewContainer.Find(WebIndex);
}

bool FNhnWebViewManager::RemoveWebView(const int32 Index, bool bIsExit)
{
    if (WebViewContainer.Contains(Index) == false)
    {
        return false;
    }
    
    check(Subsystem.IsValid() && Subsystem->GetGameInstance());

    UGameViewportClient* ViewportClient = Subsystem->GetGameInstance()->GetGameViewportClient();
    
    const auto Info = WebViewContainer.Find(Index);
    const bool bIsExistGameViewport = ViewportClient != nullptr;
    if (Info->WebViewSlate.IsValid() == true)
    {
        if (bIsExistGameViewport)
        {
            ViewportClient->RemoveViewportWidgetContent(Info->WebViewSlate.ToSharedRef());
        }
        Info->WebViewSlate = nullptr;
    }

    if (Info->WebViewPopupSlate.IsValid() == true)
    {
        if (bIsExistGameViewport)
        {
            ViewportClient->RemoveViewportWidgetContent(Info->WebViewPopupSlate.ToSharedRef());
        }
        Info->WebViewPopupSlate = nullptr;
    }
    
    if (Info->JsDialogSlate.IsValid() == true)
    {
        if (bIsExistGameViewport)
        {
            ViewportClient->RemoveViewportWidgetContent(Info->JsDialogSlate.ToSharedRef());
        }
        Info->JsDialogSlate = nullptr;
    }
    
    WebViewContainer.Remove(Index);
    
    FNhnWebViewModule::Get().RemoveWebView(Index, FCefWebViewResultDelegate::CreateLambda([](const FNhnWebViewError& Error)
        {
            if (Error.Code != NhnWebViewErrorCode::Success)
            {
                NHNWEBVIEW_LOG_ERROR("RemoveWebView %s", *Error.ToString());
            }
        }));

    return true;
}
