#include "NhnWebViewViewport.h"

#include "NhnCefTypes.h"
#include "NhnCefWebView.h"
#include "NhnWebViewImeHandler.h"
#include "NhnWebViewModule.h"

enum ECefKeyEvent
{
    KEYEVENT_RAWKEYDOWN = 0,
    KEYEVENT_KEYDOWN,
    KEYEVENT_KEYUP,
    KEYEVENT_CHAR
};

enum ECefEventFlag
{
    EVENTFLAG_NONE = 0,
    EVENTFLAG_CAPS_LOCK_ON = 1 << 0,
    EVENTFLAG_SHIFT_DOWN = 1 << 1,
    EVENTFLAG_CONTROL_DOWN = 1 << 2,
    EVENTFLAG_ALT_DOWN = 1 << 3,
    EVENTFLAG_LEFT_MOUSE_BUTTON = 1 << 4,
    EVENTFLAG_MIDDLE_MOUSE_BUTTON = 1 << 5,
    EVENTFLAG_RIGHT_MOUSE_BUTTON = 1 << 6,
};

FNhnWebViewViewport* WebViewViewportRef = nullptr;

bool OnNativeKeyEvent(int Type, unsigned int Modifiers, int KeyCode, wchar_t Character)
{
    if (WebViewViewportRef != nullptr)
    {
        WebViewViewportRef->SetKeyEvent(Type, Modifiers, KeyCode, Character);
        return true;
    }
    return false;
}

void OnImeCompositionRangeChanged(const bool bIsRangeChanged)
{
    if (WebViewViewportRef != nullptr)
    {
        WebViewViewportRef->ImeCompositionRangeChanged(bIsRangeChanged);
    }
}

FNhnWebViewViewport::FNhnWebViewViewport()
{
    WebViewViewportRef = this;

    TargetTexture = MakeShared<FSlateTexture2DRHIRef, ESPMode::ThreadSafe>(nullptr, 0, 0);

    WebIndex = -1;
    bIsMouseDown = false;
    CurrentCursorType = EMouseCursor::Default;

    Ime = MakeShared<FNhnWebViewImeHandler>();

    bIgnoreKeyDownEvent = false;
    bIgnoreKeyUpEvent = false;
    bIgnoreCharacterEvent = false;
}

FNhnWebViewViewport::~FNhnWebViewViewport()
{
    FNhnWebViewModule::Get().GetCaller()->SetOnKeyEvent(WebIndex, nullptr);
    FNhnWebViewModule::Get().GetCaller()->SetOnImeCompositionRangeChanged(WebIndex, nullptr);
    
    WebViewViewportRef = nullptr;

    Ime->UnbindCefBrowser();
    Ime->UnbindInputMethodSystem();

    FSlateTexture2DRHIRef* CurrentTexture = TargetTexture.Get();
    if (CurrentTexture != nullptr)
    {
        ENQUEUE_RENDER_COMMAND(ResetTargetTexture)(
            [CurrentTexture](FRHICommandListImmediate& RHICmdList)
            {
                CurrentTexture->SetRHIRef(nullptr, 0, 0);
            });
        BeginReleaseResource(CurrentTexture);
        FlushRenderingCommands();
        TargetTexture.Reset();
    }
}

void FNhnWebViewViewport::SetWebView(const int32 InWebIndex, const TSharedPtr<SWidget>& ParentWidget)
{
    WebIndex = InWebIndex;

    ITextInputMethodSystem* TextInputMethodSystem = FSlateApplication::Get().GetTextInputMethodSystem();
    if (TextInputMethodSystem)
    {
        Ime->SetWebView(ParentWidget, WebIndex);
        Ime->BindInputMethodSystem(TextInputMethodSystem);

        FNhnWebViewModule::Get().GetCaller()->SetOnKeyEvent(WebIndex, OnNativeKeyEvent);
        FNhnWebViewModule::Get().GetCaller()->SetOnImeCompositionRangeChanged(WebIndex, OnImeCompositionRangeChanged);
    }
} 

void FNhnWebViewViewport::OnDrawViewport(const FGeometry& AllottedGeometry, const FSlateRect& MyCullingRect,
    FSlateWindowElementList& OutDrawElements, int32 LayerId, const FWidgetStyle& InWidgetStyle, bool bParentEnabled)
{
    if (WebIndex > 0)
    {
        const TSharedPtr<FNhnCefWebView> WebViewPtr = FNhnWebViewModule::Get().GetWebView(WebIndex);
        if (WebViewPtr.IsValid())
        {
            WebViewPtr->UpdateTexture();
            Ime->UpdateCachedGeometry(AllottedGeometry);
        }
    }
}

FIntPoint FNhnWebViewViewport::GetSize() const
{
    if ((WebIndex > 0) && TargetTexture.IsValid())
    {
        return FIntPoint(TargetTexture->GetWidth(), TargetTexture->GetHeight());
    }
    return FIntPoint(0, 0);
}

FSlateShaderResource* FNhnWebViewViewport::GetViewportRenderTargetTexture() const
{
    if ((WebIndex > 0) && TargetTexture.IsValid())
    {
        return TargetTexture.Get();
    }
    return nullptr;
}

void FNhnWebViewViewport::Tick(const FGeometry& AllottedGeometry, const double InCurrentTime, const float DeltaTime)
{
    if (WebIndex > 0)
    {
        const TSharedPtr<FNhnCefWebView> WebViewPtr = FNhnWebViewModule::Get().GetWebView(WebIndex);
        if (WebViewPtr.IsValid())
        {
            const int32 UpdateStatus = WebViewPtr->EventLoop();
            UpdateMouseCursor(UpdateStatus);

            RenderWebView(WebViewPtr);

            if (bHasNativeKeyEvent)
            {
                bHasNativeKeyEvent = false;
                OnUnhandledKeyEvent(NativeKeyEventType, NativeKeyEventModifiers, NativeKeyEventKeyCode, NativeKeyEventCharacter);
            }

            Ime->UpdateFocusChangedContext();
        }
    }

    ISlateViewport::Tick(AllottedGeometry, InCurrentTime, DeltaTime);
}

void FNhnWebViewViewport::RenderWebView(TSharedPtr<FNhnCefWebView> WebViewPtr) const
{
    FSlateTexture2DRHIRef* CurrentTexture = TargetTexture.Get();
    if (CurrentTexture != nullptr)
    {
        ENQUEUE_RENDER_COMMAND(TargetTextureCommand)(
    [WebViewPtr, CurrentTexture](FRHICommandListImmediate& RHICmdList)
        {
            UTexture2D* WebViewTexture = WebViewPtr->GetTexture();
             if (WebViewTexture != nullptr)
             {
    #if ENGINE_MAJOR_VERSION <= 4 && ENGINE_MINOR_VERSION <= 26
                 FTextureResource* TextureResource = WebViewTexture->Resource;
    #else
                 FTextureResource* TextureResource = WebViewTexture->GetResource();
    #endif
                 if (TextureResource != nullptr)
                 {
    #if ENGINE_MAJOR_VERSION <= 4 && ENGINE_MINOR_VERSION <= 25
                     FRHITexture2D* RHITexture2D = TextureResource->TextureRHI.IsValid() ? TextureResource->TextureRHI->GetTexture2D() : nullptr;
    #else
                     FRHITexture2D* RHITexture2D = TextureResource->GetTexture2DRHI();
    #endif
                     if (RHITexture2D != nullptr)
                     {
                         if (CurrentTexture->IsInitialized() == false)
                         {
                             CurrentTexture->InitResource();
                         }

                         // Update the slate texture.
                         const FTexture2DRHIRef Ref = RHITexture2D;
                         CurrentTexture->SetRHIRef(Ref, WebViewTexture->GetSizeX(), WebViewTexture->GetSizeY());
                     }
                 }
             }
        });
        FlushRenderingCommands();
    }
}

bool FNhnWebViewViewport::RequiresVsync() const
{
    return false;
}

FCursorReply FNhnWebViewViewport::OnCursorQuery(const FGeometry& MyGeometry, const FPointerEvent& CursorEvent)
{
    return ISlateViewport::OnCursorQuery(MyGeometry, CursorEvent);
}

FReply FNhnWebViewViewport::OnMouseButtonDown(const FGeometry& MyGeometry, const FPointerEvent& MouseEvent)
{
    MousePosition = GetMousePosition(MyGeometry, MouseEvent);
    FNhnWebViewModule::Get().GetCaller()->InputWeb(WebIndex, CefWebInput::Mouse | CefWebInput::Press,
        static_cast<int32>(MousePosition.X), static_cast<int32>(MousePosition.Y));
    bIsMouseDown = true;

    return FReply::Handled();
}

FReply FNhnWebViewViewport::OnMouseButtonUp(const FGeometry& MyGeometry, const FPointerEvent& MouseEvent)
{
    MousePosition = GetMousePosition(MyGeometry, MouseEvent);
    FNhnWebViewModule::Get().GetCaller()->InputWeb(WebIndex, CefWebInput::Mouse | CefWebInput::Release,
        static_cast<int32>(MousePosition.X), static_cast<int32>(MousePosition.Y));

    if (bIsMouseDown == true)
    {
        Ime->SetFocus(true);
    }
    bIsMouseDown = false;

    return FReply::Handled();
}

void FNhnWebViewViewport::OnMouseEnter(const FGeometry& MyGeometry, const FPointerEvent& MouseEvent)
{
    ISlateViewport::OnMouseEnter(MyGeometry, MouseEvent);
}

void FNhnWebViewViewport::OnMouseLeave(const FPointerEvent& MouseEvent)
{
    bIsMouseDown = false;
    ISlateViewport::OnMouseLeave(MouseEvent);
}

FReply FNhnWebViewViewport::OnMouseMove(const FGeometry& MyGeometry, const FPointerEvent& MouseEvent)
{
    const FVector2D MousePositionTemp = GetMousePosition(MyGeometry, MouseEvent);
    if (MousePosition != MousePositionTemp)
    {
        MousePosition = MousePositionTemp;

        if (bIsMouseDown == true)
        {
            FNhnWebViewModule::Get().GetCaller()->InputWeb(WebIndex, CefWebInput::Mouse | CefWebInput::Move | CefWebInput::MousePressing,
                static_cast<int32>(MousePosition.X), static_cast<int32>(MousePosition.Y));
        }
        else
        {
            FNhnWebViewModule::Get().GetCaller()->InputWeb(WebIndex, CefWebInput::Mouse | CefWebInput::Move,
                static_cast<int32>(MousePosition.X), static_cast<int32>(MousePosition.Y));
        }
        return FReply::Handled();
    }

    return ISlateViewport::OnMouseMove(MyGeometry, MouseEvent);
}

FReply FNhnWebViewViewport::OnMouseWheel(const FGeometry& MyGeometry, const FPointerEvent& MouseEvent)
{
    constexpr float SpinFactor = 50.0f;
    const float TrueDelta = MouseEvent.GetWheelDelta() * SpinFactor;
    FNhnWebViewModule::Get().GetCaller()->InputWeb(WebIndex, CefWebInput::Mouse | CefWebInput::Scroll,
        MouseEvent.IsShiftDown() ? TrueDelta : 0, !MouseEvent.IsShiftDown() ? TrueDelta : 0);

    return FReply::Handled();
}

FReply FNhnWebViewViewport::OnMouseButtonDoubleClick(const FGeometry& InMyGeometry, const FPointerEvent& InMouseEvent)
{
    return ISlateViewport::OnMouseButtonDoubleClick(InMyGeometry, InMouseEvent);
}

FReply FNhnWebViewViewport::OnKeyDown(const FGeometry& MyGeometry, const FKeyEvent& InKeyEvent)
{
    if (bIgnoreKeyDownEvent == false)
    {
        PreviousKeyDownEvent = InKeyEvent;
        const uint32 KeyCode = InKeyEvent.GetKeyCode();
        const int32 Modifier = GetKeyboardModifier(InKeyEvent);
        FNhnWebViewModule::Get().GetCaller()->InputWeb(WebIndex, CefWebInput::Keyboard | CefWebInput::Press | Modifier, KeyCode, KeyCode);

        return FReply::Handled();
    }
    return FReply::Unhandled();
}

FReply FNhnWebViewViewport::OnKeyUp(const FGeometry& MyGeometry, const FKeyEvent& InKeyEvent)
{
    if (bIgnoreKeyUpEvent == false)
    {
        PreviousKeyUpEvent = InKeyEvent;
        const uint32 KeyCode = InKeyEvent.GetKeyCode();
        const int32 Modifier = GetKeyboardModifier(InKeyEvent);
        FNhnWebViewModule::Get().GetCaller()->InputWeb(WebIndex, CefWebInput::Keyboard | CefWebInput::Release | Modifier, KeyCode, KeyCode);

        return FReply::Handled();
    }
    return FReply::Unhandled();
}

FReply FNhnWebViewViewport::OnKeyChar(const FGeometry& MyGeometry, const FCharacterEvent& InCharacterEvent)
{
    if (bIgnoreCharacterEvent == false)
    {
        PreviousCharacterEvent = InCharacterEvent;
        const uint32 KeyCode = InCharacterEvent.GetCharacter();
        const int32 Modifier = GetKeyboardModifier(InCharacterEvent);
        FNhnWebViewModule::Get().GetCaller()->InputWeb(WebIndex, CefWebInput::Keyboard | CefWebInput::Character | Modifier, KeyCode, KeyCode);

        return FReply::Handled();
    }
    return FReply::Unhandled();
}

FReply FNhnWebViewViewport::OnFocusReceived(const FFocusEvent& InFocusEvent)
{
    Ime->SetFocus(true);

    return FReply::Handled();
}

void FNhnWebViewViewport::OnFocusLost(const FFocusEvent& InFocusEvent)
{
    Ime->SetFocus(false);

    ISlateViewport::OnFocusLost(InFocusEvent);
}

FVector2D FNhnWebViewViewport::GetMousePosition(const FGeometry& MyGeometry, const FPointerEvent& MouseEvent) const
{
    if (WebIndex > 0)
    {
        if (TargetTexture.IsValid())
        {
            const float XScale = TargetTexture->GetWidth() / MyGeometry.GetLocalSize().X;
            const float YScale = TargetTexture->GetHeight() / MyGeometry.GetLocalSize().Y;
            const FVector2D LocalPosition = MyGeometry.AbsoluteToLocal(MouseEvent.GetScreenSpacePosition());
            const FVector2D InputPosition = LocalPosition * FVector2D(XScale, YScale);

            return InputPosition;
        }
    }
    return FVector2D();
}

int32 FNhnWebViewViewport::GetKeyboardModifier(const FInputEvent& InputEvent)
{
    int32 Modifier = 0;
    if (InputEvent.IsShiftDown() == true)
    {
        Modifier |= CefWebInput::Shift;
    }
    if (InputEvent.IsControlDown() == true)
    {
        Modifier |= CefWebInput::Ctrl;
    }
    if (InputEvent.IsAltDown() == true)
    {
        Modifier |= CefWebInput::Alt;
    }
    return Modifier;
}

void FNhnWebViewViewport::UpdateMouseCursor(const int32 WebViewUpdateStatus)
{
    const int32 Cursor = WebViewUpdateStatus & CefWebUpdateStatus::CursorMask;
    if (Cursor > 0)
    {
        switch (Cursor)
        {
        case CefWebCursorType::Hand:
            CurrentCursorType = EMouseCursor::Hand;
            break;
        case CefWebCursorType::IBeam:
            CurrentCursorType = EMouseCursor::TextEditBeam;
            break;
        default:
            CurrentCursorType = EMouseCursor::Default;
            break;
        }
    }
}

FModifierKeysState FNhnWebViewViewport::SlateModifiersFromCefModifiers(const uint32 Modifiers)
{
    return FModifierKeysState((Modifiers & EVENTFLAG_SHIFT_DOWN) != 0,
        (Modifiers & EVENTFLAG_SHIFT_DOWN) != 0,
        (Modifiers & EVENTFLAG_CONTROL_DOWN) != 0,
        (Modifiers & EVENTFLAG_CONTROL_DOWN) != 0,
        (Modifiers & EVENTFLAG_ALT_DOWN) != 0,
        (Modifiers & EVENTFLAG_ALT_DOWN) != 0,
        false, false,
        (Modifiers & EVENTFLAG_CAPS_LOCK_ON) != 0);
}

bool FNhnWebViewViewport::OnUnhandledKeyEvent(const int32 Type, const uint32 Modifiers, const int32 KeyCode, const wchar_t Character)
{
    bool bWasHandled = false;

    switch (Type)
    {
    case KEYEVENT_RAWKEYDOWN:
    case KEYEVENT_KEYDOWN:
        {
            if (PreviousKeyDownEvent.IsSet())
            {
                // If the keydown handler is not bound or if the handler returns false, indicating the key is unhandled, we bubble it up.
                bIgnoreKeyDownEvent = true;
                bWasHandled = FSlateApplication::Get().ProcessKeyDownEvent(PreviousKeyDownEvent.GetValue());
                bIgnoreKeyDownEvent = false;
                PreviousKeyDownEvent.Reset();
            }
            else
            {
                FKey const Key = FInputKeyManager::Get().GetKeyFromCodes(KeyCode, 0);
                if (Key.IsValid())
                {
                    const FKeyEvent KeyEvent(Key, SlateModifiersFromCefModifiers(Modifiers), FSlateApplication::Get().GetUserIndexForKeyboard(), false, 0, KeyCode);

                    bIgnoreKeyDownEvent = true;
                    bWasHandled = FSlateApplication::Get().ProcessKeyDownEvent(KeyEvent);
                    bIgnoreKeyDownEvent = false;
                }
            }
        }
        break;
    case KEYEVENT_KEYUP:
        {
            if (PreviousKeyUpEvent.IsSet())
            {
                // If the keyup handler is not bound or if the handler returns false, indicating the key is unhandled, we bubble it up.
                bIgnoreKeyUpEvent = true;
                bWasHandled = FSlateApplication::Get().ProcessKeyUpEvent(PreviousKeyUpEvent.GetValue());
                bIgnoreKeyUpEvent = false;
                PreviousKeyUpEvent.Reset();
            }
            else
            {
                FKey const Key = FInputKeyManager::Get().GetKeyFromCodes(KeyCode, 0);
                const FKeyEvent KeyEvent(Key, SlateModifiersFromCefModifiers(Modifiers), FSlateApplication::Get().GetUserIndexForKeyboard(), false, 0, KeyCode);

                bIgnoreKeyUpEvent = true;
                bWasHandled = FSlateApplication::Get().ProcessKeyUpEvent(KeyEvent);
                bIgnoreKeyUpEvent = false;
            }
        }
        break;
    case KEYEVENT_CHAR:
        {
            if (PreviousCharacterEvent.IsSet())
            {
                // If the keychar handler is not bound or if the handler returns false, indicating the key is unhandled, we bubble it up.
                bIgnoreCharacterEvent = true;
                bWasHandled = FSlateApplication::Get().ProcessKeyCharEvent(PreviousCharacterEvent.GetValue());
                bIgnoreCharacterEvent = false;
                PreviousCharacterEvent.Reset();
            }
            else
            {
                const FCharacterEvent CharacterEvent(Character, SlateModifiersFromCefModifiers(Modifiers), FSlateApplication::Get().GetUserIndexForKeyboard(), false);

                bIgnoreCharacterEvent = true;
                bWasHandled = FSlateApplication::Get().ProcessKeyCharEvent(CharacterEvent);
                bIgnoreCharacterEvent = false;
            }
        }
        break;
    default:
        break;
    }
    return bWasHandled;
}

void FNhnWebViewViewport::ImeCompositionRangeChanged(const bool bIsRangeChanged) const
{
    Ime->CEFCompositionRangeChanged(bIsRangeChanged);
}

void FNhnWebViewViewport::SetKeyEvent(const int32 Type, const uint32 Modifiers, const int32 KeyCode, const wchar_t Character)
{
    NativeKeyEventType = Type;
    NativeKeyEventModifiers = Modifiers;
    NativeKeyEventKeyCode = KeyCode;
    NativeKeyEventCharacter = Character;
    bHasNativeKeyEvent = true;
}
