
#include "NhnWebViewTextInputMethodContext.h"
#include "NhnWebViewImeHandler.h"
#include "NhnWebViewModule.h"

TSharedRef<FNhnWebViewTextInputMethodContext> FNhnWebViewTextInputMethodContext::Create(const TSharedRef<FNhnWebViewImeHandler>& InOwner, const uint32 WebIndex)
{
    return MakeShareable(new FNhnWebViewTextInputMethodContext(InOwner, WebIndex));
}

FNhnWebViewTextInputMethodContext::FNhnWebViewTextInputMethodContext(const TSharedRef<FNhnWebViewImeHandler>& InOwner, const uint32 Index)
    : Owner(InOwner)
    , bIsComposing(false)
    , CompositionBeginIndex(0)
    , CompositionLength(0)
    , SelectionRangeBeginIndex(0)
    , SelectionRangeLength(0)
    , SelectionCaretPosition(ECaretPosition::Ending)
{
    WebIndex = Index;
}

FNhnWebViewTextInputMethodContext::~FNhnWebViewTextInputMethodContext()
{
}

void FNhnWebViewTextInputMethodContext::AbortComposition()
{
    bIsComposing = false;
    FNhnWebViewModule::Get().GetCaller()->ImeCancelComposition(WebIndex);
    ResetComposition();
}

bool FNhnWebViewTextInputMethodContext::UpdateCachedGeometry(const FGeometry& AllottedGeometry)
{
    bool bCachedGeometryUpdated = false;
    if (CachedGeometry != AllottedGeometry)
    {
        CachedGeometry = AllottedGeometry;
        bCachedGeometryUpdated = true;
    }

    return bCachedGeometryUpdated;
}

bool FNhnWebViewTextInputMethodContext::CEFCompositionRangeChanged(const bool bIsRangeChanged) const
{
    return bIsComposing && bIsRangeChanged;
}

bool FNhnWebViewTextInputMethodContext::IsComposing()
{
    return bIsComposing;
}

bool FNhnWebViewTextInputMethodContext::IsReadOnly()
{
    return false;
}

uint32 FNhnWebViewTextInputMethodContext::GetTextLength()
{
    return CompositionString.Len();
}

void FNhnWebViewTextInputMethodContext::GetSelectionRange(uint32& BeginIndex, uint32& Length, ECaretPosition& CaretPosition)
{
    BeginIndex = SelectionRangeBeginIndex;
    Length = SelectionRangeLength;
    CaretPosition = SelectionCaretPosition;
}

void FNhnWebViewTextInputMethodContext::SetSelectionRange(const uint32 BeginIndex, const uint32 Length, const ECaretPosition CaretPosition)
{
    SelectionRangeBeginIndex = BeginIndex;
    SelectionRangeLength = Length;
    SelectionCaretPosition = CaretPosition;

    FNhnWebViewModule::Get().GetCaller()->ImeSetComposition(WebIndex, CompositionString, SelectionRangeBeginIndex, SelectionRangeLength);
}

void FNhnWebViewTextInputMethodContext::GetTextInRange(const uint32 BeginIndex, const uint32 Length, FString& OutString)
{
    OutString = CompositionString.Mid(BeginIndex, Length);
}

void FNhnWebViewTextInputMethodContext::SetTextInRange(const uint32 BeginIndex, const uint32 Length, const FString& InString)
{
    FString NewString;
    if (BeginIndex > 0)
    {
        NewString = CompositionString.Mid(0, BeginIndex);
    }

    NewString += InString;

    if (static_cast<int32>(BeginIndex + Length) < CompositionString.Len())
    {
        NewString += CompositionString.Mid(BeginIndex + Length, CompositionString.Len() - (BeginIndex + Length));
    }
    CompositionString = NewString;

    FNhnWebViewModule::Get().GetCaller()->ImeSetComposition(WebIndex, CompositionString, 0, CompositionString.Len());
}

int32 FNhnWebViewTextInputMethodContext::GetCharacterIndexFromPoint(const FVector2D& Point)
{
    const FVector2D LocalPoint = CachedGeometry.AbsoluteToLocal(Point);
    const int32 X = FMath::RoundToInt(LocalPoint.X);
    const int32 Y = FMath::RoundToInt(LocalPoint.Y);
    const int32 ResultIdx = FNhnWebViewModule::Get().GetCaller()->GetCompositionBoundIndex(WebIndex, X, Y);
    return ResultIdx;
}

bool FNhnWebViewTextInputMethodContext::GetTextBounds(const uint32 BeginIndex, const uint32 Length, FVector2D& Position, FVector2D& Size)
{
    int32 X = 0;
    int32 Y = 0;
    int32 Width = 0;
    int32 Height = 0;
    FNhnWebViewModule::Get().GetCaller()->GetCompositionBound(WebIndex, 0, X, Y, Width, Height);

    const uint32 CefCompositionBoundSize = static_cast<uint32>(FNhnWebViewModule::Get().GetCaller()->GetCompositionBoundSize(WebIndex));

    if (CefCompositionBoundSize < BeginIndex ||
        CefCompositionBoundSize < BeginIndex + Length)
    {
        if (CefCompositionBoundSize > 0)
        {
            // Fall back to the start of the composition
            Position = CachedGeometry.LocalToAbsolute(FVector2D(X, Y));
            Size = FVector2D(Width, Height);
            return false;
        }
        else
        {
            // We  don't have any updated composition bounds so we'll just default to the window bounds and say we are clipped.
            GetScreenBounds(Position, Size);
            return true;
        }
    }

    FVector2D LocalSpaceMin(FLT_MAX, FLT_MAX);
    FVector2D LocalSpaceMax(-FLT_MAX, -FLT_MAX);

    for (uint32 CharIdx = BeginIndex; CharIdx < BeginIndex + Length; CharIdx++)
    {
        FNhnWebViewModule::Get().GetCaller()->GetCompositionBound(WebIndex, CharIdx, X, Y, Width, Height);

        if (LocalSpaceMin.X > X)
        {
            LocalSpaceMin.X = X;
        }

        if (LocalSpaceMax.X < X + Width)
        {
            LocalSpaceMax.X = X + Width;
        }

        if (LocalSpaceMin.Y > Y)
        {
            LocalSpaceMin.Y = Y;
        }

        if (LocalSpaceMax.Y < Y + Height)
        {
            LocalSpaceMax.Y = Y + Height;
        }
    }

    Position = CachedGeometry.LocalToAbsolute(LocalSpaceMin);
    Size = LocalSpaceMax - LocalSpaceMin;

    return false; // false means "not clipped"
}

void FNhnWebViewTextInputMethodContext::GetScreenBounds(FVector2D& Position, FVector2D& Size)
{
    Position = CachedGeometry.GetAccumulatedRenderTransform().GetTranslation();
    Size = TransformVector(CachedGeometry.GetAccumulatedRenderTransform(), CachedGeometry.GetLocalSize());
}

TSharedPtr<FGenericWindow> FNhnWebViewTextInputMethodContext::GetWindow()
{
    if (CachedSlateWindow.IsValid())
    {
        return CachedSlateWindow.Pin()->GetNativeWindow();
    }

    const TSharedPtr<SWidget> CachedSlateWidgetPtr = Owner->InternalBrowserSlateWidget;
    if (!CachedSlateWidgetPtr.IsValid())
    {
        return nullptr;
    }

    const TSharedRef<SWidget> CachedSlateWidgetRef = CachedSlateWidgetPtr.ToSharedRef();
    const TSharedPtr<SWindow> SlateWindow = FSlateApplication::Get().FindWidgetWindow(CachedSlateWidgetRef);
    CachedSlateWindow = SlateWindow;
    return SlateWindow.IsValid() ? SlateWindow->GetNativeWindow() : nullptr;
}

void FNhnWebViewTextInputMethodContext::BeginComposition()
{
    if (bIsComposing == false)
    {
        bIsComposing = true;
    }
}

void FNhnWebViewTextInputMethodContext::UpdateCompositionRange(const int32 InBeginIndex, const uint32 InLength)
{
    CompositionBeginIndex = InBeginIndex;
    CompositionLength = InLength;
}

void FNhnWebViewTextInputMethodContext::EndComposition()
{
    if (bIsComposing)
    {
        bIsComposing = false;

        if (CompositionString.Len() > 0)
        {
            FNhnWebViewModule::Get().GetCaller()->ImeCommitText(WebIndex, CompositionString);
        }
        else
        {
            FNhnWebViewModule::Get().GetCaller()->ImeCancelComposition(WebIndex);
        }
        ResetComposition();
    }
}

void FNhnWebViewTextInputMethodContext::ResetComposition()
{
    CompositionString.Empty();
    CompositionBeginIndex = 0;
    CompositionLength = 0;

    SelectionRangeBeginIndex = 0;
    SelectionRangeLength = 0;
}
