// Fill out your copyright notice in the Description page of Project Settings.


#include "NhnCefWebView.h"

#include "NhnWebViewErrorCode.h"
#include "NhnCefTypes.h"
#include "NhnWebViewDefines.h"
#include "NhnWebViewModule.h"

#include "Engine/Texture2D.h"

#define TEXTURE_BYTES_PER_PIXEL 4
#define MIN_KEY_VALUE_PAIR_COUNT 2

#define ERROR_SCHEME TEXT("cef://error")

#define SEPARATOR_COMMA TEXT(",")
#define DELIMITER_VERTICAL_BAR TEXT("|")
#define TEXT_0 TEXT("0")

#define KEY_PASS_POPUP_INFO_NAME TEXT("name")
#define KEY_PASS_POPUP_INFO_URL TEXT("url")
#define KEY_PASS_POPUP_INFO_LEFT TEXT("left")
#define KEY_PASS_POPUP_INFO_TOP TEXT("top")
#define KEY_PASS_POPUP_INFO_WIDTH TEXT("width")
#define KEY_PASS_POPUP_INFO_HEIGHT TEXT("height")
#define KEY_PASS_POPUP_INFO_MENUBAR TEXT("menubar")
#define KEY_PASS_POPUP_INFO_STATUS TEXT("status")
#define KEY_PASS_POPUP_INFO_TOOLBAR TEXT("toolbar")
#define KEY_PASS_POPUP_INFO_LOCATION TEXT("location")
#define KEY_PASS_POPUP_INFO_SCROLLBARS TEXT("scrollbars")
#define KEY_PASS_POPUP_INFO_RESIZABLE TEXT("resizable")

FNhnCefWebView::FNhnCefWebView(): Configuration()
{
    WebIndex = 0;
    bIsShow = false;
    bIsWebUpdated = false;
    bIsWebFocus = false;
    bIsRendering = false;
    Texture = nullptr;
}

FNhnCefWebView::~FNhnCefWebView()
{
    UrlCallback = nullptr;
    TitleCallback = nullptr;
    InputFocusCallback = nullptr;
    WebStatusCallback = nullptr;
    Texture = nullptr;
    TextureBuffer.Empty();

    FNhnWebViewModule::Get().GetCaller()->RemoveWeb(WebIndex);
    WebIndex = 0;
}

void FNhnCefWebView::CreateWebView(const int32 InWebIndex, const FCreateParams& Params, const FCefWebViewDelegateWebViewInfo& Callback)
{
    WebIndex = InWebIndex;
    Configuration = Params;

    Texture = UTexture2D::CreateTransient(Configuration.Size.X, Configuration.Size.Y, PF_B8G8R8A8);
    Texture->SRGB = false;
    Texture->AddToRoot();
    DrawDefaultColor();

    if (Configuration.PopupOption.Type == 0)
    {
        Configuration.PopupOption.Type = CefPopupType::Redirect;
    }

    if (Configuration.PopupOption.BlockMessage.IsEmpty() == false)
    {
        SetPopupBlockMessage(Configuration.PopupOption.BlockMessage);
    }

    TextureBuffer.SetNumUninitialized(Configuration.Size.X * Configuration.Size.Y * TEXTURE_BYTES_PER_PIXEL);

    FNhnWebViewModule::Get().GetCaller()->CreateWeb(InWebIndex, Configuration.Position.X, Configuration.Position.Y, Configuration.Size.X, Configuration.Size.Y,
        TextureBuffer.GetData(), Configuration.PopupOption.Type | Configuration.BgType);

    FWebViewInfo Info;
    Info.WebIndex = InWebIndex;
    Info.Texture = Texture;
    
    if (Callback.ExecuteIfBound(&Info, FNhnWebViewError(NhnWebViewErrorCode::Success)) == false)
    {
        NHNWEBVIEW_LOG_ERROR("WebView callback cannot be null. (WebIndex: %d)", WebIndex);
    }

    if(Configuration.PopupIcon.IsEmpty() == false)
    {
        FNhnWebViewModule::Get().GetCaller()->SetPopupIcon(WebIndex, Configuration.PopupIcon);
    }
}

void FNhnCefWebView::DrawDefaultColor() const
{
    if (Texture != nullptr)
    {
#if ENGINE_MAJOR_VERSION == 4 && ENGINE_MINOR_VERSION <= 27
        if (Texture->PlatformData != nullptr)
        {
            void* FormattedImageData = Texture->PlatformData->Mips[0].BulkData.Lock(LOCK_READ_WRITE);

            TArray<FColor> ColorData;
            const int32 Num = Texture->GetSizeX() * Texture->GetSizeY();
            for (int i = 0; i < Num; ++i)
            {
                ColorData.Add(FColor::White);
            }

            FMemory::Memcpy(FormattedImageData, (void*)ColorData.GetData(), Num * 4);

            Texture->PlatformData->Mips[0].BulkData.Unlock();
            Texture->UpdateResource();
        }
#else
        FTextureSource& Source = Texture->Source;
        if (Source.IsValid())
        {
            int32 Width = Source.GetSizeX();
            int32 Height = Source.GetSizeY();
            int32 Num = Width * Height;
            
            TArray<FColor> ColorData;
            ColorData.SetNum(Num);
            for (int32 i = 0; i < Num; ++i)
            {
                ColorData[i] = FColor::White;
            }
            
            uint8* DestData = Source.LockMip(0);
            if (DestData)
            {
                FMemory::Memcpy(DestData, ColorData.GetData(), Num * sizeof(FColor));
                Source.UnlockMip(0);
            }
            
            Texture->UpdateResource();
        }
#endif
    }
}

void FNhnCefWebView::ShowWebView(const FShowParams& Params)
{
    UrlCallback = Params.UrlDelegate;
    TitleCallback = Params.TitleDelegate;
    InputFocusCallback = Params.InputFocusDelegate;
    WebStatusCallback = Params.StatusDelegate;

    bIsShow = true;

    FNhnWebViewModule::Get().GetCaller()->LoadWeb(WebIndex, Params.Url, Params.bShowScrollBar);

    if (Params.OpenDelegate.ExecuteIfBound(FNhnWebViewError(NhnWebViewErrorCode::Success)) == false)
    {
        NHNWEBVIEW_LOG_DEBUG("WebView callback is null");
    }
}

void FNhnCefWebView::HideWebView()
{
    bIsShow = false;

    DrawDefaultColor();
}

void FNhnCefWebView::ResizeWebView(const int32 Width, const int32 Height)
{
    if (Texture == nullptr)
    {
        return;
    }

    Configuration.Size.X = Width;
    Configuration.Size.Y = Height;

    Texture = UTexture2D::CreateTransient(Configuration.Size.X, Configuration.Size.Y, PF_B8G8R8A8);
    Texture->AddToRoot();
    Texture->UpdateResource();

    TextureBuffer.Empty();
    TextureBuffer.SetNumUninitialized(Configuration.Size.X * Configuration.Size.Y * TEXTURE_BYTES_PER_PIXEL);
    FNhnWebViewModule::Get().GetCaller()->ResizeWeb(WebIndex, TextureBuffer.GetData(), Configuration.Size.X, Configuration.Size.Y);
}

void FNhnCefWebView::ShowScrollBar(const bool bShow) const
{
    FNhnWebViewModule::Get().GetCaller()->ShowScrollbar(WebIndex, bShow);
}

void FNhnCefWebView::InputWeb(const int32 WebIndex, const int32 Flags, const int32 X, const int32 Y)
{
    FNhnWebViewModule::Get().GetCaller()->InputWeb(WebIndex, Flags, X, Y);
}

void FNhnCefWebView::GoHome() const
{
    if (bIsShow)
    {
        FNhnWebViewModule::Get().GetCaller()->GoBackForwardHome(WebIndex, CefWebNavigationDirection::Home);
    }
}

bool FNhnCefWebView::CanGoBack() const
{
    if (bIsShow)
    {
        return FNhnWebViewModule::Get().GetCaller()->CanGoBackForward(WebIndex, CefWebNavigationDirection::Back);
    }
    return false;
}

void FNhnCefWebView::GoBack() const
{
    if (bIsShow)
    {
        FNhnWebViewModule::Get().GetCaller()->GoBackForwardHome(WebIndex, CefWebNavigationDirection::Back);
    }
}

bool FNhnCefWebView::CanGoForward() const
{
    if (bIsShow == true)
    {
        return FNhnWebViewModule::Get().GetCaller()->CanGoBackForward(WebIndex, CefWebNavigationDirection::Forward);
    }
    return false;
}

void FNhnCefWebView::GoForward() const
{
    if (bIsShow)
    {
        FNhnWebViewModule::Get().GetCaller()->GoBackForwardHome(WebIndex, CefWebNavigationDirection::Forward);
    }
}

void FNhnCefWebView::ExecuteJavaScript(const FString JavaScript) const
{
    if (bIsShow == true)
    {
        FNhnWebViewModule::Get().GetCaller()->ExecuteJavaScript(WebIndex, JavaScript);
    }
}

void FNhnCefWebView::ReserveInvalidRedirectUrlSchemes(const FString Schemes)
{
    FNhnWebViewModule::Get().GetCaller()->ReserveInvalidRedirectUrlSchemes(Schemes);
}

void FNhnCefWebView::SetDownloadCompleteOption(int Option)
{
    FNhnWebViewModule::Get().GetCaller()->SetDownloadCompleteOption(Option);
}

void FNhnCefWebView::SetDebugEnable(bool bEnable)
{
    FNhnWebViewModule::Get().GetCaller()->SetDebugEnable(bEnable);
}

void FNhnCefWebView::OpenDialog(const int32 Type, FString Message) const
{   
    if (WebStatusCallback.IsBound() == false)
    {
        return;
    }

    FWebViewStatus Vo;
    Vo.WebIndex = WebIndex;
    Vo.Status = CefWebUpdateStatus::JsDialog;
    Vo.JSDialog.Message = Message;
    Vo.JSDialog.Type = Type;
    Vo.JSDialog.ClickButtonDelegate = FCefJsDialogClickDelegate::CreateLambda([WebIndex = this->WebIndex](const bool bIsOkButton)
        {
            if (bIsOkButton == true)
            {
                FNhnWebViewModule::Get().GetCaller()->InputWeb(WebIndex, CefWebInput::JsDialog, 1, 0);
            }
            else
            {
                FNhnWebViewModule::Get().GetCaller()->InputWeb(WebIndex, CefWebInput::JsDialog, 0, 0);
            }
        });

    WebStatusCallback.ExecuteIfBound(Vo);
}

void FNhnCefWebView::OpenPopupBlock()
{
    if (WebStatusCallback.IsBound() == false)
    {
        return;
    }

    FWebViewStatus Vo;
    Vo.WebIndex = WebIndex;
    Vo.Status = CefWebUpdateStatus::PopupBlock;
    Vo.PopupBlock.Message = PopupBlockMessage;

    WebStatusCallback.ExecuteIfBound(Vo);
}

void FNhnCefWebView::SetPopupBlockMessage(FString Message)
{
    PopupBlockMessage = Message;
}

void FNhnCefWebView::UpdateTexture()
{
#if ENGINE_MAJOR_VERSION == 4 && ENGINE_MINOR_VERSION <= 27
    if (bIsShow && bIsRendering && Texture != nullptr && Texture->PlatformData != nullptr)
    {
        FTexture2DMipMap* MipMap = &Texture->PlatformData->Mips[0];
        if (MipMap != nullptr)
        {
            void* TextureDataTemp = MipMap->BulkData.Lock(LOCK_READ_WRITE);
            int32 Size = static_cast<int32>(Configuration.Size.X) * static_cast<int32>(Configuration.Size.Y) * TEXTURE_BYTES_PER_PIXEL;
            FPlatformMemory::Memcpy(TextureDataTemp, TextureBuffer.GetData(), Size);

            MipMap->BulkData.Unlock();

            Texture->UpdateResource();
        }
    }
#else
    if (bIsShow && bIsRendering && Texture != nullptr)
    {
        FTextureSource& Source = Texture->Source;
        if (Source.IsValid())
        {
            uint8* TextureDataTemp = Source.LockMip(0);
            if (TextureDataTemp != nullptr)
            {
                int32 size = static_cast<int32>(Configuration.Size.X) * static_cast<int32>(Configuration.Size.Y) * TEXTURE_BYTES_PER_PIXEL;
                FPlatformMemory::Memcpy(TextureDataTemp, TextureBuffer.GetData(), size);

                Source.UnlockMip(0);
                Texture->UpdateResource();
            }
        }
    }
#endif
}

int32 FNhnCefWebView::EventLoop()
{
    if (bIsShow == false)
    {
        return 0;
    }

    FString URL = TEXT("");
    int32 AdditionalInfo = 0;
    const int32 Status = FNhnWebViewModule::Get().GetCaller()->UpdateWeb(WebIndex, URL, AdditionalInfo);

    UpdateWebViewStatus(URL, Status, AdditionalInfo);
    return Status;
}

void FNhnCefWebView::UpdateWebViewStatus(FString URL, int32 Status, int32 AdditionalInfo)
{
    if (Status == CefWebUpdateStatus::Invalid || Status == CefWebUpdateStatus::None)
    {
        return;
    }

    if ((Status & CefWebUpdateStatus::Error) != 0)
    {
        FString ErrorText = FString::Printf(TEXT("%s?%s"), ERROR_SCHEME, *URL);
        UrlCallback.ExecuteIfBound(ErrorText);
        return;
    }

    if ((Status & CefWebUpdateStatus::Browsable) != 0)
    {
        if (URL.IsEmpty() == false)
        {
            TArray<FString> URLArray = {};
            URL.ParseIntoArray(URLArray, SEPARATOR_COMMA);
            for (int32 URLIndex = 0; URLIndex < URLArray.Num(); ++URLIndex)
            {
                FString& TrimUrl = URLArray[URLIndex];
                UrlCallback.ExecuteIfBound(TrimUrl.TrimStartAndEnd());
            }
        }
    }

    if ((Status & CefWebUpdateStatus::InputFocus) != 0)
    {
        InputFocusCallback.ExecuteIfBound(AdditionalInfo);
    }

    if ((Status & CefWebUpdateStatus::Title) != 0)
    {
        FString& Title = URL;
        TitleCallback.ExecuteIfBound(Title);
        FNhnWebViewModule::Get().GetCaller()->InputWeb(WebIndex, CefWebInput::Keyboard | CefWebInput::Event | CefWebInput::Focus, 0, 0);
    }

    if ((Status & CefWebUpdateStatus::JsDialog) != 0)
    {
        FString& DialogMessage = URL;
        int32 DialogType = AdditionalInfo;

        OpenDialog(DialogType, DialogMessage);
    }

    if ((Status & CefWebUpdateStatus::Available) != 0)
    {
        bIsWebUpdated = true;
    }

    if ((Status & CefWebUpdateStatus::Focused) != 0)
    {
        bIsWebFocus = true;
    }

    if ((Status & CefWebUpdateStatus::PopupBlock) != 0)
    {
        OpenPopupBlock();
    }

    if ((Status & CefWebUpdateStatus::LoadEnd) != 0)
    {
        bIsRendering = true;
        FWebViewStatus Vo;
        Vo.WebIndex = WebIndex;
        Vo.Status = CefWebUpdateStatus::LoadEnd;
        Vo.LoadEnd.Url = URL;
        WebStatusCallback.ExecuteIfBound(Vo);
        FNhnWebViewModule::Get().GetCaller()->InputWeb(WebIndex, CefWebInput::Keyboard | CefWebInput::Event | CefWebInput::Focus, 0, 0);
    }

    if ((Status & CefWebUpdateStatus::PassPopupInfo) != 0)
    {
        FString& popupInfo = URL;
        if (popupInfo.IsEmpty() == true)
        {
            return;
        }

        TMap<FString, FString> Dictionary;
        TArray<FString> KeyValueArray = {};
        popupInfo.ParseIntoArray(KeyValueArray, SEPARATOR_COMMA);

        for (int32 PairIndex = 0; PairIndex < KeyValueArray.Num(); ++PairIndex)
        {
            FString& PairString = KeyValueArray[PairIndex];
            TArray<FString> Pair = {};
            int32 Num = PairString.ParseIntoArray(Pair, DELIMITER_VERTICAL_BAR);
            if (Num == MIN_KEY_VALUE_PAIR_COUNT)
            {
                Dictionary.Add(Pair[0], Pair[1]);
            }
        }

        FWebViewStatus Vo;
        Vo.WebIndex = WebIndex;
        Vo.Status = CefWebUpdateStatus::PassPopupInfo;

        FPassPopupInfo PassPopupInfo;
        if (FString* FoundUrl = Dictionary.Find(KEY_PASS_POPUP_INFO_URL))
        {
            Vo.PassPopupInfo.Url = *FoundUrl;

            FString* FoundName = Dictionary.Find(KEY_PASS_POPUP_INFO_NAME);
            Vo.PassPopupInfo.Name = *FoundName;

            FString* FoundLeft = Dictionary.Find(KEY_PASS_POPUP_INFO_LEFT);
            Vo.PassPopupInfo.Left = FCString::Atoi(**FoundLeft);

            FString* FoundTop = Dictionary.Find(KEY_PASS_POPUP_INFO_TOP);
            Vo.PassPopupInfo.Top = FCString::Atoi(**FoundTop);

            FString* FoundWidth = Dictionary.Find(KEY_PASS_POPUP_INFO_WIDTH);
            Vo.PassPopupInfo.Width = FCString::Atoi(**FoundWidth);

            FString* FoundHeight = Dictionary.Find(KEY_PASS_POPUP_INFO_HEIGHT);
            Vo.PassPopupInfo.Height = FCString::Atoi(**FoundHeight);

            FString* FoundMenubar = Dictionary.Find(KEY_PASS_POPUP_INFO_MENUBAR);
            Vo.PassPopupInfo.bMenubar = !FoundMenubar->Equals(TEXT_0);

            FString* FoundStatus = Dictionary.Find(KEY_PASS_POPUP_INFO_STATUS);
            Vo.PassPopupInfo.bStatus = !FoundStatus->Equals(TEXT_0);

            FString* FoundToolbar = Dictionary.Find(KEY_PASS_POPUP_INFO_TOOLBAR);
            Vo.PassPopupInfo.bToolbar = !FoundToolbar->Equals(TEXT_0);

            FString* FoundScrollbars = Dictionary.Find(KEY_PASS_POPUP_INFO_SCROLLBARS);
            Vo.PassPopupInfo.bScrollbars = !FoundScrollbars->Equals(TEXT_0);
        }

        WebStatusCallback.ExecuteIfBound(Vo);
    }
}

UTexture2D* FNhnCefWebView::GetTexture() const
{
    return Texture;
}
