#pragma once

#include "CoreMinimal.h"
#include "Misc/Variant.h"
#include "NhnVariantType.h"

using FEngineVariant = FVariant;
using EEngineVariantTypes = EVariantTypes;

// Forward declarations
class FNhnVariant;
class FNhnVariantMap;
class FNhnVariantArray;

template<typename T> struct TNhnVariantTraits
{
    static CONSTEXPR ENhnVariantType GetType()
    {
        static_assert(!sizeof(T), "Variant trait must be specialized for this type.");
        return ENhnVariantType::Empty;
    }
    
    static CONSTEXPR EEngineVariantTypes GetVariantType()
    {
        static_assert(!sizeof(T), "Variant trait must be specialized for this type.");
        return EEngineVariantTypes::Empty;
    }
    
    static auto ToArchiveType(const T& Value)
    {
        static_assert(!sizeof(T), "ToArchiveType must be specialized for this type.");
        return Value;
    }
};

template<> struct TNhnVariantTraits<bool>
{
    static CONSTEXPR ENhnVariantType GetType() { return ENhnVariantType::Bool; }
    static CONSTEXPR EEngineVariantTypes GetVariantType() { return EEngineVariantTypes::Bool; }
    static bool ToArchiveType(const bool& Value) { return Value; }
};

template<> struct TNhnVariantTraits<int32>
{
    static CONSTEXPR ENhnVariantType GetType() { return ENhnVariantType::Int32; }
    static CONSTEXPR EEngineVariantTypes GetVariantType() { return EEngineVariantTypes::Int32; }
    static int32 ToArchiveType(const int32& Value) { return Value; }
};

template<> struct TNhnVariantTraits<int64>
{
    static CONSTEXPR ENhnVariantType GetType() { return ENhnVariantType::Int64; }
    static CONSTEXPR EEngineVariantTypes GetVariantType() { return EEngineVariantTypes::Int64; }
    static int64 ToArchiveType(const int64& Value) { return Value; }
};

template<> struct TNhnVariantTraits<uint32>
{
    static CONSTEXPR ENhnVariantType GetType() { return ENhnVariantType::UInt32; }
    static CONSTEXPR EEngineVariantTypes GetVariantType() { return EEngineVariantTypes::UInt32; }
    static uint32 ToArchiveType(const uint32& Value) { return Value; }
};

template<> struct TNhnVariantTraits<uint64>
{
    static CONSTEXPR ENhnVariantType GetType() { return ENhnVariantType::UInt64; }
    static CONSTEXPR EEngineVariantTypes GetVariantType() { return EEngineVariantTypes::UInt64; }
    static uint64 ToArchiveType(const uint64& Value) { return Value; }
};

template<> struct TNhnVariantTraits<double>
{
    static CONSTEXPR ENhnVariantType GetType() { return ENhnVariantType::Double; }
    static CONSTEXPR EEngineVariantTypes GetVariantType() { return EEngineVariantTypes::Double; }
    static double ToArchiveType(const double& Value) { return Value; }
};

template<> struct TNhnVariantTraits<float>
{
    static CONSTEXPR ENhnVariantType GetType() { return ENhnVariantType::Float; }
    static CONSTEXPR EEngineVariantTypes GetVariantType() { return EEngineVariantTypes::Float; }
    static float ToArchiveType(const float& Value) { return Value; }
};

template<> struct TNhnVariantTraits<FName>
{
    static CONSTEXPR ENhnVariantType GetType() { return ENhnVariantType::Name; }
    static CONSTEXPR EEngineVariantTypes GetVariantType() { return EEngineVariantTypes::Name; }
    static FName ToArchiveType(const FName& Value) { return Value; }
};

template<> struct TNhnVariantTraits<FString>
{
    static CONSTEXPR ENhnVariantType GetType() { return ENhnVariantType::String; }
    static CONSTEXPR EEngineVariantTypes GetVariantType() { return EEngineVariantTypes::String; }
    static const FString& ToArchiveType(const FString& Value) { return Value; }
};

template<> struct TNhnVariantTraits<TCHAR*>
{
    static CONSTEXPR ENhnVariantType GetType() { return ENhnVariantType::String; }
    static CONSTEXPR EEngineVariantTypes GetVariantType() { return EEngineVariantTypes::String; }
    static FString ToArchiveType(TCHAR* Value) { return FString(Value); }
};

template<size_t N> struct TNhnVariantTraits<TCHAR[N]>
{
    static CONSTEXPR ENhnVariantType GetType() { return ENhnVariantType::String; }
    static CONSTEXPR EEngineVariantTypes GetVariantType() { return EEngineVariantTypes::String; }
    static FString ToArchiveType(const TCHAR (&Value)[N]) { return FString(Value); }
};

template<> struct TNhnVariantTraits<FNhnVariant>
{
    static CONSTEXPR ENhnVariantType GetType() { return ENhnVariantType::Empty; }
    static CONSTEXPR EEngineVariantTypes GetVariantType() { return EEngineVariantTypes::Custom; }
    static const FNhnVariant& ToArchiveType(const FNhnVariant& Value) { return Value; }
};

template<> struct TNhnVariantTraits<FNhnVariantArray>
{
    static CONSTEXPR ENhnVariantType GetType() { return ENhnVariantType::Array; }
    static CONSTEXPR EEngineVariantTypes GetVariantType() { return EEngineVariantTypes::ByteArray; }
    static const FNhnVariantArray& ToArchiveType(const FNhnVariantArray& Value) { return Value; }
};

template<> struct TNhnVariantTraits<FNhnVariantMap>
{
    static CONSTEXPR ENhnVariantType GetType() { return ENhnVariantType::Object; }
    static CONSTEXPR EEngineVariantTypes GetVariantType() { return EEngineVariantTypes::ByteArray; }
    static const FNhnVariantMap& ToArchiveType(const FNhnVariantMap& Value) { return Value; }
};

template<> struct TNhnVariantTraits<FEngineVariant>
{
    static CONSTEXPR ENhnVariantType GetType() { return ENhnVariantType::Empty; }
    static CONSTEXPR EEngineVariantTypes GetVariantType() { return EEngineVariantTypes::Custom; }
    static const FNhnVariant& ToArchiveType(const FNhnVariant& Value) { return Value; }
    
    static ENhnVariantType GetType(const FEngineVariant& Value)
    {
        switch (Value.GetType())
        {
            case EEngineVariantTypes::Int32:  return ENhnVariantType::Int32;
            case EEngineVariantTypes::Int64:  return ENhnVariantType::Int64;
            case EEngineVariantTypes::UInt32: return ENhnVariantType::UInt32;
            case EEngineVariantTypes::UInt64: return ENhnVariantType::UInt64;
            case EEngineVariantTypes::Bool:   return ENhnVariantType::Bool;
            case EEngineVariantTypes::Float:  return ENhnVariantType::Float;
            case EEngineVariantTypes::Double: return ENhnVariantType::Double;
            case EEngineVariantTypes::String: return ENhnVariantType::String;
            case EEngineVariantTypes::Name:   return ENhnVariantType::Name;
            default:                          return ENhnVariantType::Empty;
        }
    }
    
    static EEngineVariantTypes GetVariantType(const FEngineVariant& Value)
    {
        return Value.GetType();
    }
};

template<> struct TNhnVariantTraits<std::nullptr_t>
{
    static CONSTEXPR ENhnVariantType GetType() { return ENhnVariantType::Empty; }
    static CONSTEXPR EEngineVariantTypes GetVariantType() { return EEngineVariantTypes::Empty; }
    static void ToArchiveType(const std::nullptr_t&) {}
};
