#pragma once

#include "NhnVariant.h"

class FNhnVariantMap
{
public:
    // Core type definitions for TMap compatibility
    using FKeyType = FName;
    using FValueType = FNhnVariant;
    using FVariantMapType = TMap<FKeyType, FValueType>;

    // Default constructors and operators
    FNhnVariantMap() = default;
    FNhnVariantMap(FNhnVariantMap&&) = default;
    FNhnVariantMap(const FNhnVariantMap&) = default;
    FNhnVariantMap& operator=(FNhnVariantMap&&) = default;
    FNhnVariantMap& operator=(const FNhnVariantMap&) = default;

    // Construction from TMap
    explicit FNhnVariantMap(const FVariantMapType& InMap) : Map(InMap) {}
    explicit FNhnVariantMap(FVariantMapType&& InMap) : Map(MoveTemp(InMap)) {}

    // Basic initializer_list constructors
    FNhnVariantMap(std::initializer_list<TPairInitializer<const FKeyType&, const FValueType&>> InitList)
    {
        Reserve(static_cast<int32>(InitList.size()));
        for (const TPairInitializer<const FKeyType&, const FValueType&>& Element : InitList)
        {
            Add(Element.Key, Element.Value);
        }
    }
    
    // Assignment operator which gets its elements from a native initializer list
    FNhnVariantMap& operator=(std::initializer_list<TPairInitializer<const FKeyType&, const FValueType&>> InitList)
    {
        Empty(static_cast<int32>(InitList.size()));
        for (const TPairInitializer<const FKeyType&, const FValueType&>& Element : InitList)
        {
            Add(Element.Key, Element.Value);
        }
        return *this;
    }
    
    // Core TMap-compatible methods
    FORCEINLINE void Empty(const int32 ExpectedNumElements = 0) { Map.Empty(ExpectedNumElements); }
    FORCEINLINE void Reset() { Map.Reset(); }
    FORCEINLINE void Shrink() { Map.Shrink(); }
    FORCEINLINE void Compact() { Map.Compact(); }
    FORCEINLINE void CompactStable() { Map.CompactStable(); }
    FORCEINLINE void Reserve(const int32 Number) { Map.Reserve(Number); }
    FORCEINLINE int32 Num() const { return Map.Num(); }

    // Add methods
    FORCEINLINE FValueType& Add(const FKeyType& InKey, const FValueType& InValue) { return Map.Add(InKey, InValue); }
    FORCEINLINE FValueType& Add(const FKeyType& InKey, FValueType&& InValue) { return Map.Add(InKey, MoveTempIfPossible(InValue)); }
    FORCEINLINE FValueType& Add(FKeyType&& InKey, const FValueType& InValue) { return Map.Add(MoveTempIfPossible(InKey), InValue); }
    FORCEINLINE FValueType& Add(FKeyType&& InKey, FValueType&& InValue) { return Map.Add(MoveTempIfPossible(InKey), MoveTempIfPossible(InValue)); }
    FORCEINLINE FValueType& Add(const FKeyType& InKey) { return Map.Add(InKey); }
    FORCEINLINE FValueType& Add(FKeyType&& InKey) { return Map.Add(MoveTempIfPossible(InKey)); }
    
    template<typename T>
    FValueType& Add(const FKeyType& InKey, T&& InValue)
    {
        return Map.Add(InKey, FValueType(Forward<T>(InValue)));
    }

    // Emplace methods
    template <typename InitKeyType, typename InitValueType>
    FValueType& Emplace(InitKeyType&& InKey, InitValueType&& InValue)
    {
        return Map.Emplace(Forward<InitKeyType>(InKey), Forward<InitValueType>(InValue));
    }

    template <typename InitKeyType>
    FValueType& Emplace(InitKeyType&& InKey)
    {
        return Map.Emplace(Forward<InitKeyType>(InKey));
    }

    // Other TMap methods
    FORCEINLINE int32 Remove(const FKeyType& InKey) { return Map.Remove(InKey); }
    FORCEINLINE bool RemoveAndCopyValue(const FKeyType& Key, FValueType& OutRemovedValue) { return Map.RemoveAndCopyValue(Key, OutRemovedValue); }
    FORCEINLINE FValueType FindAndRemoveChecked(const FKeyType& Key) { return Map.FindAndRemoveChecked(Key); }
    
    FORCEINLINE FValueType* Find(const FKeyType& Key) { return Map.Find(Key); }
    FORCEINLINE const FValueType* Find(const FKeyType& Key) const { return Map.Find(Key); }
    
    FORCEINLINE FValueType& FindOrAdd(const FKeyType& Key) { return Map.FindOrAdd(Key); }
    FORCEINLINE FValueType& FindOrAdd(FKeyType&& Key) { return Map.FindOrAdd(MoveTempIfPossible(Key)); }
    FORCEINLINE FValueType& FindOrAdd(const FKeyType& Key, const FValueType& Value) { return Map.FindOrAdd(Key, Value); }
    FORCEINLINE FValueType& FindOrAdd(const FKeyType& Key, FValueType&& Value) { return Map.FindOrAdd(Key, MoveTempIfPossible(Value)); }
    FORCEINLINE FValueType& FindOrAdd(FKeyType&& Key, const FValueType& Value) { return Map.FindOrAdd(MoveTempIfPossible(Key), Value); }
    FORCEINLINE FValueType& FindOrAdd(FKeyType&& Key, FValueType&& Value) { return Map.FindOrAdd(MoveTempIfPossible(Key), MoveTempIfPossible(Value)); }
    
    FORCEINLINE const FValueType& FindChecked(const FKeyType& Key) const { return Map.FindChecked(Key); }
    FORCEINLINE FValueType& FindChecked(const FKeyType& Key) { return Map.FindChecked(Key); }
    FORCEINLINE FValueType FindRef(const FKeyType& Key) const { return Map.FindRef(Key); }
    FORCEINLINE bool Contains(const FKeyType& Key) const { return Map.Contains(Key); }

    // Array generation methods
    template<typename Allocator> 
    void GenerateKeyArray(TArray<FKeyType, Allocator>& OutArray) const { Map.GenerateKeyArray(OutArray); }
    
    template<typename Allocator> 
    void GenerateValueArray(TArray<FValueType, Allocator>& OutArray) const { Map.GenerateValueArray(OutArray); }
    
    template<typename Allocator> 
    int32 GetKeys(TArray<FKeyType, Allocator>& OutKeys) const { return Map.GetKeys(OutKeys); }
    
    template<typename Allocator> 
    int32 GetKeys(TSet<FKeyType, Allocator>& OutKeys) const { return Map.GetKeys(OutKeys); }

    // Memory management
    FORCEINLINE uint32 GetAllocatedSize() const 
    { 
        return Map.GetAllocatedSize();
    }

    FORCEINLINE void CountBytes(FArchive& Ar) const 
    { 
        Map.CountBytes(Ar);
    }

    // Comparison operators
    bool OrderIndependentCompareEqual(const FNhnVariantMap& Other) const
    {
        return Map.OrderIndependentCompareEqual(Other.Map);
    }

    // Append methods
    template<typename OtherSetAllocator>
    void Append(TMap<FKeyType, FValueType, OtherSetAllocator>& OtherMap) { Map.Append(MoveTemp(OtherMap)); }
    
    template<typename OtherSetAllocator>
    void Append(const TMap<FKeyType, FValueType, OtherSetAllocator>& OtherMap) { Map.Append(OtherMap); }
    
    void Append(const FNhnVariantMap& Other)
    {
        Map.Append(Other.Map);
    }
    
    void Append(FNhnVariantMap&& Other)
    {
        Map.Append(MoveTemp(Other.Map));
    }

    // Operators
    FORCEINLINE FValueType& operator[](const FKeyType& Key) { return FindChecked(Key); }
    FORCEINLINE const FValueType& operator[](const FKeyType& Key) const { return FindChecked(Key); }
    
    // Iterator support
    FORCEINLINE auto CreateIterator() { return Map.CreateIterator(); }
    FORCEINLINE auto CreateConstIterator() const { return Map.CreateConstIterator(); }
    
    // Range-based for loop support using TMap iterators
    FORCEINLINE auto begin() { return Map.begin(); }
    FORCEINLINE auto begin() const { return Map.begin(); }
    FORCEINLINE auto end() { return Map.end(); }
    FORCEINLINE auto end() const { return Map.end(); }

    // Legacy comparison operators
    friend bool LegacyCompareEqual(const FNhnVariantMap& A, const FNhnVariantMap& B)
    {
        return LegacyCompareEqual(A.Map, B.Map);
    }
    
    friend bool LegacyCompareNotEqual(const FNhnVariantMap& A, const FNhnVariantMap& B)
    {
        return !LegacyCompareEqual(A, B);
    }

    // TMap conversion operators
    explicit operator TMap<FKeyType, FValueType>&() { return Map; }
    explicit operator const TMap<FKeyType, FValueType>&() const { return Map; }
    const TMap<FKeyType, FValueType>& GetConstTMap() const { return Map; }
    TMap<FKeyType, FValueType>& GetTMap() { return Map; }
    FVariantMapType& GetMap() { return Map; }
    const FVariantMapType& GetMap() const { return Map; }
    
    FORCEINLINE friend FArchive& operator<<(FArchive& Ar, FNhnVariantMap& InMap)
    {
        Ar << InMap.Map;
        return Ar;
    }
    
    template<typename T>
    void SetOptionalIfExists(const FKeyType& Key, TOptional<T>& OutValue) const
    {
        if (const FNhnVariant* FoundValue = Find(Key))
        {
            if (FoundValue->GetType() == TNhnVariantTraits<T>::GetType())
            {
                OutValue = FoundValue->GetValue<T>();
            }
            else
            {
                OutValue.Reset();
            }
        }
        else
        {
            OutValue.Reset();
        }
    }
    
    template<typename T>
    T GetValueOrDefault(const FKeyType& Key, const T& DefaultValue) const
    {
        if (const FValueType* FoundValue = Map.Find(Key))
        {
            if (FoundValue->GetType() == TNhnVariantTraits<T>::GetType())
            {
                return FoundValue->GetValue<T>();
            }
        }
        return DefaultValue;
    }
    
private:
    FVariantMapType Map;
};

namespace NHN {
namespace Variant {

NHNSHARED_API FString ToQueryToUrl(const FNhnVariantMap& Queries);

} // namespace Variant
} // namespace NHN