#pragma once

#include "Misc/Variant.h"
#include "NhnVariantType.h"
#include "Traits/NhnVariantTraits.h"

using FEngineVariant = FVariant;
using EEngineVariantTypes = EVariantTypes;

class FNhnVariant;
class FNhnVariantMap;
class FNhnVariantArray;

class FNhnVariant
{
public:
    FNhnVariant()
        : Type(ENhnVariantType::Empty)
    {
    }
    
    FNhnVariant(const FNhnVariant& Other)
        : Type(Other.Type)
        , Value(Other.Value)
    {
    }
    
    FNhnVariant(FNhnVariant&& Other)
        : Type(Other.Type)
        , Value(MoveTemp(Other.Value))
    {
        Other.Type = ENhnVariantType::Empty;
        Other.Value.Empty();
    }
    
    ~FNhnVariant() = default;
    
    explicit FNhnVariant(std::nullptr_t)
        : Type(ENhnVariantType::Empty)
    {
    }
    
    explicit FNhnVariant(const FEngineVariant& InValue)
        : Type(TNhnVariantTraits<FEngineVariant>::GetType(InValue))
    {
        if (Type != ENhnVariantType::Empty)
        {
            Value = InValue.GetBytes();
        }
    }

    template<typename T, typename = TEnableIf<TNhnVariantTraits<T>::GetType() != ENhnVariantType::Empty>>
    explicit FNhnVariant(const T& InValue)
        : Type(TNhnVariantTraits<T>::GetType())
    {
        auto ArchiveValue = TNhnVariantTraits<T>::ToArchiveType(InValue);
        FMemoryWriter Writer(Value, true);
        Writer << ArchiveValue;
    }

    FNhnVariant& operator=(const FNhnVariant& Other)
    {
        if (this != &Other)
        {
            Type = Other.Type;
            Value = Other.Value;
        }
        return *this;
    }
    
    FNhnVariant& operator=(FNhnVariant&& Other)
    {
        if (this != &Other)
        {
            Type = Other.Type;
            Value = MoveTemp(Other.Value);
            Other.Type = ENhnVariantType::Empty;
            Other.Value.Empty();
        }
        return *this;
    }

    template<typename T>
    FNhnVariant& operator=(const T& InValue) = delete;
    
    FORCEINLINE ENhnVariantType GetType() const
    {
        return Type;
    }

    FORCEINLINE bool IsEmpty() const
    {
        return Type == ENhnVariantType::Empty;
    }
    
    template<typename T>
    T GetValue() const
    {
        if (Type == TNhnVariantTraits<T>::GetType() && Value.Num() > 0)
        {
            T Result;
            FMemoryReader Reader(Value, true);
            Reader << Result;
            return Result;
        }
        return T();
    }

    template<typename T>
    operator T() const
    {
        return GetValue<T>();
    }
    
    bool operator==(const FNhnVariant& Other) const
    {
        return Type == Other.Type && Value == Other.Value;
    }
    bool operator!=(const FNhnVariant& Other) const
    {
        return !(*this == Other);
    }

    friend FArchive& operator<<(FArchive& Ar, FNhnVariant& InVar)
    {
        Ar << InVar.Type;
        Ar << InVar.Value;

        return Ar;
    }

private:
    ENhnVariantType Type;
    TArray<uint8> Value;
};