#pragma once

#include "Serialization/JsonReader.h"
#include "Serialization/JsonSerializer.h"
#include "NhnTraits.h"
#include "Templates/UnrealTypeTraits.h"

namespace NHN {
namespace Json {

// ================================================================================
// Type trait for value object detection
// ================================================================================
/**
 * Trait to check if a type is a value object (struct/class) that supports FromJson.
 * Returns true if T is not a pointer, not a primitive, not a shared pointer, and has a FromJson member function.
 */
template<typename T>
struct TIsValueObjectWithFromJson
{
    static constexpr bool Value =
        !TIsPointer<T>::Value &&
        !TIsSame<T, FString>::Value &&
        !TIsSame<T, bool>::Value &&
        !TIsIntegral<T>::Value &&
        !TIsFloatingPoint<T>::Value &&
        !TIsSharedPtr<T>::Value &&
        THasFromJson<T>::Value;
};

// ================================================================================
// TConvertTraits base template
// ================================================================================
/**
 * Base template for TConvertTraits (unsupported type conversion).
 * If there is no specialization, static_assert will trigger a compile error.
 */
template<typename T, typename Enable = void>
struct TConvertTraits
{
    static_assert(sizeof(T) == -1, "GamebaseCommunicatorConverter: Unsupported type conversion. Please check Traits specialization.");
};

// ================================================================================
// Primitive type specializations
// ================================================================================
/** Converts JSON string or FJsonValue to int32. */
template<>
struct TConvertTraits<int32, void>
{
    static FORCEINLINE int32 Convert(const FString& Json)
    {
        return FCString::Atoi(*Json);
    }
    static FORCEINLINE int32 Convert(const TSharedPtr<FJsonValue>& JsonValue)
    {
        return static_cast<int32>(JsonValue->AsNumber());
    }
};

/** Converts JSON string or FJsonValue to int64. */
template<>
struct TConvertTraits<int64, void>
{
    static FORCEINLINE int64 Convert(const FString& Json)
    {
        return FCString::Atoi64(*Json);
    }
    static FORCEINLINE int64 Convert(const TSharedPtr<FJsonValue>& JsonValue)
    {
        return static_cast<int64>(JsonValue->AsNumber());
    }
};

/** Converts JSON string or FJsonValue to uint32. */
template<>
struct TConvertTraits<uint32, void>
{
    static FORCEINLINE uint32 Convert(const FString& Json)
    {
        return static_cast<uint32>(FCString::Strtoui64(*Json, nullptr, 10));
    }
    static FORCEINLINE uint32 Convert(const TSharedPtr<FJsonValue>& JsonValue)
    {
        return static_cast<uint32>(JsonValue->AsNumber());
    }
};

/** Converts JSON string or FJsonValue to uint64. */
template<>
struct TConvertTraits<uint64, void>
{
    static FORCEINLINE uint64 Convert(const FString& Json)
    {
        return FCString::Strtoui64(*Json, nullptr, 10);
    }
    static FORCEINLINE uint64 Convert(const TSharedPtr<FJsonValue>& JsonValue)
    {
        return static_cast<uint64>(JsonValue->AsNumber());
    }
};

/** Converts JSON string or FJsonValue to float. */
template<>
struct TConvertTraits<float, void>
{
    static FORCEINLINE float Convert(const FString& Json)
    {
        return FCString::Atof(*Json);
    }
    static FORCEINLINE float Convert(const TSharedPtr<FJsonValue>& JsonValue)
    {
        return static_cast<float>(JsonValue->AsNumber());
    }
};

/** Converts JSON string or FJsonValue to double. */
template<>
struct TConvertTraits<double, void>
{
    static FORCEINLINE double Convert(const FString& Json)
    {
        return FCString::Atod(*Json);
    }
    static FORCEINLINE double Convert(const TSharedPtr<FJsonValue>& JsonValue)
    {
        return JsonValue->AsNumber();
    }
};

/** Converts JSON string or FJsonValue to bool. */
template<>
struct TConvertTraits<bool, void>
{
    static FORCEINLINE bool Convert(const FString& Json)
    {
        return Json.ToBool();
    }
    static FORCEINLINE bool Convert(const TSharedPtr<FJsonValue>& JsonValue)
    {
        return JsonValue->AsBool();
    }
};

/** Converts JSON string or FJsonValue to FString. */
template<>
struct TConvertTraits<FString, void>
{
    static FORCEINLINE FString Convert(const FString& Json)
    {
        return Json;
    }
    static FORCEINLINE FString Convert(const TSharedPtr<FJsonValue>& JsonValue)
    {
        return JsonValue->AsString();
    }
};

// ================================================================================
// Special type specializations
// ================================================================================
/**
 * Converts JSON to a struct that implements FromJson.
 * Used for value objects (struct/class) with FromJson support.
 */
template<typename T>
struct TConvertTraits<T, typename TEnableIf<TIsValueObjectWithFromJson<T>::Value>::Type>
{
    static FORCEINLINE T Convert(const FString& Json)
    {
        T Result;
        Result.FromJson(Json);
        return Result;
    }
    static FORCEINLINE T Convert(const TSharedPtr<FJsonValue>& JsonValue)
    {
        T Result;
        const TSharedPtr<FJsonObject> Obj = JsonValue->AsObject();
        if (Obj.IsValid())
        {
            FString ItemJson;
            const TSharedRef<TJsonWriter<>> Writer = TJsonWriterFactory<>::Create(&ItemJson);
            FJsonSerializer::Serialize(Obj.ToSharedRef(), Writer);
            Writer->Close();
            Result.FromJson(ItemJson);
        }
        else
        {
            // Handle primitive JsonValue (e.g., int, string, etc.)
            Result.FromJson(JsonValue->AsString());
        }
        return Result;
    }
};

/**
 * Converts JSON array to TArray<T>.
 */
template<typename T>
struct TConvertTraits<TArray<T>, void>
{
    static FORCEINLINE TArray<T> Convert(const FString& Json)
    {
        TArray<T> Result;
        const TSharedRef<TJsonReader<>> Reader = TJsonReaderFactory<>::Create(Json);
        TArray<TSharedPtr<FJsonValue>> JsonArray;
        if (FJsonSerializer::Deserialize(Reader, JsonArray))
        {
            for (const TSharedPtr<FJsonValue>& JsonValue : JsonArray)
            {
                Result.Add(TConvertTraits<T>::Convert(JsonValue));
            }
        }
        return Result;
    }
};

/**
 * Converts JSON to pointer to TArray<T>.
 */
template<typename T, typename Allocator>
struct TConvertTraits<TArray<T, Allocator>*, void>
{
    static FORCEINLINE TArray<T, Allocator>* Convert(const FString& Json)
    {
        TArray<T, Allocator>* Result = new TArray<T, Allocator>();
        *Result = TConvertTraits<TArray<T, Allocator>, void>::Convert(Json);
        return Result;
    }
};

/**
 * Converts JSON to pointer to ValueObject.
 */
template<typename T>
struct TConvertTraits<const T*, void>
{
    static FORCEINLINE const T* Convert(const FString& Json)
    {
        T* Result = new T();
        *Result = TConvertTraits<T, void>::Convert(Json);
        return Result;
    }
};

/**
 * Converts JSON to pointer to TArray<ValueObject>.
 */
template<typename T, typename Allocator>
struct TConvertTraits<const TArray<T, Allocator>*, void>
{
    static FORCEINLINE const TArray<T, Allocator>* Convert(const FString& Json)
    {
        TArray<T, Allocator>* Result = new TArray<T, Allocator>();
        *Result = TConvertTraits<TArray<T, Allocator>, void>::Convert(Json);
        return Result;
    }
};

} // namespace Json
} // namespace NHN
