#pragma once

#include "NhnVariant.h"
#include "NhnVariantArray.h"
#include "NhnVariantMap.h"

#include "Serialization/JsonSerializerMacros.h"

namespace NHN {
namespace Json
{
/**
 * @brief Serializes/deserializes FLinearColor to/from a JSON object.
 * @param Serializer JSON serializer
 * @param JsonName JSON field name
 * @param Color Target color to serialize/deserialize
 */
NHNJSON_API void SerializeColor(FJsonSerializerBase& Serializer, const FString& JsonName, FLinearColor& Color);

/**
 * @brief Serializes/deserializes TOptional<FLinearColor> to/from a JSON object.
 * @param Serializer JSON serializer
 * @param JsonName JSON field name
 * @param Color Target optional color to serialize/deserialize
 */
NHNJSON_API void SerializeOptionalColor(FJsonSerializerBase& Serializer, const FString& JsonName, TOptional<FLinearColor>& Color);

/**
 * @brief Serializes/deserializes FColor to/from a JSON object. Uses FLinearColor conversion internally.
 * @param Serializer JSON serializer
 * @param JsonName JSON field name
 * @param Color Target color to serialize/deserialize
 */
NHNJSON_API void SerializeColor(FJsonSerializerBase& Serializer, const FString& JsonName, FColor& Color);

/**
 * @brief Serializes/deserializes TOptional<FColor> to/from a JSON object. Uses FLinearColor conversion internally.
 * @param Serializer JSON serializer
 * @param JsonName JSON field name
 * @param Color Target optional color to serialize/deserialize
 */
NHNJSON_API void SerializeOptionalColor(FJsonSerializerBase& Serializer, const FString& JsonName, TOptional<FColor>& Color);

/**
 * @brief Serializes/deserializes FVariantMap to/from a JSON object.
 * @param Serializer JSON serializer
 * @param JsonName JSON field name
 * @param VariantMap Target map to serialize/deserialize
 */
NHNJSON_API void SerializeVariantMap(FJsonSerializerBase& Serializer, const FString& JsonName, FNhnVariantMap& VariantMap);

/**
 * @brief Serializes/deserializes TOptional<FVariantMap> to/from a JSON object.
 * @param Serializer JSON serializer
 * @param JsonName JSON field name
 * @param OptionalVariantMap Target optional map to serialize/deserialize
 */
NHNJSON_API void SerializeOptionalVariantMap(FJsonSerializerBase& Serializer, const FString& JsonName, TOptional<FNhnVariantMap>& OptionalVariantMap);

/**
 * @brief Serializes/deserializes FVariantArray to/from a JSON array.
 * @param Serializer JSON serializer
 * @param JsonName JSON field name
 * @param VariantArray Target array to serialize/deserialize
 */
NHNJSON_API void SerializeVariantArray(FJsonSerializerBase& Serializer, const FString& JsonName, FNhnVariantArray& VariantArray);

/**
 * @brief Serializes/deserializes TOptional<FVariantArray> to/from a JSON array.
 * @param Serializer JSON serializer
 * @param JsonName JSON field name
 * @param OptionalVariantArray Target optional array to serialize/deserialize
 */
NHNJSON_API void SerializeOptionalVariantArray(FJsonSerializerBase& Serializer, const FString& JsonName, TOptional<FNhnVariantArray>& OptionalVariantArray);

/**
 * @brief Serializes/deserializes FVariant to/from a JSON object. Stores type information and value together.
 * @param Serializer JSON serializer
 * @param JsonName JSON field name
 * @param Variant Target variant to serialize/deserialize
 */
NHNJSON_API void SerializeVariant(FJsonSerializerBase& Serializer, const FString& JsonName, FNhnVariant& Variant);

/**
 * @brief Serializes/deserializes TSharedPtr<T, Mode> to/from a JSON object.
 * @tparam T Target type to serialize/deserialize
 * @tparam Mode ESPMode type
 * @param Serializer JSON serializer
 * @param JsonName JSON field name
 * @param PtrField Target pointer to serialize/deserialize
 */
template<typename T, ESPMode Mode>
void SerializePtrObject(FJsonSerializerBase& Serializer, const FString& JsonName, TSharedPtr<T, Mode>& PtrField)
{
    if (Serializer.IsLoading())
    {
        if (Serializer.GetObject()->HasTypedField<EJson::Object>(JsonName))
        {
            const TSharedPtr<FJsonObject> JsonObj = Serializer.GetObject()->GetObjectField(JsonName);
            if (JsonObj.IsValid())
            {
                PtrField = MakeShared<T, Mode>();
                FJsonSerializerReader InnerSerializer(JsonObj);
                PtrField->Serialize(InnerSerializer, false);
            }
        }
    }
    else
    {
        if (PtrField.IsValid())
        {
            Serializer.StartObject(JsonName);
            PtrField->Serialize(Serializer, true);
            Serializer.EndObject();
        }
    }
}

/**
 * @brief Serializes/deserializes Optional<TMap<FString, FString>> to/from a JSON object. Only supports Unreal standard Map.
 * @tparam MapType Target map type to serialize/deserialize
 * @param Serializer JSON serializer
 * @param JsonName JSON field name
 * @param OptionalJsonMap Target optional map to serialize/deserialize
 */
template<typename MapType>
void SerializeOptionalMap(FJsonSerializerBase& Serializer, const FString& JsonName, TOptional<MapType>& OptionalJsonMap)
{
    if (Serializer.IsLoading())
    {
        if (Serializer.GetObject()->HasTypedField<EJson::Object>(JsonName))
        {
            TSharedPtr<FJsonObject> JsonObj = Serializer.GetObject()->GetObjectField(JsonName);
            if (JsonObj.IsValid())
            {
                OptionalJsonMap.Emplace();
                for (auto It = JsonObj->Values.CreateConstIterator(); It; ++It)
                {
                    OptionalJsonMap.GetValue().Add(It.Key(), It.Value()->AsString());
                }
            }
        }
    }
    else
    {
        if (OptionalJsonMap.IsSet())
        {
            Serializer.StartObject(JsonName);
            for (auto It = OptionalJsonMap.GetValue().CreateConstIterator(); It; ++It)
            {
                auto ValueCopy = It.Value();
                Serializer.Serialize(*It.Key(), ValueCopy);
            }
            Serializer.EndObject();
        }
    }
}

/**
 * @brief Serializes/deserializes Optional<TMap<KeyType, ValueType>> object maps to/from a JSON object. Also serializes internal objects.
 * @tparam ElementType Value type of the map
 * @param Serializer JSON serializer
 * @param JsonName JSON field name
 * @param OptionalJsonMap Target optional map to serialize/deserialize
 */
template<typename ElementType>
void SerializeOptionalMapSerializable(FJsonSerializerBase& Serializer, const FString& JsonName, TOptional<TMap<FString, ElementType>>& OptionalJsonMap)
{
    if (Serializer.IsLoading())
    {
        if (Serializer.GetObject()->HasTypedField<EJson::Object>(JsonName))
        {
            const TSharedPtr<FJsonObject> JsonObj = Serializer.GetObject()->GetObjectField(JsonName);
            TMap<FString, ElementType>& JsonMap = OptionalJsonMap.Emplace();
            for (auto MapIt = JsonObj->Values.CreateConstIterator(); MapIt; ++MapIt)
            {
                ElementType NewEntry;
                NewEntry.FromJson(MapIt.Value()->AsObject());
                JsonMap.Add(MapIt.Key(), NewEntry);
            }
        }
    }
    else
    {
        if (OptionalJsonMap.IsSet())
        {
            Serializer.StartObject(JsonName);
            for (auto It = OptionalJsonMap->CreateIterator(); It; ++It)
            {
                Serializer.StartObject(It.Key());
                It.Value().Serialize(Serializer, true);
                Serializer.EndObject();
            }
            Serializer.EndObject();
        }
    }
}

/**
 * @brief Serializes/deserializes Optional<TArray> to/from a JSON array. Only supports Unreal standard types.
 * @tparam ArrayType Target array type to serialize/deserialize
 * @param Serializer JSON serializer
 * @param JsonName JSON field name
 * @param OptionalJsonArray Target optional array to serialize/deserialize
 */
template<typename ArrayType>
void SerializeOptionalArray(FJsonSerializerBase& Serializer, const FString& JsonName, TOptional<ArrayType>& OptionalJsonArray)
{
    if (Serializer.IsLoading())
    {
        if (Serializer.GetObject()->HasTypedField<EJson::Array>(JsonName))
        {
            OptionalJsonArray.Emplace();
            Serializer.SerializeArray(*JsonName, OptionalJsonArray.GetValue());
        }
    }
    else
    {
        if (OptionalJsonArray.IsSet())
        {
            Serializer.SerializeArray(*JsonName, OptionalJsonArray.GetValue());
        }
    }
}

/**
 * @brief FNhnVariantMap을 JSON 문자열로 변환합니다.
 */
NHNJSON_API FString SerializeVariantMapToJsonString(const FNhnVariantMap& VariantMap);

/**
 * @brief TOptional<FNhnVariantMap>을 JSON 문자열로 변환합니다.
 */
NHNJSON_API FString SerializeOptionalVariantMapToJsonString(const TOptional<FNhnVariantMap>& OptionalVariantMap);

} // namespace Json
} // namespace NHN