#pragma once

#include "NhnJsonSerializable.h"
#include "NhnJsonSerializer.h"
#include "Serialization/JsonSerializerMacros.h"

// -------------------- Struct-Level Serialization Block --------------------
/**
 * @brief Use at the beginning of struct serialization function (wraps Unreal's JSON serializer)
 * @note This is used inside struct-level JSON serialization, not in FNhnValueObject::Serialize()
 */
#define BEGIN_NHN_JSON_SERIALIZER      BEGIN_JSON_SERIALIZER

/**
 * @brief Use at the end of struct serialization function (wraps Unreal's JSON serializer)
 * @note This is used inside struct-level JSON serialization, not in FNhnValueObject::Serialize()
 */
#define END_NHN_JSON_SERIALIZER        END_JSON_SERIALIZER

// -------------------- ValueObject Serialization Block --------------------
/**
 * @brief Use at the beginning of FNhnValueObject::Serialize() method
 * @note This starts a JSON object conditionally based on bFlatObject parameter
 */
#define BEGIN_NHN_VO_SERIALIZER \
    if (!bFlatObject) { Serializer.StartObject(); }

/**
 * @brief Use at the end of FNhnValueObject::Serialize() method
 * @note This ends a JSON object conditionally based on bFlatObject parameter
 */
#define END_NHN_VO_SERIALIZER \
    if (!bFlatObject) { Serializer.EndObject(); }

// -------------------- Basic Field/Optional/Enum Serialization --------------------
/**
 * @brief Serialize a normal field: serializes the JSON field name and value
 */
#define NHN_JSON_SERIALIZE(JsonName, JsonValue) \
    Serializer.Serialize(TEXT(JsonName), JsonValue)

/**
 * @brief Serialize string field only if it's not empty
 * @note This macro checks if the string value is empty before serialization to avoid writing empty fields
 */
#define NHN_JSON_SERIALIZE_STRING(JsonName, JsonValue) \
     if (!JsonValue.IsEmpty()) \
     { \
         Serializer.Serialize(TEXT(JsonName), JsonValue); \
     }

/**
 * @brief Serialize raw JSON string field (wraps Unreal's JSON serializer)
 * @note This is used for serializing pre-formatted JSON strings as raw values
 */
#define NHN_JSON_SERIALIZE_RAW_JSON_STRING(JsonName, JsonValue) \
    JSON_SERIALIZE_RAW_JSON_STRING(JsonName, JsonValue)

/**
 * @brief Serialize/deserialize TOptional<T> field: only if value is set
 */
#define NHN_JSON_SERIALIZE_OPTIONAL(JsonName, OptionalJsonValue) \
    JSON_SERIALIZE_OPTIONAL(JsonName, OptionalJsonValue)

/**
 * @brief Serialize/deserialize enum value as uint32 index
 */
#define NHN_JSON_SERIALIZE_ENUM_INDEX(JsonName, JsonValue, ElementType) \
    uint32 JsonValueDummy; \
    Serializer.Serialize(TEXT(JsonName), JsonValueDummy); \
    JsonValue = static_cast<ElementType>(JsonValueDummy);

// -------------------- Color/Special Type Serialization --------------------
/**
 * @brief Serialize/deserialize FLinearColor type by decomposing it into r,g,b,a
 */
#define NHN_JSON_SERIALIZE_COLOR(JsonName, JsonColor) \
    NHN::Json::SerializeColor(Serializer, JsonName, JsonColor)

/**
 * @brief Serialize/deserialize TOptional<FLinearColor> type by decomposing it into r,g,b,a (only if value is set)
 */
#define NHN_JSON_SERIALIZE_COLOR_OPTIONAL(JsonName, JsonColor) \
    NHN::Json::SerializeOptionalColor(Serializer, JsonName, JsonColor)

// -------------------- Array Serialization --------------------
/**
 * @brief Serialize/deserialize TArray<int32/float/FString etc. basic types> array
 */
#define NHN_JSON_SERIALIZE_ARRAY(JsonName, JsonArray) \
    JSON_SERIALIZE_ARRAY(JsonName, JsonArray)

/**
 * @brief Serialize/deserialize TArray<SerializableType> etc. object array (ElementType is struct/class)
 */
#define NHN_JSON_SERIALIZE_ARRAY_SERIALIZABLE(JsonName, JsonArray, ElementType) \
    JSON_SERIALIZE_ARRAY_SERIALIZABLE(JsonName, JsonArray, ElementType)

/**
 * @brief Serialize/deserialize TOptional<TArray<SerializableType>> etc. object array Optional
 */
#define NHN_JSON_SERIALIZE_OPTIONAL_ARRAY_SERIALIZABLE(JsonName, OptionalJsonArray, ElementType) \
    JSON_SERIALIZE_OPTIONAL_ARRAY_SERIALIZABLE(JsonName, OptionalJsonArray, ElementType)

/**
 * @brief Serialize/deserialize TOptional<TArray<PrimitiveType>> etc. basic type array Optional
 */
#define NHN_JSON_SERIALIZE_OPTIONAL_ARRAY(JsonName, OptionalJsonArray) \
    NHN::SerializeOptionalArray(Serializer, JsonName, OptionalJsonArray)

// -------------------- Map Serialization --------------------
/**
 * @brief Serialize/deserialize TMap<FString, FString> etc. string map
 * @code
 * NHN_JSON_SERIALIZE_MAP("FieldName", MapField);
 * @endcode
 */
#define NHN_JSON_SERIALIZE_MAP(JsonName, JsonMap) \
    JSON_SERIALIZE_MAP(JsonName, JsonMap)

/**
 * @brief Serialize/deserialize object maps such as TMap<FString, SerializableType> (Value is struct/class)
 * @code
 * NHN_JSON_SERIALIZE_MAP_SERIALIZABLE("FieldName", MapField, FMyStruct)
 * @endcode
 */
#define NHN_JSON_SERIALIZE_MAP_SERIALIZABLE(JsonName, JsonMap, ElementType) \
    JSON_SERIALIZE_MAP_SERIALIZABLE(JsonName, JsonMap, ElementType)

/**
 * @brief Serialize/deserialize Optional string maps such as TOptional<TMap<FString, FString>>
 * @code
 * NHN_JSON_SERIALIZE_OPTIONAL_MAP("FieldName", OptionalMapField);
 * @endcode
 */
#define NHN_JSON_SERIALIZE_OPTIONAL_MAP(JsonName, OptionalJsonMap) \
    NHN::Json::SerializeOptionalMap(Serializer, JsonName, OptionalJsonMap)

/**
 * @brief Serialize/deserialize Optional object maps such as TOptional<TMap<FString, SerializableType>>
 * @code
 * NHN_JSON_SERIALIZE_OPTIONAL_MAP_SERIALIZABLE("FieldName", OptionalMapField, FMyStruct)
 * @endcode
 */
#define NHN_JSON_SERIALIZE_OPTIONAL_MAP_SERIALIZABLE(JsonName, OptionalJsonMap, ElementType) \
    NHN::Json::SerializeOptionalMapSerializable<ElementType>(Serializer, JsonName, OptionalJsonMap)

// -------------------- Object Serialization --------------------
/**
 * @brief Serialize/deserialize a single object (SerializableType)
 * @code
 * NHN_JSON_SERIALIZE_OBJECT_SERIALIZABLE("FieldName", ObjectField);
 * @endcode
 */
#define NHN_JSON_SERIALIZE_OBJECT_SERIALIZABLE(JsonName, JsonSerializableObject) \
    JSON_SERIALIZE_OBJECT_SERIALIZABLE(JsonName, JsonSerializableObject)

/**
 * @brief Serialize object as JSON string using its ToJson() method
 * @note This converts the object to JSON string using ToJson(false) and serializes it as a string field
 * @code
 * NHN_JSON_SERIALIZE_OBJECT_STRING_SERIALIZABLE("FieldName", ObjectField);
 * @endcode
 */
#define NHN_JSON_SERIALIZE_OBJECT_STRING_SERIALIZABLE(JsonName, JsonValue) \
    FString JsonTextValue = JsonValue.ToJson(false); \
    Serializer.Serialize(TEXT(JsonName), JsonTextValue)

/**
 * @brief Serialize/deserialize Optional single object (SerializableType)
 * @code
 * NHN_JSON_SERIALIZE_OPTIONAL_OBJECT_SERIALIZABLE("FieldName", OptionalObjectField);
 * @endcode
 */
#define NHN_JSON_SERIALIZE_OPTIONAL_OBJECT_SERIALIZABLE(JsonName, JsonSerializableObject) \
    JSON_SERIALIZE_OPTIONAL_OBJECT_SERIALIZABLE(JsonName, JsonSerializableObject)

// -------------------- Pointer Serialization --------------------
/**
 * @brief Serialize TSharedPtr<T> object (specify ESPModeType)
 * @code
 * NHN_JSON_SERIALIZE_PTR_OBJECT("FieldName", PtrField, FMyType, ESPMode::ThreadSafe);
 * @endcode
 */
#define NHN_JSON_SERIALIZE_PTR_OBJECT(JsonName, PtrField, Type, ESPModeType) \
    NHN::Json::SerializePtrObject<Type, ESPModeType>(Serializer, JsonName, PtrField)

/**
 * @brief Serialize TSharedPtr<T> object (ThreadSafe)
 * @code
 * NHN_JSON_SERIALIZE_PTR_OBJECT_THREADSAFE("FieldName", PtrField, FMyType);
 * @endcode
 */
#define NHN_JSON_SERIALIZE_PTR_OBJECT_THREADSAFE(JsonName, PtrField, Type) \
    NHN_JSON_SERIALIZE_PTR_OBJECT(JsonName, PtrField, Type, ESPMode::ThreadSafe)

// -------------------- Variant Type Serialization --------------------
/**
 * @brief Serialize FGamebaseVariant
 * @code
 * NHN_JSON_SERIALIZE_VARIANT("FieldName", VariantField);
 * @endcode
 */
#define NHN_JSON_SERIALIZE_VARIANT(JsonName, Variant) \
    NHN::Json::SerializeVariant(Serializer, JsonName, Variant)

/**
 * @brief Serialize FGamebaseVariantArray
 * @code
 * NHN_JSON_SERIALIZE_VARIANT_ARRAY("FieldName", VariantArrayField);
 * @endcode
 */
#define NHN_JSON_SERIALIZE_VARIANT_ARRAY(JsonName, VariantArray) \
    NHN::Json::SerializeVariantArray(Serializer, JsonName, VariantArray)

/**
 * @brief Serialize TOptional<FGamebaseVariantArray>
 * @code
 * NHN_JSON_SERIALIZE_OPTIONAL_VARIANT_ARRAY("FieldName", OptionalVariantArrayField);
 * @endcode
 */
#define NHN_JSON_SERIALIZE_OPTIONAL_VARIANT_ARRAY(JsonName, OptionalVariantArray) \
    NHN::Json::SerializeOptionalVariantArray(Serializer, JsonName, OptionalVariantArray)

/**
 * @brief Serialize FGamebaseVariantMap
 * @code
 * NHN_JSON_SERIALIZE_VARIANT_MAP("FieldName", VariantMapField);
 * @endcode
 */
#define NHN_JSON_SERIALIZE_VARIANT_MAP(JsonName, VariantMap) \
    NHN::Json::SerializeVariantMap(Serializer, JsonName, VariantMap)

/**
 * @brief Serialize TOptional<FGamebaseVariantMap>
 * @code
 * NHN_JSON_SERIALIZE_OPTIONAL_VARIANT_MAP("FieldName", OptionalVariantMapField);
 * @endcode
 */
#define NHN_JSON_SERIALIZE_OPTIONAL_VARIANT_MAP(JsonName, OptionalVariantMap) \
    NHN::Json::SerializeOptionalVariantMap(Serializer, JsonName, OptionalVariantMap)
