#include "NhnJsonSerializer.h"

#include "NhnJsonUtilites.h"

#include "Dom/JsonObject.h"
#include "Serialization/JsonSerializerMacros.h"

namespace NHN {
namespace Json {

constexpr const TCHAR* DummyArrayElementName = TEXT("");

void SerializeColor(FJsonSerializerBase& Serializer, const FString& JsonName, FLinearColor& Color)
{
    if (Serializer.IsLoading())
    {
        if (Serializer.GetObject()->HasTypedField<EJson::Object>(JsonName))
        {
            const TSharedPtr<FJsonObject> JsonObj = Serializer.GetObject()->GetObjectField(JsonName);
            if (JsonObj.IsValid())
            {
                FJsonSerializerReader InnerSerializer(JsonObj);
                float R = 0, G = 0, B = 0, A = 1;
                InnerSerializer.Serialize(TEXT("r"), R);
                InnerSerializer.Serialize(TEXT("g"), G);
                InnerSerializer.Serialize(TEXT("b"), B);
                InnerSerializer.Serialize(TEXT("a"), A);
                Color = FLinearColor(R, G, B, A);
            }
        }
    }
    else
    {
        Serializer.StartObject(JsonName);
        Serializer.Serialize(TEXT("r"), Color.R);
        Serializer.Serialize(TEXT("g"), Color.G);
        Serializer.Serialize(TEXT("b"), Color.B);
        Serializer.Serialize(TEXT("a"), Color.A);
        Serializer.EndObject();
    }
}

void SerializeOptionalColor(FJsonSerializerBase& Serializer, const FString& JsonName, TOptional<FLinearColor>& Color)
{
    if (Serializer.IsLoading())
    {
        if (Serializer.GetObject()->HasTypedField<EJson::Object>(JsonName))
        {
            FLinearColor LinearColor;
            SerializeColor(Serializer, JsonName, LinearColor);
            Color = LinearColor;
        }
    }
    else
    {
        if (Color.IsSet())
        {
            SerializeColor(Serializer, JsonName, Color.GetValue());
        }
    }
}

void SerializeColor(FJsonSerializerBase& Serializer, const FString& JsonName, FColor& Color)
{
    FLinearColor LinearColor = Color.ReinterpretAsLinear();
    SerializeColor(Serializer, JsonName, LinearColor);
    Color = LinearColor.ToFColor(true);
}

void SerializeOptionalColor(FJsonSerializerBase& Serializer, const FString& JsonName, TOptional<FColor>& Color)
{
    if (Serializer.IsLoading())
    {
        if (Serializer.GetObject()->HasTypedField<EJson::Object>(JsonName))
        {
            FLinearColor LinearColor;
            SerializeColor(Serializer, JsonName, LinearColor);
            Color = LinearColor.ToFColor(true);
        }
    }
    else
    {
        if (Color.IsSet())
        {
            SerializeColor(Serializer, JsonName, Color.GetValue());
        }
    }
}

void SerializeVariantMap(FJsonSerializerBase& Serializer, const FString& JsonName, ::FNhnVariantMap& VariantMap)
{
    if (Serializer.IsLoading())
    {
        if (Serializer.GetObject()->HasTypedField<EJson::Object>(JsonName))
        {
            const TSharedPtr<FJsonObject> JsonObj = Serializer.GetObject()->GetObjectField(JsonName);
            if (JsonObj.IsValid())
            {
                VariantMap = ::FNhnVariantMap();
                for (auto It = JsonObj->Values.CreateConstIterator(); It; ++It)
                {
                    VariantMap.Add(FName(It.Key()), FNhnVariant(It.Value()->AsString()));
                }
            }
        }
    }
    else
    {
        Serializer.StartObject(JsonName);
        for (auto It = VariantMap.GetMap().CreateConstIterator(); It; ++It)
        {
            FString ValueStr = It.Value().GetValue<FString>();
            Serializer.Serialize(*It.Key().ToString(), ValueStr);
        }
        Serializer.EndObject();
    }
}

void SerializeOptionalVariantMap(FJsonSerializerBase& Serializer, const FString& JsonName, TOptional<::FNhnVariantMap>& OptionalVariantMap)
{
    if (Serializer.IsLoading())
    {
        ::FNhnVariantMap TempMap;
        SerializeVariantMap(Serializer, JsonName, TempMap);
        if (TempMap.GetMap().Num() > 0)
        {
            OptionalVariantMap = TempMap;
        }
    }
    else
    {
        if (OptionalVariantMap.IsSet())
        {
            SerializeVariantMap(Serializer, JsonName, OptionalVariantMap.GetValue());
        }
    }
}

void SerializeVariantArray(FJsonSerializerBase& Serializer, const FString& JsonName, ::FNhnVariantArray& VariantArray)
{
}

void SerializeOptionalVariantArray(FJsonSerializerBase& Serializer, const FString& JsonName, TOptional<::FNhnVariantArray>& OptionalVariantArray)
{
    if (Serializer.IsLoading())
    {
        if (Serializer.GetObject()->HasTypedField<EJson::Array>(JsonName))
        {
            ::FNhnVariantArray TempArray;
            SerializeVariantArray(Serializer, JsonName, TempArray);
            if (TempArray.Num() > 0)
            {
                OptionalVariantArray = TempArray;
            }
        }
    }
    else
    {
        if (OptionalVariantArray.IsSet())
        {
            SerializeVariantArray(Serializer, JsonName, OptionalVariantArray.GetValue());
        }
    }
}

void SerializeVariant(FJsonSerializerBase& Serializer, const FString& JsonName, FNhnVariant& Variant)
{
    if (Serializer.IsLoading())
    {
        if (Serializer.GetObject()->HasTypedField<EJson::String>(JsonName))
        {
            FString Value;
            Serializer.Serialize(*JsonName, Value);
            Variant = FNhnVariant(Value);
        }
        else if (Serializer.GetObject()->HasTypedField<EJson::Number>(JsonName))
        {
            double Value = 0.0;
            Serializer.Serialize(*JsonName, Value);
            Variant = FNhnVariant(Value);
        }
        else if (Serializer.GetObject()->HasTypedField<EJson::Boolean>(JsonName))
        {
            bool Value = false;
            Serializer.Serialize(*JsonName, Value);
            Variant = FNhnVariant(Value);
        }
        else if (Serializer.GetObject()->HasTypedField<EJson::Array>(JsonName))
        {
            const TArray<TSharedPtr<FJsonValue>> JsonArray = Serializer.GetObject()->GetArrayField(JsonName);
            FNhnVariantArray ArrayVariants;
            for (const auto& Elem : JsonArray)
            {
                ArrayVariants.Add(ConvertJsonToVariant(Elem));
            }
            Variant = FNhnVariant(ArrayVariants);
        }
        else if (Serializer.GetObject()->HasTypedField<EJson::Object>(JsonName))
        {
            const TSharedPtr<FJsonObject> JsonObj = Serializer.GetObject()->GetObjectField(JsonName);
            if (JsonObj.IsValid())
            {
                const TSharedPtr<FJsonValue> JsonValue = MakeShareable(new FJsonValueObject(JsonObj));
                Variant = ConvertJsonToVariant(JsonValue);
            }
        }
        else if (Serializer.GetObject()->HasTypedField<EJson::Null>(JsonName) || Serializer.GetObject()->HasTypedField<EJson::None>(JsonName))
        {
            Variant = FNhnVariant();
        }
    }
    else
    {
        switch (Variant.GetType())
        {
            case ENhnVariantType::Int32:
                {
                    int32 TempValue = Variant.GetValue<int32>();
                    Serializer.Serialize(*JsonName, TempValue);
                    break;
                }
            case ENhnVariantType::Float:
                {
                    float TempValue = Variant.GetValue<float>();
                    Serializer.Serialize(*JsonName, TempValue);
                    break;
                }
            case ENhnVariantType::Bool:
                {
                    bool TempValue = Variant.GetValue<bool>();
                    Serializer.Serialize(*JsonName, TempValue);
                    break;
                }
            case ENhnVariantType::String:
                {
                    FString TempValue = Variant.GetValue<FString>();
                    Serializer.Serialize(*JsonName, TempValue);
                    break;
                }
            case ENhnVariantType::Double:
                {
                    double TempValue = Variant.GetValue<double>();
                    Serializer.Serialize(*JsonName, TempValue);
                    break;
                }
            case ENhnVariantType::Int64:
                {
                    int64 TempValue = Variant.GetValue<int64>();
                    Serializer.Serialize(*JsonName, TempValue);
                    break;
                }
            case ENhnVariantType::UInt64:
                {
                    //@todo: 엔진에서 uint64를 지원하지 않아서 int64로 변환하여 저장
                    int64 TempValue = static_cast<int64>(Variant.GetValue<uint64>());
                    Serializer.Serialize(*JsonName, TempValue);
                    break;
                }
            case ENhnVariantType::Name:
                {
                    FString TempValue = Variant.GetValue<FName>().ToString();
                    Serializer.Serialize(*JsonName, TempValue);
                    break;
                }
            default:
                {
#if UE_BUILD_DEBUG
                    checkf(false, TEXT("NHNJson: SerializeVariant: Unsupported FNhnVariant type (%d) for key: %s"),
                        static_cast<int32>(Variant.GetType()), *JsonName);
#endif
                }
        }
    }
}

FString SerializeVariantMapToJsonString(const FNhnVariantMap& VariantMap)
{
    const TSharedPtr<FJsonObject> JsonObject = MakeShared<FJsonObject>();

    for (const auto& Pair : VariantMap)
    {
        FString Key = Pair.Key.ToString();
        if (!Key.IsEmpty())
        {
            Key[0] = FChar::ToLower(Key[0]);
        }
        
        const FNhnVariant& Value = Pair.Value;

        switch (Value.GetType())
        {
            case ENhnVariantType::Empty:
                JsonObject->SetField(Key, MakeShareable(new FJsonValueNull()));
                break;
            case ENhnVariantType::Bool:
                JsonObject->SetBoolField(Key, Value.GetValue<bool>());
                break;
            case ENhnVariantType::Int32:
                JsonObject->SetNumberField(Key, Value.GetValue<int32>());
                break;
            case ENhnVariantType::Int64:
                JsonObject->SetNumberField(Key, static_cast<double>(Value.GetValue<int64>()));
                break;
            case ENhnVariantType::UInt32:
                JsonObject->SetNumberField(Key, Value.GetValue<uint32>());
                break;
            case ENhnVariantType::UInt64:
                JsonObject->SetNumberField(Key, static_cast<double>(Value.GetValue<uint64>()));
                break;
            case ENhnVariantType::Double:
                JsonObject->SetNumberField(Key, Value.GetValue<double>());
                break;
            case ENhnVariantType::Float:
                JsonObject->SetNumberField(Key, Value.GetValue<float>());
                break;
            case ENhnVariantType::Name:
                JsonObject->SetStringField(Key, Value.GetValue<FName>().ToString());
                break;
            case ENhnVariantType::String:
                JsonObject->SetStringField(Key, Value.GetValue<FString>());
                break;
            case ENhnVariantType::Array:
                {
                    FNhnVariantArray VariantArray = Value.GetValue<FNhnVariantArray>();
                    TArray<TSharedPtr<FJsonValue>> JsonArray;
                    
                    for (const FNhnVariant& ArrayElement : VariantArray)
                    {
                        switch (ArrayElement.GetType())
                        {
                            case ENhnVariantType::Empty:
                                JsonArray.Add(MakeShareable(new FJsonValueNull()));
                                break;
                            case ENhnVariantType::Bool:
                                JsonArray.Add(MakeShareable(new FJsonValueBoolean(ArrayElement.GetValue<bool>())));
                                break;
                            case ENhnVariantType::Int32:
                                JsonArray.Add(MakeShareable(new FJsonValueNumber(ArrayElement.GetValue<int32>())));
                                break;
                            case ENhnVariantType::Int64:
                                JsonArray.Add(MakeShareable(new FJsonValueNumber(static_cast<double>(ArrayElement.GetValue<int64>()))));
                                break;
                            case ENhnVariantType::UInt32:
                                JsonArray.Add(MakeShareable(new FJsonValueNumber(ArrayElement.GetValue<uint32>())));
                                break;
                            case ENhnVariantType::UInt64:
                                JsonArray.Add(MakeShareable(new FJsonValueNumber(static_cast<double>(ArrayElement.GetValue<uint64>()))));
                                break;
                            case ENhnVariantType::Double:
                                JsonArray.Add(MakeShareable(new FJsonValueNumber(ArrayElement.GetValue<double>())));
                                break;
                            case ENhnVariantType::Float:
                                JsonArray.Add(MakeShareable(new FJsonValueNumber(ArrayElement.GetValue<float>())));
                                break;
                            case ENhnVariantType::Name:
                                JsonArray.Add(MakeShareable(new FJsonValueString(ArrayElement.GetValue<FName>().ToString())));
                                break;
                            case ENhnVariantType::String:
                                JsonArray.Add(MakeShareable(new FJsonValueString(ArrayElement.GetValue<FString>())));
                                break;
                            case ENhnVariantType::Array:
                                JsonArray.Add(MakeShareable(new FJsonValueArray(TArray<TSharedPtr<FJsonValue>>())));
                                break;
                            case ENhnVariantType::Object:
                                {
                                    FNhnVariantMap NestedMap = ArrayElement.GetValue<FNhnVariantMap>();
                                    FString NestedJsonString = SerializeVariantMapToJsonString(NestedMap);
                                    TSharedPtr<FJsonObject> NestedJsonObject;
                                    TSharedRef<TJsonReader<>> Reader = TJsonReaderFactory<>::Create(NestedJsonString);
                                    if (FJsonSerializer::Deserialize(Reader, NestedJsonObject) && NestedJsonObject.IsValid())
                                    {
                                        JsonArray.Add(MakeShareable(new FJsonValueObject(NestedJsonObject)));
                                    }
                                    else
                                    {
                                        JsonArray.Add(MakeShareable(new FJsonValueObject(MakeShared<FJsonObject>())));
                                    }
                                }
                                break;
                            default:
                                JsonArray.Add(MakeShareable(new FJsonValueNull()));
                                break;
                        }
                    }
                    
                    JsonObject->SetArrayField(Key, JsonArray);
                }
                break;
            case ENhnVariantType::Object:
                {
                    FNhnVariantMap NestedMap = Value.GetValue<FNhnVariantMap>();
                    FString NestedJsonString = SerializeVariantMapToJsonString(NestedMap);
                    TSharedPtr<FJsonObject> NestedJsonObject;
                    TSharedRef<TJsonReader<>> Reader = TJsonReaderFactory<>::Create(NestedJsonString);
                    if (FJsonSerializer::Deserialize(Reader, NestedJsonObject) && NestedJsonObject.IsValid())
                    {
                        JsonObject->SetObjectField(Key, NestedJsonObject);
                    }
                    else
                    {
                        JsonObject->SetObjectField(Key, MakeShared<FJsonObject>());
                    }
                }
                break;
            default:
                JsonObject->SetField(Key, MakeShareable(new FJsonValueNull()));
                break;
        }
    }

    FString JsonString;
    const TSharedRef<TJsonWriter<>> Writer = TJsonWriterFactory<>::Create(&JsonString);

    if (FJsonSerializer::Serialize(JsonObject.ToSharedRef(), Writer))
    {
        JsonString.ReplaceInline(TEXT("\t"), TEXT(""));
        JsonString.ReplaceInline(TEXT("\n"), TEXT(""));
        return JsonString;
    }
    
    return JsonString;
}

FString SerializeOptionalVariantMapToJsonString(const TOptional<FNhnVariantMap>& OptionalVariantMap)
{
    if (OptionalVariantMap.IsSet())
    {
        return SerializeVariantMapToJsonString(OptionalVariantMap.GetValue());
    }
    
    return FString();
}

} // namespace Json
} // namespace NHN
