#include "GamebaseUtils.h"

#include "GamebaseDebugLogger.h"
#include "GamebaseSystemUtils.h"

#include <functional>
#include <string>

namespace GamebaseUtils
{
    struct FApi
    {
        using FCryptoCallback = std::function<void(int Code, const std::string& Output)>;

        using FDecryptLaunchingEncryptedKey = void(*)(const char* AppId, const char* Input, FCryptoCallback Callback);
        using FDecryptDefaultStabilityEncryptedKey = void(*)(const char* Input, FCryptoCallback Callback);

        FDecryptLaunchingEncryptedKey DecryptLaunchingEncryptedKey = nullptr;
        FDecryptDefaultStabilityEncryptedKey DecryptDefaultStabilityEncryptedKey = nullptr;
    } Api;

    void* DllHandle = nullptr;

    void* GetDllHandle()
    {
        void* Handle = nullptr;
#if PLATFORM_WINDOWS
        const FString Path = GamebaseSystemUtils::GetBinariesPath();
        GAMEBASE_LOG_GLOBAL_DEBUG("Import GamebaseUtils DLL (Path: %s)", *Path);
        FPlatformProcess::PushDllDirectory(*Path);
        const FString FilePath = FPaths::Combine(Path, TEXT("GamebaseUtils.dll"));
        if (FPaths::FileExists(FilePath))
        {
            Handle = FPlatformProcess::GetDllHandle(*FilePath);
            if (Handle == nullptr)
            {
                const int32 ErrorNum = FPlatformMisc::GetLastError();
                TCHAR ErrorMsg[1024];
                FPlatformMisc::GetSystemErrorMessage(ErrorMsg, 1024, ErrorNum);
                GAMEBASE_LOG_GLOBAL_ERROR("Failed to get GamebaseUtils.dll handle for %s: %s (%d))", *FilePath, ErrorMsg, ErrorNum);
            }
        }
        else
        {
            GAMEBASE_LOG_GLOBAL_ERROR("GamebaseUtils.dll file is missing. (Path: %s)", *FilePath);
        }
        FPlatformProcess::PopDllDirectory(*Path);
#endif
        return Handle;
    }

    namespace Crypto
    {
        constexpr int32 SuccessCode = 0;

        TOptional<FString> GetResultValue(const int Code, const std::string& Output)
        {
            TOptional<FString> Result;
            if (Code == SuccessCode)
            {
                Result = UTF8_TO_TCHAR(Output.c_str());
            }
            else
            {
                GAMEBASE_LOG_GLOBAL_WARNING("Failed to decrypt. (Code: %d)", Code);
            }

            return Result;
        }
    }
}


bool GamebaseUtils::ImportDll()
{
    DllHandle = GetDllHandle();
    if (DllHandle == nullptr)
    {
#if PLATFORM_WINDOWS
        GAMEBASE_LOG_GLOBAL_ERROR("Failed to import the GamebaseUtils.dll");
#endif
        return false;
    }

    void* LambdaDLLHandle = DllHandle;

    auto GetDllExport = [&LambdaDLLHandle](const TCHAR* FuncName, bool& bInSuccess) -> void*
    {
        if (bInSuccess)
        {
            void* FuncPtr = FPlatformProcess::GetDllExport(LambdaDLLHandle, FuncName);
            if (FuncPtr == nullptr)
            {
                bInSuccess = false;
                GAMEBASE_LOG_GLOBAL_WARNING("Failed to load GamebaseUtils.dll");
                FPlatformProcess::FreeDllHandle(LambdaDLLHandle);
                LambdaDLLHandle = nullptr;
            }
            return FuncPtr;
        }

        return nullptr;
    };

#define GAMEBASE_UTILS_DLL_EXPORT(Name) \
    Api.Name = static_cast<FApi::F##Name>(GetDllExport(TEXT(#Name), bSuccess));

    bool bSuccess = true;
#if PLATFORM_WINDOWS
    GAMEBASE_UTILS_DLL_EXPORT(DecryptLaunchingEncryptedKey);
    GAMEBASE_UTILS_DLL_EXPORT(DecryptDefaultStabilityEncryptedKey);
#endif

    check(bSuccess);
    return bSuccess;
}

bool GamebaseUtils::ShutdownDll()
{
    if (DllHandle)
    {
        FPlatformProcess::FreeDllHandle(DllHandle);

        DllHandle = nullptr;
        Api.DecryptLaunchingEncryptedKey = nullptr;
        Api.DecryptDefaultStabilityEncryptedKey = nullptr;
    }

    return true;
}

void GamebaseUtils::Crypto::DecryptLaunchingEncryptedKey(const FString& AppId, const FString& Input, const FCallback& Callback)
{
#if PLATFORM_WINDOWS
    Api.DecryptLaunchingEncryptedKey(
        TCHAR_TO_UTF8(*AppId),
        TCHAR_TO_UTF8(*Input),
        [&Callback](const int Code, const std::string& Output)
    {
        Callback(GetResultValue(Code, Output));
    });
#endif
}

void GamebaseUtils::Crypto::DecryptDefaultStabilityEncryptedKey(const FString& Input, const FCallback& Callback)
{
#if PLATFORM_WINDOWS
    Api.DecryptDefaultStabilityEncryptedKey(
        TCHAR_TO_UTF8(*Input),
        [&Callback](const int Code, const std::string& Output)
    {
        Callback(GetResultValue(Code, Output));
    });
#endif
}
