﻿#include "GamebaseImageNoticeViewer.h"

#include "GamebaseDebugLogger.h"
#include "GamebaseErrorCode.h"
#include "GamebaseImageNoticePreferences.h"
#include "GamebaseStandaloneSubsystem.h"
#include "GamebaseWebViewUtils.h"
#include "GenericPlatform/GenericPlatformHttp.h"
#include "WebView/GamebaseWebViewManager.h"
#include "WebView/GamebaseWebViewOpenParams.h"

class UGamebaseStandaloneSubsystem;

void FGamebaseImageNoticeViewer::Show()
{
    if (OwningGameInstance.IsValid() == false)
    {
        GAMEBASE_LOG_GLOBAL_WARNING("OwningGameInstance is invalid.");
        CloseCallback.ExecuteIfBound(&MakeShared<FGamebaseError, ESPMode::ThreadSafe>(GamebaseErrorCode::UNREAL_GET_ENGINE_INFO_ERROR,
            TEXT("OwningGameInstance is invalid."), GamebaseImageNotice::Domain).Get());
        return;
    }
    
    const auto WebViewCloseCallback = FGamebaseErrorDelegate::CreateLambda([this](const FGamebaseError* Error)
    {
        CloseCallback.ExecuteIfBound(Error);
    });
    
    const auto WebViewEventCallback = [this](const FString& Scheme, const FGamebaseError* Error)
    {
        using namespace GamebaseImageNotice;
        
        const TSharedPtr<GamebaseWebViewUtils::FSchemeInfo> SchemeInfo = GamebaseWebViewUtils::GetSchemeInfo(Scheme);
        if (!SchemeInfo.IsValid())
        {
            Close();
            return;
        }

        if (SchemeInfo->Scheme.Equals(Scheme::CefError))
        {
            Close();
            return;
        }

        if (SchemeInfo->Scheme.Equals(Scheme::Dismiss))
        {
            Close();
            return;
        }
        
        if (SchemeInfo->Scheme.Equals(Scheme::ImageNotice))
        {
            if (SchemeInfo->ParameterMap[Scheme::Action::Action].Equals(Scheme::Action::Click))
            {
                if (ResponseData.PageList.IsSet())
                {
                    const int64 PageId = FCString::Atoi64(*SchemeInfo->ParameterMap[Scheme::Action::Id]);
                    const auto& PageList = ResponseData.PageList.GetValue();
                    const auto CurrentPageInfo = PageList.FindByPredicate(
                        [PageId](const FGamebaseImageNoticePageInfo& PageInfo) { return PageInfo.ImageNoticeId == PageId; });
                    if (CurrentPageInfo != nullptr)
                    {
                        if (CurrentPageInfo->ClickScheme.IsSet())
                        {
                            const FString ClickScheme = FGenericPlatformHttp::UrlDecode(CurrentPageInfo->ClickScheme.GetValue());

                            if (ClickScheme.IsEmpty() == false)
                            {
                                if (CurrentPageInfo->ClickType.Equals(ClickType::OpenUrl))
                                {
                                    FPlatformProcess::LaunchURL(*ClickScheme, nullptr, nullptr);
                                }
                                else if (CurrentPageInfo->ClickType.Equals(ClickType::Custom))
                                {
                                    EventCallback.ExecuteIfBound(ClickScheme, nullptr);
                                }
                            }
                        }
                    }
                }
            }
            else if (SchemeInfo->ParameterMap[Scheme::Action::Action].Equals(Scheme::Action::NaverShowToday))
            {
                const int64 NextPopupTimeMs = ResponseData.NextPopupTimeMillis.Get(Info::InvalidTime);
                Preferences->AddNextPopupTime(NoticeId, NextPopupTimeMs);
                Close();
            }
        }
    };
    
    const FString Url = FString::Printf(TEXT("%s&orientation=landscape"), *ResponseData.Address);
    
    constexpr float ScreenWidth = 1920.f;
    constexpr float ScreenHeight = 1080.f;
    constexpr float StandardWebviewWidth = 600.0f;
    constexpr float StandardWebviewHeight = 450.0f;
    constexpr float NavigationBarHeight = 41.0f;
    
    FGamebaseWebViewOpenParams OpenParams;
    OpenParams.Url = Url;
    OpenParams.BackgroundColor = Configuration.BackgroundColor;

    FVector2D CurrentViewportSize;
    OwningGameInstance->GetGameViewportClient()->GetViewportSize(CurrentViewportSize);

    const float ScreenW = CurrentViewportSize.X / ScreenWidth;
    const float ScreenH = CurrentViewportSize.Y / ScreenHeight;

    const FVector2D ScreenScale = FVector2D(ScreenW, ScreenH);
    const float Scale = FMath::Min(ScreenScale.X, ScreenScale.Y);

    FVector2D ImageSize = FVector2D(StandardWebviewWidth * Scale, StandardWebviewHeight * Scale);
    ImageSize.Y += NavigationBarHeight;
    
    float Ratio = 1;
    if (ImageSize.X > StandardWebviewWidth)
    {
        Ratio = StandardWebviewWidth / ImageSize.X;
    }
    else if (ImageSize.Y > StandardWebviewHeight)
    {
        Ratio = StandardWebviewHeight / ImageSize.Y;
    }

    ImageSize *= Ratio;
    
    OpenParams.ViewportSize = ImageSize;
    
    OpenParams.CloseCallback= [WebViewCloseCallback](const FGamebaseErrorResult& Error)
    {
        WebViewCloseCallback.ExecuteIfBound(nullptr);
    };
    OpenParams.SchemeList = GamebaseImageNotice::Scheme::DefaultSchemeList;
    OpenParams.EventCallback = [WebViewEventCallback](const FGamebaseWebviewEventResult& EventResult)
    {
        WebViewEventCallback(EventResult.GetOkValue(), nullptr);
    };
    
    if (const UGamebaseStandaloneSubsystem* Subsystem = UGameInstance::GetSubsystem<UGamebaseStandaloneSubsystem>(OwningGameInstance.Get()))
    {
        Subsystem->GetWebViewManager()->Open(OpenParams);
    }
}

void FGamebaseImageNoticeViewer::Close()
{
    if (OwningGameInstance.IsValid() == false)
    {
        GAMEBASE_LOG_GLOBAL_WARNING("OwningGameInstance is invalid.");
        return;
    }
    
    if (const UGamebaseStandaloneSubsystem* Subsystem = UGameInstance::GetSubsystem<UGamebaseStandaloneSubsystem>(OwningGameInstance.Get()))
    {
        Subsystem->GetWebViewManager()->Close();
    }
}