#include "GamebaseSystemPopupHelper.h"

#include "GamebaseDebugLogger.h"
#include "GamebaseLaunchingStatus.h"
#include "GamebasePopup.h"
#include "GamebaseStandaloneSystemInfo.h"
#include "GamebaseSystemUtils.h"
#include "GamebaseTimeUtils.h"
#include "GenericPlatformHttp.h"
#include "IGamebaseDisplayLanguage.h"
#include "Kismet/GameplayStatics.h"
#include "Kismet/KismetSystemLibrary.h"
#include "Types/GamebaseLaunchingTypes.h"
#include "Utils/GamebaseHelper.h"

#define LOCTEXT_NAMESPACE "GamebaseStandalone"

using FGamebasePopupButton = GamebasePopup::FGamebasePopupButton;

void GamebaseSystemPopup::ShowLaunchingNotice(const IGamebaseDisplayLanguage& Language, const FGamebaseLaunchingInfo& LaunchingInfo, const FCloseFunction& Callback)
{
    if (LaunchingInfo.Launching.Notice.IsSet() == false)
    {
        Callback();
        return;
    }
    
    const auto& NoticeInfo = LaunchingInfo.Launching.Notice.GetValue();

    auto ShowDetailBtn = MakeShared<FGamebasePopupButton>(Language.GetLocalizedString(TEXT("common_show_detail_button")), [=]
        {
            FPlatformProcess::LaunchURL(*NoticeInfo.Url, nullptr, nullptr);
            FGenericPlatformMisc::RequestExit(false);
        });
    
    auto CloseBtn = MakeShared<FGamebasePopupButton>(Language.GetLocalizedString(TEXT("common_close_button")), Callback);
    
    ShowPopup(NoticeInfo.Title, NoticeInfo.Message, ShowDetailBtn, CloseBtn);
}

void GamebaseSystemPopup::ShowUnknownErrorPopup(const IGamebaseDisplayLanguage& Language, const uint32 StatusCode, const FCloseFunction& Callback)
{
    FString Title = Language.GetLocalizedString(TEXT("common_error_label"));
    FString Message = FString::Printf(TEXT("LaunchingUnknownErrorPopup_Message\n%s(%d)"), *Language.GetLocalizedString(TEXT("common_unknown_error_message")), StatusCode);

    auto CloseBtn = MakeShared<FGamebasePopupButton>(Language.GetLocalizedString(TEXT("common_close_button")), Callback);
        
    ShowPopup(Title, Message, CloseBtn);

    Callback();
}

void GamebaseSystemPopup::ShowLaunchingStatus(const IGamebaseInternalData* InternalData, const IGamebaseDisplayLanguage& Language, const FGamebaseLaunchingInfo& LaunchingInfo, const FCloseFunction& Callback)
{
    const auto& LaunchingStatus = LaunchingInfo.Launching.Status;
    switch (LaunchingStatus.Code)
    {
        case GamebaseLaunchingStatus::IN_SERVICE:
        case GamebaseLaunchingStatus::IN_TEST:
        case GamebaseLaunchingStatus::IN_REVIEW:
            {
                Callback();
                break;
            }
        case GamebaseLaunchingStatus::IN_SERVICE_BY_QA_WHITE_LIST:
        case GamebaseLaunchingStatus::IN_BETA:
            {
                GAMEBASE_LOG_DEBUG("Status : %s", *LaunchingStatus.Message);
                Callback();
                break;
            }
        case GamebaseLaunchingStatus::RECOMMEND_UPDATE:
            {
                ShowUpdateRecommend(Language, LaunchingInfo, Callback);
                break;
            }
        case GamebaseLaunchingStatus::REQUIRE_UPDATE:
            {
                ShowUpdateRequire(Language, LaunchingInfo, Callback);
                break;
            }
        case GamebaseLaunchingStatus::BLOCKED_USER:
            {
                ShowBlockUser(Language, LaunchingInfo, Callback);
                break;
            }
        case GamebaseLaunchingStatus::TERMINATED_SERVICE:
            {
                ShowServiceClosed(Language, LaunchingInfo, Callback);
                break;
            }
        case GamebaseLaunchingStatus::INSPECTING_SERVICE:
        case GamebaseLaunchingStatus::INSPECTING_ALL_SERVICES:
            {
                ShowMaintenance(Language, LaunchingInfo, Callback);
                break;
            }
        case GamebaseLaunchingStatus::INTERNAL_SERVER_ERROR:
            {
                ShowUnknownErrorPopup(Language, LaunchingStatus.Code, Callback);
                break;
            }
        default:
            {
                if (GamebaseHelper::IsLaunchingPlayable(LaunchingStatus.Code))
                {
                    GAMEBASE_LOG_WARNING("Playable but undefined (status: %d)", LaunchingStatus.Code);
                    Callback();
                }
                else
                {
                    ShowUnknownErrorPopup(Language, LaunchingStatus.Code, Callback);
                }
                break;
            }
    }
}

void GamebaseSystemPopup::ShowUpdateRecommend(const IGamebaseDisplayLanguage& Language, const FGamebaseLaunchingInfo& LaunchingInfo, const FCloseFunction& Callback)
{
    const auto& LaunchingStatus = LaunchingInfo.Launching.Status;
    
    const FString Title = Language.GetLocalizedString(TEXT("launching_update_recommended_title"));
    const FString Message = LaunchingStatus.Message;

    auto UpdateNowBtn = MakeShared<FGamebasePopupButton>(Language.GetLocalizedString(TEXT("launching_update_now_label")), [=]
        {
            FPlatformProcess::LaunchURL(*LaunchingInfo.Launching.App.Install.Url, nullptr, nullptr);
             FGenericPlatformMisc::RequestExit(false);
        });
    
    auto UpdateLaterBtn = MakeShared<FGamebasePopupButton>(Language.GetLocalizedString(TEXT("launching_update_later_label")), Callback);

    ShowPopup(Title, Message, UpdateNowBtn, UpdateLaterBtn);
}

void GamebaseSystemPopup::ShowUpdateRequire(const IGamebaseDisplayLanguage& Language, const FGamebaseLaunchingInfo& LaunchingInfo, const FCloseFunction& Callback)
{
    const auto& LaunchingStatus = LaunchingInfo.Launching.Status;
    
    const FString Title = Language.GetLocalizedString(TEXT("launching_update_required_title"));
    const FString Message = LaunchingStatus.Message;

    auto UpdateNow = MakeShared<FGamebasePopupButton>(Language.GetLocalizedString(TEXT("launching_update_now_label")), [=]
    {
        FPlatformProcess::LaunchURL(*LaunchingInfo.Launching.App.Install.Url, nullptr, nullptr);
        FGenericPlatformMisc::RequestExit(false);
    });

    if(LaunchingInfo.Launching.App.Install.DetailUrl.IsEmpty() == false)
    {
        auto ShowDetailBtn = MakeShared<FGamebasePopupButton>(Language.GetLocalizedString(TEXT("common_show_detail_button")), [=]
        {
            FPlatformProcess::LaunchURL(*LaunchingInfo.Launching.App.Install.DetailUrl, nullptr, nullptr);
            Callback();
        });
        
        ShowPopup(Title, Message, ShowDetailBtn, UpdateNow);
    }
    else
    {
        ShowPopup(Title, Message, UpdateNow);
    }
}

void GamebaseSystemPopup::ShowBlockUser(const IGamebaseDisplayLanguage& Language, const FGamebaseLaunchingInfo& LaunchingInfo, const FCloseFunction& Callback)
{
    const auto& LaunchingStatus = LaunchingInfo.Launching.Status;
    
    const FString Title = Language.GetLocalizedString(TEXT("launching_blocked_user_title"));
    const FString Message = LaunchingStatus.Message;
    
    auto CloseBtn = MakeShared<FGamebasePopupButton>(Language.GetLocalizedString(TEXT("common_close_button")), Callback);
    ShowPopup(Title, Message, CloseBtn);
}

void GamebaseSystemPopup::ShowServiceClosed(const IGamebaseDisplayLanguage& Language, const FGamebaseLaunchingInfo& LaunchingInfo, const FCloseFunction& Callback)
{
    const auto& LaunchingStatus = LaunchingInfo.Launching.Status;
    
    const FString Title = Language.GetLocalizedString(TEXT("launching_service_closed_title"));
    const FString Message = LaunchingStatus.Message;
    
    auto CloseBtn = MakeShared<FGamebasePopupButton>(Language.GetLocalizedString(TEXT("common_close_button")), Callback);
    ShowPopup(Title, Message, CloseBtn);
}

void GamebaseSystemPopup::ShowMaintenance(const IGamebaseDisplayLanguage& Language, const FGamebaseLaunchingInfo& LaunchingInfo, const FCloseFunction& Callback)
{
    if (LaunchingInfo.Launching.Maintenance.IsSet() == false)
    {
        Callback();
        return;
    }
    
    const FString Title = Language.GetLocalizedString(TEXT("launching_maintenance_title"));
    FString Message = FGenericPlatformHttp::UrlDecode(LaunchingInfo.Launching.Maintenance->Message);

    const auto& MaintenanceInfo = LaunchingInfo.Launching.Maintenance.GetValue();
    if (MaintenanceInfo.bHideDate == false)
    {
        FString MaintenanceDate = GamebaseTimeUtils::PeriadText(MaintenanceInfo.LocalBeginDate, MaintenanceInfo.LocalEndDate);
        Message = FString::Printf(TEXT("%s\n%s"), *Message, *MaintenanceDate);
    }
    else
    {
        Message = FString::Printf(TEXT("%s"), *Message);
    }
    
    auto CloseBtn = MakeShared<FGamebasePopupButton>(Language.GetLocalizedString(TEXT("common_close_button")), Callback);
    
    if (MaintenanceInfo.Url.IsEmpty() == false)
    {
        auto ShowDetailBtn = MakeShared<FGamebasePopupButton>(Language.GetLocalizedString(TEXT("common_show_detail_button")), [=]
        {
            FPlatformProcess::LaunchURL(*MaintenanceInfo.Url, nullptr, nullptr);
            Callback();
        });
        
        ShowPopup(Title, Message, ShowDetailBtn, CloseBtn);
    }
    else
    {
        ShowPopup(Title, Message, CloseBtn);
    }
}

void GamebaseSystemPopup::ShowUnregisteredClient(const IGamebaseDisplayLanguage& Language, const FGamebaseLaunchingUpdateInfo& UpdateInfo, const FCloseFunction& Callback)
{
    const FString Title = Language.GetLocalizedString(TEXT("launching_unregistered_client_version"));
    const FString Message = UpdateInfo.Message;

    auto GoStoreBtn = MakeShared<FGamebasePopupButton>(Language.GetLocalizedString(TEXT("launching_goto_store")), [=]
    {
        FPlatformProcess::LaunchURL(*UpdateInfo.InstallUrl, nullptr, nullptr);        
        
        FGenericPlatformMisc::RequestExit(false);
    });

    ShowPopup(Title, Message, GoStoreBtn);
}

void GamebaseSystemPopup::ShowBan(const IGamebaseDisplayLanguage& Language, const FGamebaseBanInfo& BanInfo, const FCloseFunction& Callback)
{
    const FString UserMessage = Language.GetLocalizedString(TEXT("ban_user_message"));

    const FString UserIdLabel = FString::Printf(TEXT("%s : %s"), *Language.GetLocalizedString(TEXT("ban_user_id_label")), *BanInfo.UserId);
    const FString ReasonLabel = FString::Printf(TEXT("%s : %s"), *Language.GetLocalizedString(TEXT("ban_reason_label")), *FGenericPlatformHttp::UrlDecode(*BanInfo.Message));

    FString BanDate;
    if (true == BanInfo.BanType.Equals("TEMPORARY", ESearchCase::IgnoreCase))
    {
        BanDate = GamebaseTimeUtils::PeriadText(BanInfo.BeginDate, BanInfo.EndDate);
    }
    else
    {
        BanDate = Language.GetLocalizedString(TEXT("ban_permanent_label"));
    }
    const FString DateLabel = FString::Printf(TEXT("%s : %s"), *Language.GetLocalizedString(TEXT("ban_date_label")), *BanDate);
    
    const FString CsGuideMessage = Language.GetLocalizedString(TEXT("ban_detailpage_cs_guide_message"));

    const FString Title = Language.GetLocalizedString(TEXT("ban_title"));

    FString Message = FString::Printf(TEXT("%s\n\n%s\n%s\n\n%s\n\n%s"), *UserMessage, *UserIdLabel, *ReasonLabel, *DateLabel, *CsGuideMessage);

    auto CloseBtn = MakeShared<FGamebasePopupButton>(Language.GetLocalizedString(TEXT("common_close_button")), Callback);
    if (BanInfo.CsUrl.IsEmpty() == false)
    {
        const FString ShowCsMessage = Language.GetLocalizedString(TEXT("ban_show_cs_message"));
        Message = FString::Printf(TEXT("%s\n\n%s"), *Message, *ShowCsMessage);
        
        auto CsBtn = MakeShared<FGamebasePopupButton>(Language.GetLocalizedString(TEXT("ban_cs_label")), [=]
        {
            FPlatformProcess::LaunchURL(*BanInfo.CsUrl, nullptr, nullptr);
            Callback();
        });
        
        ShowPopup(Title, Message, CsBtn, CloseBtn);    
    }
    else
    {
        ShowPopup(Title, Message, CloseBtn);
    }
}

void GamebaseSystemPopup::ShowHeartbeatBan(const IGamebaseDisplayLanguage& Language, const FCloseFunction& Callback)
{
    const FString Title = Language.GetLocalizedString(TEXT("ban_title"));
    const FString Message = Language.GetLocalizedString(TEXT("ban_user_message"));

    auto CloseBtn = MakeShared<FGamebasePopupButton>(Language.GetLocalizedString(TEXT("common_close_button")), Callback);
    ShowPopup(Title, Message, CloseBtn);
}

void GamebaseSystemPopup::ShowServerPushKickout(const IGamebaseInternalData* InternalData, const IGamebaseDisplayLanguage& Language, const FGamebaseServerPushPopupInfo& Popup, const FCloseFunction& Callback)
{
    FString LanguageCode = Language.GetDisplayLanguageCode();
    if (LanguageCode.IsEmpty() || Popup.Messages.Contains(LanguageCode) == false)
    {
        LanguageCode = GamebaseSystemUtils::GetDeviceLanguage();
        if (LanguageCode.IsEmpty() || Popup.Messages.Contains(LanguageCode) == false)
        {
            LanguageCode = Popup.DefaultLanguage;
            if (LanguageCode.IsEmpty() || Popup.Messages.Contains(LanguageCode) == false)
            {
                GAMEBASE_LOG_WARNING("No messages were found for this language.\npopup: %s", *Popup.ToJson());
                Callback();
                return;
            }
        }
    }

    const auto& MessageInfo = Popup.Messages[LanguageCode];

    const FString Title = MessageInfo.Title;
    const FString Message = MessageInfo.Message;

    auto OKBtn = MakeShared<FGamebasePopupButton>(Language.GetLocalizedString(TEXT("common_ok_button")), Callback);
    ShowPopup(Title, Message, OKBtn);
}
