#include "GamebaseStandaloneWebView.h"

#include "GamebaseErrorUtil.h"
#include "GamebaseInternalData.h"
#include "GamebaseInternalReport.h"
#include "GamebaseProvider.h"
#include "GamebaseStandaloneEventHandler.h"
#include "GamebaseStandaloneSubsystem.h"
#include "WebView/GamebaseWebViewConstants.h"
#include "WebView/GamebaseWebViewManager.h"
#include "WebView/GamebaseWebViewOpenParams.h"

FGamebaseStandaloneWebView::FGamebaseStandaloneWebView(
    const FGamebaseWebSocketPtr& WebSocket,
    const FGamebaseInternalDataPtr& InternalData,
    const TSharedPtr<FGamebaseStandaloneEventHandler>& EventHandler)
    : FGamebaseInternalModule(WebSocket, InternalData)
    , EventHandler(EventHandler)
{
}

void FGamebaseStandaloneWebView::OpenWebBrowser(const FString& Url)
{
    FPlatformProcess::LaunchURL(*Url, nullptr, nullptr);
}

void FGamebaseStandaloneWebView::ShowWebView(
    const FString& Url,
    const FGamebaseWebViewConfiguration& Configuration,
    const FGamebaseErrorDelegate& CloseCallback,
    const TArray<FString>& SchemeList,
    const FGamebaseSchemeEventDelegate& SchemeEvent)
{
    if (InternalData.IsValid() == false)
    {
        const auto Error = GamebaseErrorUtil::NewError(GamebaseWebView::Domain, GamebaseErrorCode::NOT_INITIALIZED);
        CloseCallback.ExecuteIfBound(Error.Get());
        
        GamebaseInternalReport::WebView::Open(*InternalData, Url, &Configuration, Error.Get());
        return;
    }
    
    if (Url.IsEmpty())
    {
        const auto Error = GamebaseErrorUtil::NewError(GamebaseWebView::Domain, GamebaseErrorCode::WEBVIEW_INVALID_URL);
        CloseCallback.ExecuteIfBound(Error.Get());
            
        GamebaseInternalReport::WebView::Open(*InternalData, Url, &Configuration, Error.Get());
        return;
    }

    OnEventWebView(GamebaseWebViewEventType::Opened);

    FGamebaseWebViewOpenParams OpenParams;
    OpenParams.Url = Url;

    if (Configuration.bIsNavigationBarVisible)
    {
        FGamebaseWebViewOpenParams::FNavigation Navigation;
        Navigation.Title = Configuration.Title;
        Navigation.TitleColor = Configuration.NavigationTitleColor;
        Navigation.IconTintColor = Configuration.NavigationIconTintColor.Get(FColor::White);
        Navigation.BarColor = Configuration.NavigationColor;
        Navigation.BarHeight = Configuration.NavigationBarHeight;
        Navigation.bIsBackButton = Configuration.bIsBackButtonVisible;
        
        OpenParams.Navigation = Navigation;
    }
    
    OpenParams.SchemeList = SchemeList;
    OpenParams.EventCallback = [SchemeEvent](const FGamebaseWebviewEventResult& EventResult)
    {
        SchemeEvent.ExecuteIfBound(EventResult.GetOkOrDefault(TEXT("")), nullptr);
    };
    OpenParams.CloseCallback= [=](const FGamebaseErrorResult& Error)
    {
        OnEventWebView(GamebaseWebViewEventType::Closed);
        CloseCallback.ExecuteIfBound(nullptr);
    };
    
    if (const UGamebaseStandaloneSubsystem* Subsystem = UGameInstance::GetSubsystem<UGamebaseStandaloneSubsystem>(InternalData->GetGameInstance().Get()))
    {
        Subsystem->GetWebViewManager()->Open(OpenParams);
    }
}

void FGamebaseStandaloneWebView::CloseWebView()
{
    if (const UGamebaseStandaloneSubsystem* Subsystem = UGameInstance::GetSubsystem<UGamebaseStandaloneSubsystem>(InternalData->GetGameInstance().Get()))
    {
        Subsystem->GetWebViewManager()->Close();
    }
}

void FGamebaseStandaloneWebView::OnEventWebView(const int32 EventType)
{
    FGamebaseEventObserverData EventData;
    EventData.Code = EventType;
                
    FGamebaseEventMessage EventMessage;
    EventMessage.Category = GamebaseEventCategory::ObserverWebView;
    EventMessage.Data = EventData.ToJson();

    EventHandler->Notify(EventMessage);
}
