#include "GamebaseStandaloneServerPush.h"

#include "GamebaseDebugLogger.h"
#include "GamebaseStandaloneEventHandler.h"
#include "GamebaseWebSocket.h"
#include "Helper/GamebaseSystemPopupHelper.h"
#include "Server/GamebaseServerPushResponse.h"
#include "Server/GamebaseServerPushType.h"


FString GetServerPushTypeToEventCategory(const FString& ServerPushType)
{
    if (ServerPushType.Equals(GamebaseServerPushType::AppKickout))
    {
        return GamebaseEventCategory::ServerPushAppKickOut;
    }

    if (ServerPushType.Equals(GamebaseServerPushType::TransferKickout))
    {
        return GamebaseEventCategory::ServerPushTransferKickout;
    }

    return FString();
}

FGamebaseStandaloneServerPush::FGamebaseStandaloneServerPush(const FGamebaseWebSocketPtr& WebSocket, const FGamebaseInternalDataPtr& InternalData, const TSharedPtr<FGamebaseStandaloneEventHandler>& EventHandler)
    : FGamebaseInternalModule(WebSocket, InternalData)
    , EventHandler(EventHandler)
{
}

void FGamebaseStandaloneServerPush::OnReceiveServerPush(const FGamebaseWebSocketResponse& ServerPushResponse)
{
    const auto& ServerPush = ServerPushResponse.Header.ServerPush.GetValue();
    
    if (ServerPush.Type.Equals(GamebaseServerPushType::AppKickout) && InternalData->IsLogin() == false)
    {
        return;
    }

    if (KickoutStampList.Contains(ServerPush.Stamp))
    {
        GAMEBASE_LOG_DEBUG("KickoutMessage is already arrived and is on the process. This event is ignored until all processes will be finished.");
        return;
    }
    
    KickoutStampList.Add(ServerPush.Stamp);
    
    if (ServerPush.StopHeartbeat)
    {
        OnStopScheduler.Broadcast();
    }

    if (ServerPush.Logout)
    {
        OnCallLogout.ExecuteIfBound();
    }

    if (ServerPush.Disconnect)
    {
        WebSocket->Disconnect();
    }
    
    GamebaseServerPush::FServerPushPopup ServerPushPopup;
    ServerPushPopup.FromJson(ServerPushResponse.ServerPushPopup);
    
    FGamebaseEventServerPushData::FPopup Popup;
    Popup.DefaultLanguage = ServerPushPopup.DefaultLanguage;
    for (const auto& Message : ServerPushPopup.Messages)
    {
        FGamebaseEventServerPushData::FPopup::FMessage MessageData;
        MessageData.Message = Message.Value.Message;
        MessageData.Title = Message.Value.Title;
        
        Popup.Messages.FindOrAdd(Message.Key, MessageData);
    }

    FGamebaseEventServerPushData EventData;
    EventData.Extras = ServerPushResponse.Result;
    EventData.Popup = Popup;

    FGamebaseEventMessage EventMessage;
    EventMessage.Data = EventData.ToJson();
    
    if (ServerPush.Type.Equals(GamebaseServerPushType::AppKickout))
    {
        EventMessage.Category = GamebaseEventCategory::ServerPushAppKickOutMessageReceived;
        EventHandler->Notify(EventMessage);
    }
    
    EventMessage.Category = GetServerPushTypeToEventCategory(ServerPush.Type);

    if (ServerPush.Popup)
    {
        GamebaseSystemPopup::ShowServerPushKickout(
            InternalData.Get(),
            *InternalData->GetDisplayLanguage(),
            ServerPushPopup,
            [=]
            {
                EventHandler->Notify(EventMessage);
            });
    }
    else
    {
        EventHandler->Notify(EventMessage);
    }
}
