#include "GamebaseStandaloneLaunching.h"

#include "GamebaseDebugLogger.h"
#include "GamebaseErrorExtras.h"
#include "GamebaseErrorUtil.h"
#include "GamebaseSaveInfo.h"
#include "GamebaseStandaloneEventHandler.h"
#include "GamebaseSystemUtils.h"
#include "GamebaseWebSocket.h"
#include "IGamebaseDisplayLanguage.h"
#include "Helper/GamebaseSystemPopupHelper.h"
#include "Launching/GamebaseLaunchingConstants.h"
#include "Launching/GamebaseLaunchingRequest.h"
#include "Launching/GamebaseLaunchingResponse.h"
#include "Scheduler/GamebaseScheduler.h"
#include "Server/GamebaseServerInfo.h"
#include "Terms/GamebaseTermsDefines.h"
#include "Utils/GamebaseUtils.h"

class FGamebaseLaunchingScheduler final : public FGamebaseScheduler
{
public:
    explicit FGamebaseLaunchingScheduler(FGamebaseStandaloneLaunching& InLaunching)
        : FGamebaseScheduler(GamebaseLaunching::Interval) 
        , Launching(InLaunching)
    {
    }
    virtual ~FGamebaseLaunchingScheduler() override = default;

protected:
    virtual void OnUpdateSchedule() override
    {
        Launching.RequestLaunchingStatus([=](const FGamebaseLaunchingStatusResult& Result)
        {
        });
    }

private:
    FGamebaseStandaloneLaunching& Launching;
};


FGamebaseStandaloneLaunching::FGamebaseStandaloneLaunching(const FGamebaseWebSocketPtr& WebSocket, const FGamebaseInternalDataPtr& InternalData, const TSharedPtr<FGamebaseStandaloneEventHandler>& EventHandler)
    : FGamebaseInternalModule(WebSocket, InternalData)
    , LaunchingScheduler(MakeShared<FGamebaseLaunchingScheduler>(*this))
    , LastUpdateTime(0)
    , EventHandler(EventHandler)
{
}

FGamebaseStandaloneLaunching::~FGamebaseStandaloneLaunching()
{
}

const FGamebaseLaunchingInfoPtr FGamebaseStandaloneLaunching::GetLaunchingInformations() const
{
    if (InternalData.IsValid() == false)
    {
        GAMEBASE_LOG_WARNING("The Gamebase SDK must be initialized");
        return nullptr;
    }
    
    return InternalData->GetLaunchingInfoPtr();
}

int32 FGamebaseStandaloneLaunching::GetLaunchingStatus() const
{
    if (InternalData.IsValid() == false)
    {
        GAMEBASE_LOG_WARNING("The Gamebase SDK must be initialized");
        return 0;
    }

    return InternalData->GetLaunchingInfo()->Launching.Status.Code;
}

void FGamebaseStandaloneLaunching::RequestLaunchingInfo(const FLaunchingInfoFunction& Callback)
{
    using namespace GamebaseServerInfo;

    GamebaseLaunching::FParameters Parameters;
    Parameters.AppId = InternalData->GetAppId();
    Parameters.ClientVersion = InternalData->GetAppVersion();
    Parameters.StoreCode = InternalData->GetStoreCode();
    Parameters.SDKVersion = GamebaseSystemUtils::GetPluginVersion();
    Parameters.UUID = InternalData->GetUuid();
    Parameters.DeviceKey = GamebaseSystemUtils::GetDeviceKey();
    Parameters.OSCode = GamebaseSystemUtils::GetPlatform();
    Parameters.OSVersion = GamebaseSystemUtils::GetOSVersion();
    Parameters.DeviceModel = GamebaseSystemUtils::GetDeviceModel();
    Parameters.DisplayLanguage = InternalData->GetDisplayLanguageCode();
    Parameters.DeviceLanguage = GamebaseSystemUtils::GetDeviceLanguage();
    Parameters.DeviceCountryCode = GamebaseSystemUtils::GetCountryCode();
    Parameters.UsimCountryCode = GamebaseSystemUtils::GetSimCode();
    Parameters.UserId = IsGamebaseLogin() ? InternalData->GetUserId() : TEXT("0");
    Parameters.LastCheckedNoticeTime = 0;
    
    TOptional<FString> SaveTermsVersion = GamebaseSaveInfo::GetStoreValue(InternalData->GetAppId(), GamebaseTerms::Preference::Version);
    if (SaveTermsVersion.IsSet())
    {
        Parameters.TermsVersion = SaveTermsVersion.GetValue();
    }
    
    WebSocket->Request(Launching::ProductId, Launching::Id::GetLaunching, ApiVersion, Parameters, [=](const FGamebaseWebSocketResponseResult& Response)
    {
        if (Response.IsError())
        {
            Callback(FGamebaseLaunchingInfoResult(Response.GetErrorValue()));
            return;
        }

        const auto& ResponseSuccessData = Response.GetOkValue();
        
        if (ResponseSuccessData.Header.bIsSuccessful == false)
        {
            const FGamebaseErrorPtr Error = GamebaseErrorUtil::NewError(ResponseSuccessData, GamebaseLaunching::Domain);

            GamebaseLaunching::FErrorExtras ErrorExtras;
            ErrorExtras.FromJson(ResponseSuccessData.ErrorExtras);

            if (ErrorExtras.UpdateInfo.IsEmpty() == false)
            {
                Error->AddExtra(GamebaseErrorExtras::UpdateInfo, ErrorExtras.UpdateInfo);
            }

            if (ErrorExtras.Language.IsEmpty() == false)
            {
                GamebaseLaunching::FLanguageInfo Language;
                Language.FromJson(ErrorExtras.Language);

                InternalData->GetDisplayLanguage()->UpdateLaunchingLanguageCode(Language.DeviceLanguage, Language.DefaultLanguage);
            }

            LaunchingError(*Error, [=]
                {
                    Callback(FGamebaseLaunchingInfoResult(*Error));
                });
            return;
        }
        
        const FGamebaseLaunchingInfoPtr LaunchingInfo = MakeShared<FGamebaseLaunchingInfo, ESPMode::ThreadSafe>();
        LaunchingInfo->FromJson(ResponseSuccessData.OriginData);
        
        auto DecryptFunction = [&](FString& Value)
        {
            if (Value.IsEmpty())
            {
                return;
            }
            
            GamebaseUtils::Crypto::DecryptLaunchingEncryptedKey(
                InternalData->GetAppId(),
                Value,
                [&Value](const TOptional<FString>& Output)
                {
                    Value = Output.Get(FString());
                });
        };

        DecryptFunction(LaunchingInfo->Launching.TcgbClient->Stability.AppKey);
        DecryptFunction(LaunchingInfo->Launching.TcgbClient->ForceRemoteSettings->Log.AppKeyIndicator);
        DecryptFunction(LaunchingInfo->Launching.TcgbClient->ForceRemoteSettings->Log.AppKeyLog);
        DecryptFunction(LaunchingInfo->TcProduct.Gamebase.AppKey);
        DecryptFunction(LaunchingInfo->TcProduct.TcLaunching.AppKey);
        DecryptFunction(LaunchingInfo->TcProduct.Iap.AppKey);
        DecryptFunction(LaunchingInfo->TcProduct.Push.AppKey);
        for (auto& IdP : LaunchingInfo->Launching.App.Idp)
        {
            if (IdP.Value.ClientSecret.IsSet())
            {
                DecryptFunction(IdP.Value.ClientSecret.GetValue());
            }
            
            if (IdP.Value.Channels.Num() > 0)
            {
                for (auto& Channel : IdP.Value.Channels)
                {
                    DecryptFunction(Channel.ClientSecret);
                }
            }
        }
        
        ShowNoticePopup(*LaunchingInfo,[=]
        {
            ShowStatusPopup(*LaunchingInfo, [=]
            {
                InternalData->SetLaunchingInfo(LaunchingInfo);
                LaunchingScheduler->StartScheduler();
                LastUpdateTime = FPlatformTime::Seconds();
                
                OnUpdateLaunchingInfo.Broadcast(*LaunchingInfo);
                Callback(FGamebaseLaunchingInfoResult(*LaunchingInfo));
            });
        });
    });
}

void FGamebaseStandaloneLaunching::RequestGetLaunchingStatusIfNeedUpdate(const FLaunchingStatusFunction& Callback)
{
    if (IsNeedUpdate() == false)
    {
        Callback(FGamebaseLaunchingStatusResult(GetLaunchingStatus()));
        return;
    }
    
    RequestLaunchingStatus([=](const FGamebaseLaunchingStatusResult& Result)
    {
        Callback(Result);
    });
}

void FGamebaseStandaloneLaunching::RequestLaunchingStatus(const FLaunchingStatusFunction& Callback)
{
    using namespace GamebaseServerInfo;
    
    GamebaseLaunching::FParameters Parameters;
    Parameters.AppId = InternalData->GetAppId();
    Parameters.ClientVersion = InternalData->GetAppVersion();
    Parameters.StoreCode = InternalData->GetStoreCode();
    Parameters.SDKVersion = GamebaseSystemUtils::GetPluginVersion();
    Parameters.UUID = InternalData->GetUuid();
    Parameters.DeviceKey = GamebaseSystemUtils::GetDeviceKey();
    Parameters.OSCode = GamebaseSystemUtils::GetPlatform();
    Parameters.OSVersion = GamebaseSystemUtils::GetOSVersion();
    Parameters.DeviceModel = GamebaseSystemUtils::GetDeviceModel();
    Parameters.DisplayLanguage = InternalData->GetDisplayLanguageCode();
    Parameters.DeviceLanguage = GamebaseSystemUtils::GetDeviceLanguage();
    Parameters.DeviceCountryCode = GamebaseSystemUtils::GetCountryCode();
    Parameters.UsimCountryCode = GamebaseSystemUtils::GetSimCode();
    Parameters.UserId = IsGamebaseLogin() ? InternalData->GetUserId() : TEXT("0");
    Parameters.LastCheckedNoticeTime = 0;
    
    TOptional<FString> SaveTermsVersion = GamebaseSaveInfo::GetStoreValue(InternalData->GetAppId(), GamebaseTerms::Preference::Version);
    if (SaveTermsVersion.IsSet())
    {
        Parameters.TermsVersion = SaveTermsVersion.GetValue();
    }
    
    WebSocket->Request(Launching::ProductId, Launching::Id::GetLaunchingStatus, ApiVersion, Parameters, [=](const FGamebaseWebSocketResponseResult& Response)
    {
        if (Response.IsError())
        {
            Callback(FGamebaseLaunchingStatusResult(Response.GetErrorValue()));
            return;
        }
        
        const auto& ResponseSuccessData = Response.GetOkValue();
        if (ResponseSuccessData.Header.bIsSuccessful == false)
        {
            const FGamebaseErrorPtr RecvError = GamebaseErrorUtil::NewError(ResponseSuccessData, GamebaseLaunching::Domain);
            Callback(FGamebaseLaunchingStatusResult(*RecvError));
            return;
        }

        FGamebaseLaunchingInfo NewLaunchingInfo;
        NewLaunchingInfo.FromJson(ResponseSuccessData.OriginData);
        
        const auto SavedLaunchingInfo = InternalData->GetLaunchingInfo();
        if (NewLaunchingInfo.Launching.Status.Code == SavedLaunchingInfo->Launching.Status.Code)
        {
            Callback(FGamebaseLaunchingStatusResult(NewLaunchingInfo.Launching.Status.Code));
            return;
        }

        RequestLaunchingInfo([=](const FGamebaseLaunchingInfoResult&)
        {
            //TODO: 추가 처리 있을지 확인
            FGamebaseEventObserverData EventData;
        
            EventData.Code = NewLaunchingInfo.Launching.Status.Code;
            EventData.Message = NewLaunchingInfo.Launching.Status.Message;
        
            FGamebaseEventMessage EventMessage;
            EventMessage.Category = GamebaseEventCategory::ObserverLaunching;
            EventMessage.Data = EventData.ToJson();

            EventHandler->Notify(EventMessage);
            
            Callback(FGamebaseLaunchingStatusResult(NewLaunchingInfo.Launching.Status.Code));
        });
    });
}

void FGamebaseStandaloneLaunching::OnUpdateAuthToken(const TOptional<FGamebaseAuthToken>& AuthToken, const TOptional<FString>& ProviderName)
{
    // Gamebase-Client/1590
    if (AuthToken.IsSet() == true)
    {
        LaunchingScheduler->StopScheduler();
        LaunchingScheduler->StartScheduler();
    }
}

bool FGamebaseStandaloneLaunching::IsNeedUpdate() const
{
    if (InternalData->IsInitialize() == false)
    {
        return true;
    }
    
    static constexpr double LaunchingExpireIntervalTimeSecond = 30;
    return FPlatformTime::Seconds() - LastUpdateTime > LaunchingExpireIntervalTimeSecond;
}

void FGamebaseStandaloneLaunching::ShowNoticePopup(const FGamebaseLaunchingInfo& LaunchingInfo, FSystemPopupCloseFunction& Callback)
{
    if (LaunchingInfo.Launching.Notice.IsSet() == false)
    {
        Callback();
        return;
    }
    
    GamebaseSystemPopup::ShowLaunchingNotice(
        *InternalData->GetDisplayLanguage(),
        LaunchingInfo,
        [=]
        {
            Callback(); 
        });
}

void FGamebaseStandaloneLaunching::ShowStatusPopup(const FGamebaseLaunchingInfo& LaunchingInfo, FSystemPopupCloseFunction& Callback)
{
    const bool bEnablePopup = InternalData->GetConfiguration().bEnablePopup && InternalData->GetConfiguration().bEnableLaunchingStatusPopup;
    if (bEnablePopup == false)
    {
        Callback();
        return;
    }
    
    GamebaseSystemPopup::ShowLaunchingStatus(
    InternalData.Get(),
        *InternalData->GetDisplayLanguage(),
        LaunchingInfo,
        [=]
        {
            Callback();
        });
}

bool FGamebaseStandaloneLaunching::LaunchingError(const FGamebaseError& Error, FSystemPopupCloseFunction& Callback) const
{
    if (Error.Code != GamebaseErrorCode::LAUNCHING_UNREGISTERED_CLIENT)
    {
        Callback();
        return false;
    }

    const bool bEnablePopup = InternalData->GetConfiguration().bEnablePopup && InternalData->GetConfiguration().bEnableLaunchingStatusPopup;
    if (bEnablePopup == false)
    {
        Callback();
        return false;
    }

    const TOptional<FString>& InfoString = Error.FindExtra(GamebaseErrorExtras::UpdateInfo);
    if (InfoString.IsSet())
    {
        FGamebaseLaunchingUpdateInfo UpdateInfo;
        if (UpdateInfo.FromJson(InfoString.GetValue()))
        {
            GamebaseSystemPopup::ShowUnregisteredClient(
                *InternalData->GetDisplayLanguage(),
                UpdateInfo,
                [Callback]
                {
                    Callback();
                }
            );
            return true;
        }
    }

    Callback();
    return false;
}