﻿#include "GamebaseStandaloneImageNotice.h"

#include "GamebaseDebugLogger.h"
#include "GamebaseErrorUtil.h"
#include "GamebaseSystemUtils.h"
#include "GamebaseWebSocket.h"
#include "ImageNotice/GamebaseImageNoticeConstants.h"
#include "ImageNotice/GamebaseImageNoticePreferences.h"
#include "ImageNotice/GamebaseImageNoticeRequest.h"
#include "ImageNotice/GamebaseImageNoticeViewer.h"
#include "Server/GamebaseServerInfo.h"

void FGamebaseStandaloneImageNotice::ShowImageNotices(const FGamebaseImageNoticeConfiguration& Configuration, const FGamebaseErrorDelegate& CloseCallback, const FGamebaseImageNoticeEventDelegate& EventCallback)
{
    // if (GamebaseImageNoticeViewer::IsEnabled() == false)
    // {
    //     CloseCallback.ExecuteIfBound(&MakeShared<FGamebaseError, ESPMode::ThreadSafe>(
    //         GamebaseErrorCode::NOT_SUPPORTED,
    //         TEXT("WebView module not loaded"),
    //         GamebaseImageNotice::Domain).Get());
    //     return;
    // }
    
    if (IsInitialize() == false)
    {
        CloseCallback.ExecuteIfBound(GamebaseErrorUtil::NewError(GamebaseImageNotice::Domain, GamebaseErrorCode::NOT_INITIALIZED).Get());
        return;
    }
    
    if (Preferences.IsValid() == false)
    {
        Preferences = MakeShared<FGamebaseImageNoticePreferences>(InternalData->GetAppId());
        Preferences->Initialize();
    }
    
    if (bWaitRequest || Viewer.IsValid())
    {
        GAMEBASE_LOG_WARNING("Already show process image notice.");
        return;
    }

    bWaitRequest = true;

    RequestGetImageNotices([this, Configuration, CloseCallback, EventCallback](const FGetImageNoticesResult& Result)
    {
        bWaitRequest = false;

        if (bShowCancel)
        {
            CloseCallback.ExecuteIfBound(nullptr);
            return;
        }

        if (Result.IsError())
        {
            CloseCallback.ExecuteIfBound(&Result.GetErrorValue());
            return;
        }

        const auto& ImageNoticeInfo = Result.GetOkValue();
        if (ImageNoticeInfo.bHasImageNotice == false)
        {
            CloseCallback.ExecuteIfBound(nullptr);
            return;
        }

        if (ImageNoticeInfo.Type.Equals(GamebaseImageNotice::ViewType::Rolling))
        {
            if (ImageNoticeInfo.RollingImageNoticeId.IsSet() == false)
            {
                GAMEBASE_LOG_WARNING("The ID for the rolling type is empty.");
                CloseCallback.ExecuteIfBound(&MakeShared<FGamebaseError, ESPMode::ThreadSafe>(GamebaseErrorCode::SERVER_INVALID_RESPONSE,
                    TEXT("The ID for the rolling type is empty."), GamebaseImageNotice::Domain).Get());
                return;
            }

            const int64 NoticeId = ImageNoticeInfo.RollingImageNoticeId.GetValue();
            Preferences->UpdateNextTimeData(NoticeId);
            
            if (Preferences->ShouldHideUntilTimeForNoticeId(NoticeId))
            {
                CloseCallback.ExecuteIfBound(nullptr);
                return;
            }

            const auto WebViewCloseCallback = FGamebaseErrorDelegate::CreateLambda([CloseCallback, this](const FGamebaseError* Error)
            {
                CloseCallback.ExecuteIfBound(Error);
                if (Viewer.IsValid())
                {
                    Viewer = nullptr;
                }
            });
            
            Viewer = MakeShared<FGamebaseImageNoticeViewer>(InternalData->GetGameInstance(), Preferences.ToSharedRef(), Configuration, ImageNoticeInfo, WebViewCloseCallback, EventCallback);
            Viewer->Show();
        }
        else if (ImageNoticeInfo.Type.Equals(GamebaseImageNotice::ViewType::Popup))
        {
            GAMEBASE_LOG_WARNING("Image notice in pop-up type are not supported.");
            CloseCallback.ExecuteIfBound(&MakeShared<FGamebaseError, ESPMode::ThreadSafe>(GamebaseErrorCode::UI_IMAGE_NOTICE_NOT_SUPPORTED_OS,
                TEXT("This OS does not support pop-up type image notice."), GamebaseImageNotice::Domain).Get());
        }
        else
        {
            GAMEBASE_LOG_WARNING("Received an undefined type. (type: %s)", *ImageNoticeInfo.Type);
            CloseCallback.ExecuteIfBound(&MakeShared<FGamebaseError, ESPMode::ThreadSafe>(GamebaseErrorCode::SERVER_INVALID_RESPONSE,
                TEXT("Received an undefined type."), GamebaseImageNotice::Domain).Get());
        }
    });
}

void FGamebaseStandaloneImageNotice::CloseImageNotices()
{
    if (bWaitRequest)
    {
        bShowCancel = true;
        return;
    }

    if (Viewer.IsValid())
    {
        Viewer->Close();
        Viewer = nullptr;
    }
}

void FGamebaseStandaloneImageNotice::RequestGetImageNotices(const FGetImageNoticesResultCallback& Callback)
{
    GamebaseImageNotice::FGetParameters Parameters;

    Parameters.ClientVersion = InternalData->GetAppVersion();
    Parameters.StoreCode = InternalData->GetStoreCode();
    Parameters.OSCode = GamebaseSystemUtils::GetPlatform();
    Parameters.DisplayLanguage = InternalData->GetDisplayLanguageCode();
    Parameters.DeviceLanguage = GamebaseSystemUtils::GetDeviceLanguage();
    Parameters.DeviceCountryCode = GamebaseSystemUtils::GetCountryCode();
    Parameters.UsimCountryCode = GamebaseSystemUtils::GetSimCode();

    const auto RequestCallback = [this, Callback](const FGamebaseWebSocketResponseResult& Response)
    {
        if (Response.IsError())
        {
            Callback(FGetImageNoticesResult(Response.GetErrorValue()));
            return;
        }

        const auto& ResponseData = Response.GetOkValue();
        if (ResponseData.Header.bIsSuccessful == false)
        {
            GAMEBASE_LOG_WARNING("Failed to send request to getImageNotices.\n%s", *ResponseData.Header.ToJson(true));
            Callback(FGetImageNoticesResult(*GamebaseErrorUtil::NewError(Response.GetOkValue(), GamebaseImageNotice::Domain)));
            return;
        }

        GamebaseImageNotice::FResponse ResponseImageNotices;
        ResponseImageNotices.FromJson(ResponseData.OriginData);
        Callback(FGetImageNoticesResult(ResponseImageNotices.ImageNoticeWeb));
    };
    
    WebSocket->Request(GamebaseServerInfo::Launching::ProductId,
        GamebaseServerInfo::Launching::Id::GetImageNotices,
        GamebaseServerInfo::ApiVersion,
        Parameters,
        RequestCallback);
}