#include "GamebaseStandaloneContact.h"

#include "GamebaseDebugLogger.h"
#include "GamebaseErrorUtil.h"
#include "GamebaseWebSocket.h"
#include "Contact/GamebaseContactConstants.h"
#include "ShortTermTicket/GamebaseShortTermTicketConstants.h"

void FGamebaseStandaloneContact::OpenContact(const FGamebaseErrorDelegate& CloseCallback)
{
    OpenContact(FGamebaseContactConfiguration(), CloseCallback);
}

void FGamebaseStandaloneContact::OpenContact(const FGamebaseContactConfiguration& Configuration, const FGamebaseErrorDelegate& CloseCallback)
{
    RequestContactURL(Configuration, FGamebaseContactUrlDelegate::CreateLambda([=](const FString& Url, const FGamebaseError* Error)
    {
        if (Error == nullptr)
        {
            GAMEBASE_LOG_DEBUG("URL : %s", *Url);
            FPlatformProcess::LaunchURL(*Url, nullptr, nullptr);
        }

        CloseCallback.ExecuteIfBound(Error);
    }));
}

void FGamebaseStandaloneContact::RequestContactURL(const FGamebaseContactUrlDelegate& Callback)
{
    RequestContactURL(FGamebaseContactConfiguration(), Callback);
}

void FGamebaseStandaloneContact::RequestContactURL(const FGamebaseContactConfiguration& Configuration, const FGamebaseContactUrlDelegate& Callback)
{
    if (IsInitialize() == false)
    {
        Callback.ExecuteIfBound({}, GamebaseErrorUtil::NewError(GamebaseContact::Domain, GamebaseErrorCode::NOT_INITIALIZED).Get());
        return;
    }

    RequestContactURLImpl(Configuration, FGamebaseContactUrlDelegate::CreateLambda([=](const FString& Url, const FGamebaseError* Error)
    {
        if (Error == nullptr)
        {
            GAMEBASE_LOG_DEBUG("URL : %s", *Url);
        }

        Callback.ExecuteIfBound(Url, Error);
    }));
}

void FGamebaseStandaloneContact::RequestContactURL(const GamebaseShortTermTicket::FIssueParameters& IssueParameter, const TOptional<FGamebaseContactConfiguration>& Configuration, const FGamebaseContactUrlDelegate& Callback) const
{
    const auto LaunchingInfo = InternalData->GetLaunchingInfo();
    if (LaunchingInfo->Launching.App.CustomerService.IsSet() == false)
    {
        Callback.ExecuteIfBound({}, GamebaseErrorUtil::NewError(GamebaseContact::Domain, GamebaseErrorCode::NOT_INITIALIZED).Get());
        return;
    }

    const auto CustomerService = LaunchingInfo->Launching.App.CustomerService.GetValue();
    if (CustomerService.Type == GamebaseContact::CsType::Toast)
    {
        ShortTermTicket->IssueShortTermTicket(IssueParameter, FIssueShortTermTicketDelegate::CreateLambda([=](const FString& Ticket, const FGamebaseError* Error)
        {
            if (Error == nullptr)
            {
                GamebaseContact::FToastTypeParameters Parameters = {
                    CustomerService.Url,
                    Ticket,
                    IssueParameter.Purpose,
                    IssueParameter.UserId,
                };
                
                Callback.ExecuteIfBound(GamebaseContact::GetToastCsUrlLogin(Parameters, Configuration), nullptr);
            }
            else
            {
                Callback.ExecuteIfBound(FString(), Error);
            }
        }));
    }
    else if(CustomerService.Type == GamebaseContact::CsType::Gamebase)
    {
        ShortTermTicket->IssueShortTermTicket(IssueParameter, FIssueShortTermTicketDelegate::CreateLambda([=](const FString& Ticket, const FGamebaseError* Error)
        {
            if (Error == nullptr)
            {
                const auto StoreCode = InternalData->GetStoreCode();
                const auto AppVersion = InternalData->GetAppVersion();
                    
                GamebaseContact::FGamebaseTypeParameters Parameters = {
                    CustomerService.Url,
                    Ticket,
                    IssueParameter.Purpose,
                    IssueParameter.UserId,
                    StoreCode,
                    AppVersion
                };
                    
                Callback.ExecuteIfBound(GamebaseContact::GetGamebaseCsUrl(Parameters, Configuration), nullptr);
            }
            else
            {
                Callback.ExecuteIfBound(FString(), Error);
            }
        }));
    }
    else // GamebaseContact::CsType::Custom
    {
        Callback.ExecuteIfBound(GamebaseContact::GetCustomCsUrl(CustomerService.Url, Configuration), nullptr);
    }
}

void FGamebaseStandaloneContact::RequestContactURLImpl(const TOptional<FGamebaseContactConfiguration>& Configuration, const FGamebaseContactUrlDelegate& Callback) const
{
    const auto LaunchingInfo = InternalData->GetLaunchingInfo();
    if (LaunchingInfo->Launching.App.CustomerService.IsSet() == false)
    {
        Callback.ExecuteIfBound({}, GamebaseErrorUtil::NewError(GamebaseContact::Domain, GamebaseErrorCode::NOT_INITIALIZED).Get());
        return;
    }

    const auto CustomerService = LaunchingInfo->Launching.App.CustomerService.GetValue();
    if (IsGamebaseLogin())
    {
        const auto AuthToken = InternalData->GetAuthToken();

        GamebaseShortTermTicket::FIssueParameters Parameter = GamebaseShortTermTicket::FIssueParameters();
        Parameter.UserId = AuthToken->Member.UserId;
        Parameter.Purpose = GamebaseShortTermTicket::Purpose::OpenContact;
        Parameter.ExpiresIn = GamebaseShortTermTicket::ExpiresIn;

        RequestContactURL(Parameter, Configuration, Callback);
    }
    else
    {
        if (CustomerService.Type == GamebaseContact::CsType::Toast)
        {
            Callback.ExecuteIfBound(GamebaseContact::GetToastCsUrlNotLogin(CustomerService.Url, Configuration), nullptr);
        }
        else if (CustomerService.Type == GamebaseContact::CsType::Gamebase)
        {
            Callback.ExecuteIfBound(GamebaseContact::GetGamebaseCsUrlNotLogin(CustomerService.Url, Configuration), nullptr);
        }
        else // GamebaseContact::CsType::Custom
        {
            Callback.ExecuteIfBound(GamebaseContact::GetCustomCsUrl(CustomerService.Url, Configuration), nullptr);
        }
    }
}
