#pragma once

#include "GamebaseCoreAuthTypes.h"
#include "GamebaseInternalModule.h"
#include "GamebaseInternalResult.h"

#include "Auth/GamebaseAuthProcessInfo.h"
#include "Types/GamebaseAuthTypes.h"
#include "Types/GamebaseBaseTypes.h"

struct FGamebaseLaunchingInfo;
struct FGamebaseTransferAccountRenewConfiguration;
struct FGamebaseWebSocketResponse;

class IAuthTokenUpdateHandler;
class IGamebaseAuthAdapter;
class FGamebaseAuthAdapterManager;
class FGamebaseAuthClient;
class FGamebaseSessionTicketClient;
class FGamebaseStandaloneLaunching;

using FGamebaseAuthTokenResult = FGamebaseInternalResult<FGamebaseAuthToken>;
using FGamebaseBindAuthTokenResult = FGamebaseErrorResult;
using FGamebaseLogoutResult = FGamebaseErrorResult;
using FGamebaseLaunchingStatusResult = FGamebaseInternalResult<int32>;

using FGamebaseStandaloneContactPtr = TSharedPtr<class FGamebaseStandaloneContact>;

class FGamebaseStandaloneAuth final : public FGamebaseInternalModule
{
    using FProviderName = FString;
    
public:
    explicit FGamebaseStandaloneAuth(const FGamebaseWebSocketPtr& WebSocket, const FGamebaseInternalDataPtr& InternalData, const FGamebaseStandaloneContactPtr& Contact);
    virtual ~FGamebaseStandaloneAuth() override;
    
    using FAuthTokenFunction = TFunction<void(const FGamebaseAuthTokenResult&)>;
    using FBanContactURLFunction = TFunction<void(FString)>;
    using FBindAuthTokenFunction = TFunction<void(const FGamebaseBindAuthTokenResult&)>;
    using FLogoutFunction = TFunction<void(const FGamebaseLogoutResult&)>;
    
    void Login(const FString& ProviderName, const FGamebaseVariantMap& AdditionalInfo, const FAuthTokenFunction& Callback);
    void Login(const FGamebaseVariantMap& CredentialInfo, const FAuthTokenFunction& Callback);
    void CancelLoginWithExternalBrowser(const FGamebaseErrorDelegate& Callback);
    
    void LoginForLastLoggedInProvider(const FGamebaseAuthTokenDelegate& Callback);
    void LoginForLastLoggedInProvider(const FGamebaseVariantMap& AdditionalInfo, const FGamebaseAuthTokenDelegate& Callback);
    
    void AddMapping(const FString& ProviderName, const FGamebaseVariantMap& AdditionalInfo, const FAuthTokenFunction& Callback);
    void AddMapping(const FGamebaseVariantMap& CredentialInfo, const FAuthTokenFunction& Callback);

    void ChangeLogin(const FGamebaseForcingMappingTicket& ForcingMappingTicket, const FAuthTokenFunction& Callback);
    void AddMappingForcibly(const FGamebaseForcingMappingTicket& ForcingMappingTicket, const FAuthTokenFunction& Callback);

    void AddMappingForcibly(const FString& ProviderName, const FString& ForcingMappingKey, const FGamebaseAuthTokenDelegate& Callback);
    void AddMappingForcibly(const FString& ProviderName, const FString& ForcingMappingKey, const FGamebaseVariantMap& AdditionalInfo, const FGamebaseAuthTokenDelegate& Callback);
    void AddMappingForcibly(const FGamebaseVariantMap& CredentialInfo, const FString& ForcingMappingKey, const FGamebaseAuthTokenDelegate& Callback);
    void RemoveMapping(const FString& ProviderName, const FGamebaseErrorDelegate& Callback);

    void Logout(const FGamebaseErrorDelegate& Callback);
    void Logout(const FGamebaseVariantMap& AdditionalInfo, const FGamebaseErrorDelegate& Callback);
    void Withdraw(const FGamebaseErrorDelegate& Callback);
    void WithdrawImmediately(const FGamebaseErrorDelegate& Callback);
    void RequestTemporaryWithdrawal(const FGamebaseTemporaryWithdrawalDelegate& Callback);
    void CancelTemporaryWithdrawal(const FGamebaseErrorDelegate& Callback);

    void IssueTransferAccount(const FGamebaseTransferAccountDelegate& Callback);
    void QueryTransferAccount(const FGamebaseTransferAccountDelegate& Callback);
    void RenewTransferAccount(const FGamebaseTransferAccountRenewConfiguration& Configuration, const FGamebaseTransferAccountDelegate& Callback);
    void TransferAccountWithIdPLogin(const FString& AccountId, const FString& AccountPassword, const FGamebaseAuthTokenDelegate& Callback);

    TArray<FString> GetAuthMappingList();
    FString GetAuthProviderUserID(const FString& ProviderName);
    FString GetAuthProviderAccessToken(const FString& ProviderName);
    FGamebaseAuthProviderProfilePtr GetAuthProviderProfile(const FString& ProviderName);

    FGamebaseBanInfoPtr GetBanInfo();
    
    void OnUpdateLaunchingInfo(const FGamebaseLaunchingInfo& LaunchingInfo);
    
    DECLARE_MULTICAST_DELEGATE_TwoParams(FUpdateAuthTokenHandler, const TOptional<FGamebaseAuthToken>&, const TOptional<FString>&);
    FUpdateAuthTokenHandler OnUpdateAuthToken;
    
private:
    void LoginWithGamebaseAccessToken(const FGamebaseVariantMap& CredentialInfo, const FAuthTokenFunction& Callback);

    void LogoutIdP(const FGamebaseVariantMap& AdditionalInfo, FLogoutFunction&& Callback);
    
    void RequestIdpLogin(const FString& ProviderName, bool bIsAutoLogin, const FGamebaseAuthCredentials& Credentials, const FAuthTokenFunction& Callback);
    void RequestBindIdPToken(const FString& ProviderName, const FGamebaseAuthCredentials& Credentials, const FBindAuthTokenFunction& Callback);

    void OnLoginSuccess(const FGamebaseAuthTokenPtr& AuthToken);
    void OnLoginComplete(const FGamebaseAuthTokenPtr& AuthToken, const FString& ProviderName);

    void OnMappingComplete(const FGamebaseAuthToken& AuthToken, const FString& ProviderName);

    void RemoveLoginData();

    void SaveLastAuthError(const FGamebaseError* Error);
    
    FGamebaseErrorPtr CanLogin(bool bIsIgnoreAlreadyLoggedIn = false) const;
    FGamebaseErrorPtr CanMapping() const;

private:
    TUniquePtr<FGamebaseAuthAdapterManager> AdapterManager;
    TUniquePtr<FGamebaseAuthClient> AuthClient;
    TSharedPtr<FGamebaseSessionTicketClient> SessionTicketClient;
    
    FGamebaseAuthProcessInfo AuthProcessInfo;
    FGamebaseErrorPtr LastAuthError;

    FGamebaseStandaloneContactPtr Contact;
};
