#include "GamebaseStandaloneAuth.h"

#include "Auth/GamebaseAuthAdapterManager.h"
#include "Auth/GamebaseAuthClient.h"
#include "Auth/GamebaseAuthConstants.h"
#include "Auth/GamebaseSessionTicketClient.h"

#include "GamebaseAuthProviderCredential.h"
#include "GamebaseDebugLogger.h"
#include "GamebaseErrorCode.h"
#include "GamebaseErrorUtil.h"
#include "GamebaseInternalReport.h"
#include "GamebaseStandaloneAuthProviderCredential.h"
#include "GamebaseWebSocket.h"

FGamebaseStandaloneAuth::FGamebaseStandaloneAuth(const FGamebaseWebSocketPtr& WebSocket, const FGamebaseInternalDataPtr& InternalData, const FGamebaseStandaloneContactPtr& Contact)
    : FGamebaseInternalModule(WebSocket, InternalData)
    , AdapterManager(MakeUnique<FGamebaseAuthAdapterManager>(InternalData))
    , AuthClient(MakeUnique<FGamebaseAuthClient>(WebSocket, InternalData))
    , SessionTicketClient(MakeShared<FGamebaseSessionTicketClient>(WebSocket, InternalData))
    , Contact(Contact)
{
    AuthClient->SetContact(Contact);
    
    AdapterManager->SetSessionTicketClient(SessionTicketClient);
}

FGamebaseStandaloneAuth::~FGamebaseStandaloneAuth()
{
    AdapterManager.Reset();
    AuthClient.Reset();
    SessionTicketClient.Reset();
}

void FGamebaseStandaloneAuth::Login(const FString& ProviderName, const FGamebaseVariantMap& AdditionalInfo, const FAuthTokenFunction& Callback)
{
    const bool bIsIgnoreAlreadyLoggedIn = AdditionalInfo.Contains(GamebaseAuthProviderCredential::IgnoreAlreadyLoggedIn) && AdditionalInfo[GamebaseAuthProviderCredential::IgnoreAlreadyLoggedIn];
    if (const auto Error = CanLogin(bIsIgnoreAlreadyLoggedIn))
    {
        Callback(FGamebaseAuthTokenResult(*Error.Get()));
        return;
    }
    
    AuthProcessInfo = FGamebaseAuthProcessInfo(ProviderName, TEXT(""), InternalData->GetStoreCode());
    AuthProcessInfo.Step = EGamebaseAuthStep::AdapterLogin;
    bool bHasLoggedInWithoutStore = true;
    
    const FString StoreCode = InternalData->GetStoreCode();
    
    FGamebaseVariantMap UpdateAdditionalInfo = AdditionalInfo;
    if (const auto& SupportStoreCode = GamebaseAuth::Store::SupportProviderMap.GetStoreByProvider(ProviderName))
    {
        if (SupportStoreCode->Code.Equals(StoreCode))
        {
            UpdateAdditionalInfo.Emplace(GamebaseAuthProviderCredential::Store, SupportStoreCode->Code);
            bHasLoggedInWithoutStore = false;
        }
    }

    bool bIsAutoLogin = AdditionalInfo.Contains(GamebaseAuthProviderCredential::AutoLogin) && AdditionalInfo[GamebaseAuthProviderCredential::AutoLogin];
    
    AdapterManager->Login(ProviderName, UpdateAdditionalInfo, [this, ProviderName, bIsAutoLogin, StoreCode, bHasLoggedInWithoutStore, Callback](const FGamebaseAuthCredentialsResult& Result)
    {
        SaveLastAuthError(Result.TryGetErrorValue());
        
        if (Result.IsError())
        {
            AuthProcessInfo.Reset();
            const FGamebaseError& Error = Result.GetErrorValue(); 
            Callback(FGamebaseAuthTokenResult(Error));
            return;
        }

        AuthProcessInfo.Step = EGamebaseAuthStep::GamebaseLogin;
        RequestIdpLogin(ProviderName, bIsAutoLogin, Result.GetOkValue(), [this, ProviderName, StoreCode, bHasLoggedInWithoutStore, Callback](const FGamebaseAuthTokenResult& LoginResult)
        {
            if (LoginResult.IsError())
            {
                AuthProcessInfo.Reset();
                Callback(LoginResult);
                return;
            }

            const FGamebaseAuthTokenPtr AuthToken = MakeShared<FGamebaseAuthToken, ESPMode::ThreadSafe>(LoginResult.GetOkValue());
            OnLoginSuccess(AuthToken);

            const auto CompleteFunc = [this, ProviderName, AuthToken, LoginResult, Callback]
            {
                OnLoginComplete(AuthToken, ProviderName);
                Callback(LoginResult);
                AuthProcessInfo.Reset();
            };

            if (bHasLoggedInWithoutStore)
            {
                if (const auto& StoreProviderName = GamebaseAuth::Store::SupportProviderMap.GetProviderByStore(StoreCode))
                {
                    FGamebaseVariantMap StoreAdditionalInfo;
                    StoreAdditionalInfo.Add(GamebaseAuthProviderCredential::Store, StoreCode);

                    AuthProcessInfo.StoreProviderName = StoreProviderName->Code;
                    AuthProcessInfo.Step = EGamebaseAuthStep::AdapterStoreLogin;
                    
                    AdapterManager->Login(StoreProviderName->Code, StoreAdditionalInfo, [this, StoreProviderName, CompleteFunc, Callback](const FGamebaseAuthCredentialsResult& Result)
                    {
                        if (Result.IsError())
                        {
                            GAMEBASE_LOG_DEBUG("Failed to get credentials for %s.\n%s", *StoreProviderName->Code, *Result.GetErrorValue().ToPrettyJson());
                            CompleteFunc();
                            return;
                        }

                        if (StoreProviderName->bBindToken)
                        {
                            AuthProcessInfo.Step = EGamebaseAuthStep::StoreBindToken;
                            RequestBindIdPToken(StoreProviderName->Code, Result.GetOkValue(), [this, StoreProviderName, CompleteFunc](const FGamebaseBindAuthTokenResult& BindTokenResult)
                            {
                                if (BindTokenResult.IsError())
                                {
                                    GAMEBASE_LOG_DEBUG("Failed to bind token to %s.\n%s", *StoreProviderName->Code, *BindTokenResult.GetErrorValue().ToPrettyJson());
                                }
                                CompleteFunc();
                            });
                        }
                        else
                        {
                            CompleteFunc();
                        }
                    });
                    return;
                }
            }
            
            CompleteFunc();
        });
    });
}

void FGamebaseStandaloneAuth::Login(const FGamebaseVariantMap& CredentialInfo, const FAuthTokenFunction& Callback)
{
    const bool bIsIgnoreAlreadyLoggedIn = CredentialInfo.Contains(GamebaseAuthProviderCredential::IgnoreAlreadyLoggedIn) && CredentialInfo[GamebaseAuthProviderCredential::IgnoreAlreadyLoggedIn];
    if (const auto Error = CanLogin(bIsIgnoreAlreadyLoggedIn))
    {
        Callback(FGamebaseAuthTokenResult(*Error.Get()));
        return;
    }

    if (CredentialInfo.Contains(GamebaseAuthProviderCredential::GamebaseAccessToken))
    {
        LoginWithGamebaseAccessToken(CredentialInfo, Callback);
        return;
    }

    if (!CredentialInfo.Contains(GamebaseAuthProviderCredential::ProviderName))
    {
        Callback(FGamebaseAuthTokenResult(*GamebaseErrorUtil::NewError(GamebaseAuth::Domain, GamebaseErrorCode::INVALID_PARAMETER, TEXT("ProviderName is empty"))));
        return;
    }

    const FString ProviderName = CredentialInfo.FindRef(GamebaseAuthProviderCredential::ProviderName);
    bool bIsAutoLogin = CredentialInfo.Contains(GamebaseAuthProviderCredential::AutoLogin) && CredentialInfo[GamebaseAuthProviderCredential::AutoLogin];

    bool bHasLoggedInWithoutStore = true;
    const FString StoreCode = InternalData->GetStoreCode();

    if (const auto& SupportStoreCode = GamebaseAuth::Store::SupportProviderMap.GetStoreByProvider(ProviderName))
    {
        if (SupportStoreCode->Code.Equals(StoreCode))
        {
            bHasLoggedInWithoutStore = false;
        }
    }

    AdapterManager->Login(ProviderName, CredentialInfo, [this, ProviderName, StoreCode, bHasLoggedInWithoutStore, bIsAutoLogin, Callback](const FGamebaseAuthCredentialsResult& Result)
    {
        SaveLastAuthError(Result.TryGetErrorValue());
        
        if (Result.IsError())
        {
            AuthProcessInfo.Reset();
            Callback(FGamebaseAuthTokenResult(Result.GetErrorValue()));
            return;
        }
        
        AuthProcessInfo.Step = EGamebaseAuthStep::GamebaseLogin;
        RequestIdpLogin(ProviderName, bIsAutoLogin, Result.GetOkValue(), [this, ProviderName, StoreCode, bHasLoggedInWithoutStore, Callback](const FGamebaseAuthTokenResult& LoginResult)
        {
            if (LoginResult.IsError())
            {
                Callback(LoginResult);
                return;
            }

            const FGamebaseAuthTokenPtr AuthToken = MakeShared<FGamebaseAuthToken, ESPMode::ThreadSafe>(LoginResult.GetOkValue());
            OnLoginSuccess(AuthToken);

            const auto CompleteFunc = [this, ProviderName, AuthToken, LoginResult, Callback]
            {
                OnLoginComplete(AuthToken, ProviderName);
                Callback(LoginResult);
                AuthProcessInfo.Reset();
            };

            if (bHasLoggedInWithoutStore)
            {
                if (const auto& StoreProviderName = GamebaseAuth::Store::SupportProviderMap.GetProviderByStore(StoreCode))
                {
                    FGamebaseVariantMap StoreAdditionalInfo;
                    StoreAdditionalInfo.Add(GamebaseAuthProviderCredential::Store, FVariant(StoreCode));
                    
                    AuthProcessInfo.StoreProviderName = StoreProviderName->Code;
                    AuthProcessInfo.Step = EGamebaseAuthStep::AdapterStoreLogin;
                    
                    AdapterManager->Login(StoreProviderName->Code, StoreAdditionalInfo, [this, StoreProviderName, CompleteFunc, Callback](const FGamebaseAuthCredentialsResult& Result)
                    {
                        if (Result.IsError())
                        {
                            GAMEBASE_LOG_DEBUG("Failed to get credentials for %s.\n%s", *StoreProviderName->Code, *Result.GetErrorValue().ToPrettyJson());
                            CompleteFunc();
                            return;
                        }

                        if (StoreProviderName->bBindToken)
                        {
                            AuthProcessInfo.Step = EGamebaseAuthStep::StoreBindToken;
                            RequestBindIdPToken(StoreProviderName->Code, Result.GetOkValue(), [this, StoreProviderName, CompleteFunc](const FGamebaseBindAuthTokenResult& BindTokenResult)
                            {
                                if (BindTokenResult.IsError())
                                {
                                    GAMEBASE_LOG_DEBUG("Failed to bind token to %s.\n%s", *StoreProviderName->Code, *BindTokenResult.GetErrorValue().ToPrettyJson());
                                }
                                CompleteFunc();
                            });
                        }
                        else
                        {
                            CompleteFunc();
                        }
                    });
                    return;
                }
            }
            
            CompleteFunc();
        });
    });
}

void FGamebaseStandaloneAuth::CancelLoginWithExternalBrowser(const FGamebaseErrorDelegate& Callback)
{
    auto CanCancelLogin = [this]() -> FGamebaseErrorPtr
    {
        if (IsInitialize() == false)
        {
            return GamebaseErrorUtil::NewError(GamebaseAuth::Domain, GamebaseErrorCode::NOT_INITIALIZED);
        }
        
        if (AuthProcessInfo.IsSet() == false)
        {
            return GamebaseErrorUtil::NewError(GamebaseAuth::Domain, GamebaseErrorCode::AUTH_LOGIN_CANCEL_FAILED, TEXT("Not logged in"));
        }
        
        return nullptr;
    };
    
    if (const auto Error = CanCancelLogin())
    {
        Callback.ExecuteIfBound(Error.Get());
        return;
    }

    FString CancelProviderName;
    if (AuthProcessInfo.Step == EGamebaseAuthStep::AdapterLogin)
    {
        CancelProviderName = AuthProcessInfo.ProviderName;
    }
    else if (AuthProcessInfo.Step == EGamebaseAuthStep::AdapterStoreLogin)
    {
        CancelProviderName = AuthProcessInfo.StoreProviderName;
    }

    if (CancelProviderName.IsEmpty())
    {
        Callback.ExecuteIfBound(GamebaseErrorUtil::NewError(GamebaseAuth::Domain, GamebaseErrorCode::AUTH_LOGIN_CANCEL_FAILED, TEXT("No provider to cancel login")).Get());
        return;
    }

    AdapterManager->CancelLogin(CancelProviderName, [this, Callback](const FGamebaseErrorResult& Result)
    {
        if (Result.IsError())
        {
            Callback.ExecuteIfBound(&Result.GetErrorValue());
            return;
        }

        AuthProcessInfo.Reset();
        Callback.ExecuteIfBound(nullptr);
        
        SaveLastAuthError(nullptr);
        //GamebaseInternalReport::Auth::CancelLogin(*InternalData, nullptr);
    });
}

void FGamebaseStandaloneAuth::LoginForLastLoggedInProvider(const FGamebaseAuthTokenDelegate& Callback)
{
    GAMEBASE_NOT_SUPPORT_API();
}

void FGamebaseStandaloneAuth::LoginForLastLoggedInProvider(const FGamebaseVariantMap& AdditionalInfo, const FGamebaseAuthTokenDelegate& Callback)
{
    GAMEBASE_NOT_SUPPORT_API();
}

void FGamebaseStandaloneAuth::AddMapping(const FString& ProviderName, const FGamebaseVariantMap& AdditionalInfo, const FAuthTokenFunction& Callback)
{
    if (const auto Error = CanMapping())
    {
        Callback(FGamebaseAuthTokenResult(*Error.Get()));
        return;
    }
    
    AuthProcessInfo = FGamebaseAuthProcessInfo(ProviderName, TEXT(""), InternalData->GetStoreCode());
    AuthProcessInfo.Step = EGamebaseAuthStep::AdapterLogin;
    
    const FString StoreCode = InternalData->GetStoreCode();
    
    AdapterManager->Login(ProviderName, AdditionalInfo, [this, ProviderName, Callback](const FGamebaseAuthCredentialsResult& Result)
    {
        if (Result.IsError())
        {
            AuthProcessInfo.Reset();
            const FGamebaseError& Error = Result.GetErrorValue(); 
            Callback(FGamebaseAuthTokenResult(Error));
            return;
        }

        AuthProcessInfo = FGamebaseAuthProcessInfo();
        AuthProcessInfo.Step = EGamebaseAuthStep::AddMapping;
        
        AuthClient->AddMapping(InternalData->GetUserId(), ProviderName, Result.GetOkValue(), [this, ProviderName, Callback](const FGamebaseAuthTokenResult& MappingResult)
        {
            AuthProcessInfo.Reset();

            if (MappingResult.IsOk())
            {
                OnMappingComplete(MappingResult.GetOkValue(), ProviderName);
                Callback(MappingResult);
            }
            else
            {
                const int32 ErrorCode = MappingResult.GetErrorValue().Code;
                if (ErrorCode == GamebaseErrorCode::SOCKET_ERROR || ErrorCode == GamebaseErrorCode::SOCKET_RESPONSE_TIMEOUT)
                {
                }
                
                if (ErrorCode != GamebaseErrorCode::AUTH_ADD_MAPPING_ALREADY_MAPPED_TO_OTHER_MEMBER)
                {
                    AdapterManager->Logout(ProviderName, [](const FGamebaseErrorResult& LogoutResult) {});
                }
                
                Callback(MappingResult);
            }
        });
    });
}

void FGamebaseStandaloneAuth::AddMapping(const FGamebaseVariantMap& CredentialInfo, const FAuthTokenFunction& Callback)
{
    if (const auto Error = CanMapping())
    {
        Callback(FGamebaseAuthTokenResult(*Error.Get()));
        return;
    }

    if (!CredentialInfo.Contains(GamebaseAuthProviderCredential::ProviderName))
    {
        Callback(FGamebaseAuthTokenResult(*GamebaseErrorUtil::NewError(GamebaseAuth::Domain, GamebaseErrorCode::INVALID_PARAMETER, TEXT("ProviderName is empty"))));
        return;
    }

    const FString ProviderName = CredentialInfo.FindRef(GamebaseAuthProviderCredential::ProviderName);

    AuthProcessInfo = FGamebaseAuthProcessInfo();
    AuthProcessInfo.Step = EGamebaseAuthStep::AddMapping;
    
    AuthClient->AddMapping(InternalData->GetUserId(), ProviderName, CredentialInfo, [this, Callback](const FGamebaseAuthTokenResult& MappingResult)
    {
        Callback(MappingResult);
        AuthProcessInfo.Reset();
    });
}

void FGamebaseStandaloneAuth::ChangeLogin(const FGamebaseForcingMappingTicket& ForcingMappingTicket, const FAuthTokenFunction& Callback)
{
    if (const auto Error = CanLogin())
    {
        Callback(FGamebaseAuthTokenResult(*Error.Get()));
        return;
    }

    FGamebaseVariantMap CredentialInfo;
    CredentialInfo.Add(GamebaseAuthProviderCredential::ProviderName, ForcingMappingTicket.IdPCode);
    CredentialInfo.Add(GamebaseAuthProviderCredential::GamebaseAccessToken, ForcingMappingTicket.AccessToken);
    CredentialInfo.Add(GamebaseAuthProviderCredential::IgnoreAlreadyLoggedIn, true);

    if (ForcingMappingTicket.SubCode.IsSet())
    {
        CredentialInfo.Add(GamebaseAuthProviderCredential::SubCode, ForcingMappingTicket.SubCode.GetValue());
    }

    if (ForcingMappingTicket.ExtraParams.IsSet())
    {
        CredentialInfo.Add(GamebaseAuthProviderCredential::ExtraParams, ForcingMappingTicket.ExtraParams.GetValue());
    }

    AuthProcessInfo = FGamebaseAuthProcessInfo();
    AuthProcessInfo.Step = EGamebaseAuthStep::ChangeLogin;
    
    Login(CredentialInfo, [this, CredentialInfo, ForcingMappingTicket, Callback](const FGamebaseAuthTokenResult& Result)
    {
        AuthProcessInfo.Reset();
        if (Result.IsError())
        {
            Callback(Result);
            return;
        }

        Callback(Result);
        GamebaseInternalReport::Auth::ChangeLogin(*InternalData, CredentialInfo, ForcingMappingTicket, Result.TryGetErrorValue());
    });
}

void FGamebaseStandaloneAuth::AddMappingForcibly(const FGamebaseForcingMappingTicket& ForcingMappingTicket, const FAuthTokenFunction& Callback)
{
    if (!IsGamebaseLogin())
    {
        Callback(FGamebaseAuthTokenResult(*GamebaseErrorUtil::NewError(GamebaseAuth::Domain, GamebaseErrorCode::NOT_LOGGED_IN).Get()));
        return;
    }
    
    if (AuthProcessInfo.IsSet())
    {
        Callback(FGamebaseAuthTokenResult(*GamebaseErrorUtil::NewError(GamebaseAuth::Domain, GamebaseErrorCode::AUTH_ALREADY_IN_PROGRESS_ERROR).Get()));
        return;
    }
    
    AuthProcessInfo = FGamebaseAuthProcessInfo();
    AuthProcessInfo.Step = EGamebaseAuthStep::AddMappingForcibly;
    
    AuthClient->AddMappingForcibly(InternalData->GetUserId(), ForcingMappingTicket, [this, ForcingMappingTicket, Callback](const FGamebaseAuthTokenResult& Result)
    {
        AuthProcessInfo.Reset();
        
        if (Result.IsError())
        {
            Callback(Result);
            return;
        }
        
        OnMappingComplete(Result.GetOkValue(), ForcingMappingTicket.IdPCode);
        Callback(Result);
        GamebaseInternalReport::Auth::MappingWithProvider(*InternalData, ForcingMappingTicket, Result.TryGetErrorValue());
    });
    
    GAMEBASE_NOT_SUPPORT_API();
}

void FGamebaseStandaloneAuth::AddMappingForcibly(const FString& ProviderName, const FString& ForcingMappingKey, const FGamebaseAuthTokenDelegate& Callback)
{
    GAMEBASE_NOT_SUPPORT_API();
}

void FGamebaseStandaloneAuth::AddMappingForcibly(const FString& ProviderName, const FString& ForcingMappingKey, const FGamebaseVariantMap& AdditionalInfo, const FGamebaseAuthTokenDelegate& Callback)
{
    GAMEBASE_NOT_SUPPORT_API();
}

void FGamebaseStandaloneAuth::AddMappingForcibly(const FGamebaseVariantMap& CredentialInfo, const FString& ForcingMappingKey, const FGamebaseAuthTokenDelegate& Callback)
{
    GAMEBASE_NOT_SUPPORT_API();
}

void FGamebaseStandaloneAuth::RemoveMapping(const FString& ProviderName, const FGamebaseErrorDelegate& Callback)
{
    if (const auto Error = CanMapping())
    {
        Callback.ExecuteIfBound(Error.Get());
        return;
    }

    AuthProcessInfo = FGamebaseAuthProcessInfo();
    AuthProcessInfo.Step = EGamebaseAuthStep::RemoveMapping;
    
    AuthClient->RemoveMapping(ProviderName, InternalData->GetUserId(), InternalData->GetAccessToken(), [this, ProviderName, Callback](const FGamebaseAuthTokenResult& Result)
    {
        AuthProcessInfo.Reset();
        
        if (Result.IsError())
        {
            Callback.ExecuteIfBound(&Result.GetErrorValue());
            return;
        }

        OnMappingComplete(Result.GetOkValue(), ProviderName);
        Callback.ExecuteIfBound(nullptr);
        
        AdapterManager->Logout(ProviderName, [this, ProviderName, Callback](const FGamebaseErrorResult& LogoutResult)
        {
        });
    });
}

void FGamebaseStandaloneAuth::Logout(const FGamebaseErrorDelegate& Callback)
{
    Logout({}, Callback);
}

void FGamebaseStandaloneAuth::Logout(const FGamebaseVariantMap& AdditionalInfo, const FGamebaseErrorDelegate& Callback)
{
    if (InternalData->IsLogin() == false)
    {
        Callback.ExecuteIfBound(GamebaseErrorUtil::NewError(GamebaseAuth::Domain, GamebaseErrorCode::NOT_LOGGED_IN).Get());
        return;
    }
    
    if (AdditionalInfo.GetValueOrDefault(GamebaseAuth::Logout::SkipExpireGamebaseToken, false))
    {
        RemoveLoginData();
        SaveLastAuthError(nullptr);
        Callback.ExecuteIfBound(nullptr);
        return;
    }

    LogoutIdP(AdditionalInfo, [this, AdditionalInfo, Callback](const FGamebaseLogoutResult& Result)
    {
        if (Result.IsError())
        {
            Callback.ExecuteIfBound(&Result.GetErrorValue());
            return;
        }
        
        AuthClient->Logout([this, AdditionalInfo, Callback](const FGamebaseErrorResult& Result)
        {
            if (Result.IsOk())
            {
                RemoveLoginData();
                Callback.ExecuteIfBound(nullptr);
            }
            else
            {
                Callback.ExecuteIfBound(&Result.GetErrorValue());
            }
            
            SaveLastAuthError(nullptr);
            GamebaseInternalReport::Auth::Logout(*InternalData, Result.TryGetErrorValue(), AdditionalInfo);
        });
    });
}

void FGamebaseStandaloneAuth::Withdraw(const FGamebaseErrorDelegate& Callback)
{
    if (InternalData->IsLogin() == false)
    {
        Callback.ExecuteIfBound(GamebaseErrorUtil::NewError(GamebaseAuth::Domain, GamebaseErrorCode::NOT_LOGGED_IN).Get());
        return;
    }

    AdapterManager->WithdrawAll([this, Callback](const FGamebaseErrorResult& Result)
    {
        SaveLastAuthError(nullptr);
        
        if (Result.IsError())
        {
            Callback.ExecuteIfBound(&Result.GetErrorValue());
            return;
        }

        AuthClient->Withdraw([this, Callback](const FGamebaseErrorResult& Result)
        {
            if (Result.IsOk())
            {
                RemoveLoginData();
                Callback.ExecuteIfBound(nullptr);
            }
            else
            {
                Callback.ExecuteIfBound(&Result.GetErrorValue());
            }
            
            GamebaseInternalReport::Auth::Withdraw(*InternalData, Result.TryGetErrorValue());
        });
    });
}

void FGamebaseStandaloneAuth::WithdrawImmediately(const FGamebaseErrorDelegate& Callback)
{
    Withdraw(Callback);
}

void FGamebaseStandaloneAuth::RequestTemporaryWithdrawal(const FGamebaseTemporaryWithdrawalDelegate& Callback)
{
    if (InternalData->IsLogin() == false)
    {
        Callback.ExecuteIfBound(nullptr, GamebaseErrorUtil::NewError(GamebaseAuth::Domain, GamebaseErrorCode::NOT_LOGGED_IN).Get());
        return;
    }

    AuthClient->TemporaryWithdrawal([this, Callback](const FGamebaseTemporaryWithdrawalResult& Result)
    {
        if (Result.IsOk())
        {
            const auto TemporaryWithdrawalInfo = MakeShared<FGamebaseTemporaryWithdrawalInfo, ESPMode::ThreadSafe>(Result.GetOkValue());
            Callback.ExecuteIfBound(&TemporaryWithdrawalInfo.Get(), nullptr);
        }
        else
        {
            Callback.ExecuteIfBound(nullptr, &Result.GetErrorValue());
        }
        
        SaveLastAuthError(nullptr);
        GamebaseInternalReport::Auth::RequestTemporaryWithdrawal(*InternalData, Result.TryGetErrorValue());
    });
}

void FGamebaseStandaloneAuth::CancelTemporaryWithdrawal(const FGamebaseErrorDelegate& Callback)
{
    if (InternalData->IsLogin() == false)
    {
        Callback.ExecuteIfBound(GamebaseErrorUtil::NewError(GamebaseAuth::Domain, GamebaseErrorCode::NOT_LOGGED_IN).Get());
        return;
    }

    AuthClient->CancelTemporaryWithdrawal([this, Callback](const FGamebaseErrorResult& Result)
    {
        if (Result.IsOk())
        {
            Callback.ExecuteIfBound(nullptr);
        }
        else
        {
            Callback.ExecuteIfBound(&Result.GetErrorValue());
        }
        
        SaveLastAuthError(nullptr);
        GamebaseInternalReport::Auth::CancelTemporaryWithdrawal(*InternalData, Result.TryGetErrorValue());
    });
}

void FGamebaseStandaloneAuth::IssueTransferAccount(const FGamebaseTransferAccountDelegate& Callback)
{
    GAMEBASE_NOT_SUPPORT_API();
}

void FGamebaseStandaloneAuth::QueryTransferAccount(const FGamebaseTransferAccountDelegate& Callback)
{
    GAMEBASE_NOT_SUPPORT_API();
}

void FGamebaseStandaloneAuth::RenewTransferAccount(const FGamebaseTransferAccountRenewConfiguration& Configuration, const FGamebaseTransferAccountDelegate& Callback)
{
    GAMEBASE_NOT_SUPPORT_API();
}

void FGamebaseStandaloneAuth::TransferAccountWithIdPLogin(const FString& AccountId, const FString& AccountPassword, const FGamebaseAuthTokenDelegate& Callback)
{
    GAMEBASE_NOT_SUPPORT_API();
}

TArray<FString> FGamebaseStandaloneAuth::GetAuthMappingList()
{
    if (IsInitialize() == false)
    {
        GAMEBASE_LOG_DEBUG("Auth is not initialized.");
        return {};
    }

    if (InternalData->GetAuthToken() == nullptr)
    {
        GAMEBASE_LOG_DEBUG("Auth token is not set.");
        return {};
    }

    TArray<FString> AuthMappingList;
    for (const auto& Mapping : InternalData->GetAuthToken()->Member.AuthList)
    {
        AuthMappingList.Add(Mapping.IdpCode);
    }
    
    return AuthMappingList;
}

FString FGamebaseStandaloneAuth::GetAuthProviderUserID(const FString& ProviderName)
{
    return AdapterManager.IsValid() ? AdapterManager->GetUserID(ProviderName) : FString();
}

FString FGamebaseStandaloneAuth::GetAuthProviderAccessToken(const FString& ProviderName)
{
    return AdapterManager.IsValid() ? AdapterManager->GetAccessToken(ProviderName) : FString();
}

FGamebaseAuthProviderProfilePtr FGamebaseStandaloneAuth::GetAuthProviderProfile(const FString& ProviderName)
{
    return AdapterManager.IsValid() ? AdapterManager->GetProfile(ProviderName) : nullptr;
}

FGamebaseBanInfoPtr FGamebaseStandaloneAuth::GetBanInfo()
{
    if (LastAuthError.IsValid() == false)
    {
        GAMEBASE_LOG_WARNING("No ban data");
        return nullptr;
    }
    
    return FGamebaseBanInfo::From(LastAuthError.Get());
}

void FGamebaseStandaloneAuth::OnUpdateLaunchingInfo(const FGamebaseLaunchingInfo& LaunchingInfo)
{
    AdapterManager->OnUpdateLaunchingInfo(LaunchingInfo);
}

void FGamebaseStandaloneAuth::LoginWithGamebaseAccessToken(const FGamebaseVariantMap& CredentialInfo, const FAuthTokenFunction& Callback)
{
    if (CredentialInfo.Contains(GamebaseAuthProviderCredential::GamebaseAccessToken) == false)
    {
        Callback(FGamebaseAuthTokenResult(FGamebaseError(*GamebaseErrorUtil::NewError(GamebaseAuth::Domain, GamebaseErrorCode::INVALID_PARAMETER, TEXT("GamebaseAccessToken is empty")))));
        return;
    }

    if (CredentialInfo.Contains(GamebaseAuthProviderCredential::ProviderName) == false)
    {
        Callback(FGamebaseAuthTokenResult(FGamebaseError(*GamebaseErrorUtil::NewError(GamebaseAuth::Domain, GamebaseErrorCode::INVALID_PARAMETER, TEXT("ProviderName is empty")))));
        return;
    }
    
    const FString GamebaseAccessToken = CredentialInfo.FindRef(GamebaseAuthProviderCredential::GamebaseAccessToken);
    const FString ProviderName = CredentialInfo.FindRef(GamebaseAuthProviderCredential::ProviderName);
    if (AdapterManager->HasAdapter(ProviderName) == false)
    {
        Callback(FGamebaseAuthTokenResult(FGamebaseError(*GamebaseErrorUtil::NewError(GamebaseAuth::Domain, GamebaseErrorCode::AUTH_NOT_SUPPORTED_PROVIDER, FString::Printf(TEXT("Provider %s is not supported"), *ProviderName)))));
        return;
    }

    bool bHasLoggedInWithoutStore = true;
    const FString StoreCode = InternalData->GetStoreCode();
    
    if (const auto& SupportStoreCode = GamebaseAuth::Store::SupportProviderMap.GetStoreByProvider(ProviderName))
    {
        if (SupportStoreCode->Code.Equals(StoreCode))
        {
            bHasLoggedInWithoutStore = false;
        }
    }
    
    AuthProcessInfo = FGamebaseAuthProcessInfo(ProviderName, TEXT(""), StoreCode);
    
    AuthProcessInfo.Step = EGamebaseAuthStep::GamebaseLogin;
    AuthClient->TokenLogin(
        ProviderName,
        GamebaseAccessToken,
        CredentialInfo,
        [this, Callback, ProviderName, bHasLoggedInWithoutStore, StoreCode](const FGamebaseAuthTokenResult& Result)
        {
            if (Result.IsError())
            {
                AuthProcessInfo.Reset();
                Callback(FGamebaseAuthTokenResult(Result.GetErrorValue()));
                return;
            }
            
            const FGamebaseAuthTokenPtr AuthToken = MakeShared<FGamebaseAuthToken, ESPMode::ThreadSafe>(Result.GetOkValue());
            OnLoginSuccess(AuthToken);

            const auto CompleteFunc = [this, ProviderName, AuthToken, Result, Callback]
            {
                OnLoginComplete(AuthToken, ProviderName);
                Callback(Result);
                AuthProcessInfo.Reset();
            };

            if (bHasLoggedInWithoutStore)
            {
                if (const auto& StoreProviderName = GamebaseAuth::Store::SupportProviderMap.GetProviderByStore(StoreCode))
                {
                    FGamebaseVariantMap StoreAdditionalInfo;
                    StoreAdditionalInfo.Add(GamebaseAuthProviderCredential::Store, FVariant(StoreCode));

                    AuthProcessInfo.StoreProviderName = StoreProviderName->Code;
                    AuthProcessInfo.Step = EGamebaseAuthStep::AdapterStoreLogin;
                    
                    AdapterManager->Login(StoreProviderName->Code, StoreAdditionalInfo, [this, StoreProviderName, CompleteFunc, Callback](const FGamebaseAuthCredentialsResult& Result)
                        {
                            if (Result.IsError())
                            {
                                GAMEBASE_LOG_DEBUG("Failed to get credentials for %s.\n%s", *StoreProviderName->Code, *Result.GetErrorValue().ToPrettyJson());
                                CompleteFunc();
                                return;
                            }

                            if (StoreProviderName->bBindToken)
                            {
                                AuthProcessInfo.Step = EGamebaseAuthStep::StoreBindToken;
                                RequestBindIdPToken(StoreProviderName->Code, Result.GetOkValue(), [this, StoreProviderName, CompleteFunc](const FGamebaseBindAuthTokenResult& BindTokenResult)
                                {
                                    if (BindTokenResult.IsError())
                                    {
                                        GAMEBASE_LOG_DEBUG("Failed to bind token to %s.\n%s", *StoreProviderName->Code, *BindTokenResult.GetErrorValue().ToPrettyJson());
                                    }
                                    CompleteFunc();
                                });
                            }
                            else
                            {
                                CompleteFunc();
                            }
                        });
                    return;
                }
            }

            CompleteFunc();
        });
}

void FGamebaseStandaloneAuth::LogoutIdP(
    const FGamebaseVariantMap& AdditionalInfo,
    FLogoutFunction&& Callback)
{
    if (AdditionalInfo.GetValueOrDefault(GamebaseAuth::Logout::SkipIdpLogout, false))
    {
        Callback(FGamebaseErrorResult());
        return;
    }

    AdapterManager->LogoutAll([this, Callback](const FGamebaseErrorResult& Result)
    {
        if (Result.IsError())
        {
            Callback(FGamebaseErrorResult(Result.GetErrorValue()));
        }
        else
        {
            Callback(FGamebaseErrorResult());
        }
    });
}

void FGamebaseStandaloneAuth::RequestIdpLogin(const FString& ProviderName, const bool bIsAutoLogin, const FGamebaseAuthCredentials& Credentials, const FAuthTokenFunction& Callback)
{
    AuthClient->IdpLogin(ProviderName, bIsAutoLogin, Credentials, [this, Callback](FGamebaseAuthTokenResult Result)
    {
        if (Result.IsError())
        {
            AuthProcessInfo.Reset();
            Callback(FGamebaseAuthTokenResult(Result.GetErrorValue()));
        }
        else
        {
            Callback(FGamebaseAuthTokenResult(Result.GetOkValue()));
        }
    });
}

void FGamebaseStandaloneAuth::RequestBindIdPToken(const FString& ProviderName, const FGamebaseAuthCredentials& Credentials, const FBindAuthTokenFunction& Callback)
{
    AuthClient->BindIdPToken(ProviderName, Credentials, [Callback](const FGamebaseErrorResult& Result)
    {
        if (Result.IsOk())
        {
            Callback(FGamebaseBindAuthTokenResult());
        }
        else
        {
            Callback(FGamebaseBindAuthTokenResult(Result.GetErrorValue()));
        }
    });
}

void FGamebaseStandaloneAuth::OnLoginSuccess(const FGamebaseAuthTokenPtr& AuthToken)
{
    InternalData->SetAuthToken(AuthToken);
}

void FGamebaseStandaloneAuth::OnLoginComplete(const FGamebaseAuthTokenPtr& AuthToken, const FString& ProviderName)
{
    AuthProcessInfo.Reset();

    const TOptional<FGamebaseAuthToken> EventAuthToken = AuthToken.IsValid() ? *AuthToken : TOptional<FGamebaseAuthToken>();

    OnUpdateAuthToken.Broadcast(EventAuthToken, ProviderName);
}

void FGamebaseStandaloneAuth::OnMappingComplete(
    const FGamebaseAuthToken& AuthToken,
    const FString& ProviderName)
{
    InternalData->SetAuthMappingList(AuthToken.Member.AuthList);
}

void FGamebaseStandaloneAuth::RemoveLoginData()
{
    InternalData->SetAuthToken(nullptr);
    OnUpdateAuthToken.Broadcast(TOptional<FGamebaseAuthToken>(), TOptional<FString>());
    
    GamebaseInternalReport::TAA::ResetUserLevel(*InternalData);
}

void FGamebaseStandaloneAuth::SaveLastAuthError(const FGamebaseError* Error)
{
    if (LastAuthError.IsValid())
    {
        LastAuthError.Reset();
        LastAuthError = nullptr;
    }
    
    if (Error != nullptr)
    {
        LastAuthError = MakeShared<FGamebaseError, ESPMode::ThreadSafe>(*Error);
    }
}

FGamebaseErrorPtr FGamebaseStandaloneAuth::CanLogin(const bool bIsIgnoreAlreadyLoggedIn) const
{
    if (IsInitialize() == false)
    {
        return GamebaseErrorUtil::NewError(GamebaseAuth::Domain, GamebaseErrorCode::NOT_INITIALIZED);
    }

    if (AuthProcessInfo.IsSet())
    {
        return GamebaseErrorUtil::NewError(GamebaseAuth::Domain, GamebaseErrorCode::AUTH_ALREADY_IN_PROGRESS_ERROR);
    }
    
    if (IsGamebaseLogin() && !bIsIgnoreAlreadyLoggedIn)
    {
        return GamebaseErrorUtil::NewError(GamebaseAuth::Domain, GamebaseErrorCode::AUTH_IDP_LOGIN_FAILED, TEXT("You are already logged in"));
    }
    
    return nullptr;
}

FGamebaseErrorPtr FGamebaseStandaloneAuth::CanMapping() const
{
    if (IsInitialize() == false)
    {
        return GamebaseErrorUtil::NewError(GamebaseAuth::Domain, GamebaseErrorCode::NOT_INITIALIZED);
    }

    if (AuthProcessInfo.IsSet())
    {
        return GamebaseErrorUtil::NewError(GamebaseAuth::Domain, GamebaseErrorCode::AUTH_ALREADY_IN_PROGRESS_ERROR);
    }
    
    if (!IsGamebaseLogin())
    {
        return GamebaseErrorUtil::NewError(GamebaseAuth::Domain, GamebaseErrorCode::NOT_LOGGED_IN);
    }
    
    return nullptr;
}
