#include "GamebaseErrorUtil.h"

#include "GamebaseErrorCode.h"
#include "Server/GamebaseServerErrorCode.h"
#include "Server/GamebaseServerInfo.h"

namespace GamebaseErrorUtil
{
    FGamebaseErrorPtr CreateGamebaseErrorFromTraceError(const FGamebaseHeaderResponse::FTraceError& TraceError);
}

FGamebaseErrorPtr GamebaseErrorUtil::NewError(const FGamebaseWebSocketResponse& Response, const FName& ErrorDomain)
{
    int32 ErrorCode;

    switch (Response.Header.ResultCode)
    {
        //----------------------------------------
        //  Common
        //----------------------------------------
        case GamebaseServerErrorCode::TypeMismatch:
        case GamebaseServerErrorCode::IllegalArgument:
        case GamebaseServerErrorCode::HttpMessageNotReadable:
        case GamebaseServerErrorCode::MissingServletRequestParameter:
        case GamebaseServerErrorCode::MethodArgumentNotValid:
        case GamebaseServerErrorCode::MethodArgumentTypeMismatch:
        case GamebaseServerErrorCode::InvalidAppId:
        case GamebaseServerErrorCode::InvalidAppKey:
        case GamebaseServerErrorCode::InvalidSecretKey:
            {
                ErrorCode = GamebaseErrorCode::INVALID_PARAMETER;
                break;
            }
        case GamebaseServerErrorCode::NotAuthenticated:
            {
                ErrorCode = Response.ApiId.Equals(GamebaseServerInfo::Presence::Id::Heartbeat) ?
                                GamebaseErrorCode::AUTH_INVALID_GAMEBASE_TOKEN : GamebaseErrorCode::NOT_LOGGED_IN;
                break;
            }
        case GamebaseServerErrorCode::UnknownSystem:
            {
                ErrorCode = GamebaseErrorCode::SERVER_INTERNAL_ERROR;
                break;
            }
        case GamebaseServerErrorCode::RemoteSystem:
            {
                ErrorCode = GamebaseErrorCode::SERVER_REMOTE_SYSTEM_ERROR;
                break;
            }
        case GamebaseServerErrorCode::Gateway_BannedMember:
            {
                ErrorCode = GamebaseErrorCode::BANNED_MEMBER;
                break;
            }
        case GamebaseServerErrorCode::Gateway_InvalidMember:
            {
                ErrorCode = GamebaseErrorCode::INVALID_MEMBER;
                break;
            }

        //----------------------------------------
        //  Lighthouse
        //----------------------------------------
        case GamebaseServerErrorCode::LightHouse_NotAuthenticated:
            {
                ErrorCode = Response.ApiId.Equals(GamebaseServerInfo::Presence::Id::Heartbeat) ?
                                GamebaseErrorCode::AUTH_INVALID_GAMEBASE_TOKEN : GamebaseErrorCode::NOT_LOGGED_IN;
                break;
            }
        case GamebaseServerErrorCode::LightHouse_NoSuchRequestAPI:
            {
                ErrorCode = GamebaseErrorCode::NOT_SUPPORTED;
                break;
            }
        case GamebaseServerErrorCode::LightHouse_JSONParsingError:
            {
                ErrorCode = GamebaseErrorCode::INVALID_JSON_FORMAT;
                break;
            }
        case GamebaseServerErrorCode::LightHouse_GatewayConnectionError:
            {
                ErrorCode = GamebaseErrorCode::SERVER_INTERNAL_ERROR;
                break;
            }

        //----------------------------------------
        //  Gateway
        //----------------------------------------
        case GamebaseServerErrorCode::Gateway_JSONParsingError:
            {
                ErrorCode = GamebaseErrorCode::INVALID_JSON_FORMAT;
                break;
            }
        case GamebaseServerErrorCode::Gateway_MissingRequestParameter:
        case GamebaseServerErrorCode::Gateway_InvalidAppID:
            {
                ErrorCode = GamebaseErrorCode::INVALID_PARAMETER;
                break;
            }
        case GamebaseServerErrorCode::Gateway_InvalidAccessToken:
            {
                bool bHasForceLogoutFeature = Response.ApiId.Equals(GamebaseServerInfo::Presence::Id::Heartbeat) ||
                                              Response.ApiId.Equals(GamebaseServerInfo::Gateway::Id::Withdraw) ||
                                              Response.ApiId.Equals(GamebaseServerInfo::Gateway::Id::AddMapping) ||
                                              Response.ApiId.Equals(GamebaseServerInfo::Gateway::Id::AddMappingForcibly);
                
                ErrorCode = bHasForceLogoutFeature ?
                    GamebaseErrorCode::AUTH_INVALID_GAMEBASE_TOKEN : GamebaseErrorCode::AUTH_AUTHENTICATION_SERVER_ERROR;
                break;
            }
        case GamebaseServerErrorCode::Gateway_TTLExpiredOrInvalidSessionTicketId:
            {
                ErrorCode = GamebaseErrorCode::AUTH_IDP_LOGIN_FAILED;
                break;
            }
        case GamebaseServerErrorCode::Gateway_IdpLoginFailure:
            {
                ErrorCode = GamebaseErrorCode::AUTH_IDP_LOGIN_FAILED;
                break;
            }
        case GamebaseServerErrorCode::Gateway_LoginInProgress:
            {
                ErrorCode = GamebaseErrorCode::AUTH_ALREADY_IN_PROGRESS_ERROR;
                break;
            }
        case GamebaseServerErrorCode::Gateway_ProductDataNotFound:
            {
                ErrorCode = GamebaseErrorCode::INVALID_PARAMETER;
                break;
            }
        case GamebaseServerErrorCode::Gateway_RequestAPINotFound:
            {
                ErrorCode = GamebaseErrorCode::NOT_SUPPORTED;
                break;
            }
        case GamebaseServerErrorCode::Gateway_LoggedInIdpCouldNotDeleted:
            {
                ErrorCode = GamebaseErrorCode::AUTH_REMOVE_MAPPING_LOGGED_IN_IDP;
                break;
            }
        case GamebaseServerErrorCode::Gateway_UnknownSystem:
            {
                ErrorCode = GamebaseErrorCode::SERVER_UNKNOWN_ERROR;
                break;
            }
        case GamebaseServerErrorCode::Gateway_RequestWorkerError:
        case GamebaseServerErrorCode::Gateway_QueueTimeOut:
        case GamebaseServerErrorCode::Gateway_QueueCapacityFull:
            {
                ErrorCode = GamebaseErrorCode::SERVER_INTERNAL_ERROR;
                break;
            }
        case GamebaseServerErrorCode::Gateway_GBLncSystemError:
            {
                ErrorCode = GamebaseErrorCode::LAUNCHING_SERVER_ERROR;
                break;
            }
        case GamebaseServerErrorCode::Gateway_GBIDSystemError:
            {
                ErrorCode = GamebaseErrorCode::SERVER_INTERNAL_ERROR;
                break;
            }

        //----------------------------------------
        //  Launching
        //----------------------------------------
        case GamebaseServerErrorCode::LAUNCHING_NOT_EXIST_CLIENT_ID:
            {
                ErrorCode = GamebaseErrorCode::LAUNCHING_NOT_EXIST_CLIENT_ID;
                break;
            }
        case GamebaseServerErrorCode::LAUNCHING_UNREGISTERED_APP:
            {
                ErrorCode = GamebaseErrorCode::LAUNCHING_UNREGISTERED_APP;
                break;
            }
        case GamebaseServerErrorCode::LAUNCHING_UNREGISTERED_CLIENT:
            {
                ErrorCode = GamebaseErrorCode::LAUNCHING_UNREGISTERED_CLIENT;
                break;
            }

        //----------------------------------------
        //  Member
        //----------------------------------------
        case GamebaseServerErrorCode::MEMBER_APP_ID_MISS_MATCH:
            {
                ErrorCode = GamebaseErrorCode::INVALID_PARAMETER;
                break;
            }
        case GamebaseServerErrorCode::MEMBER_USER_ID_MISS_MATCH:
        case GamebaseServerErrorCode::MEMBER_INVALID_MEMBER:
            {
                ErrorCode = GamebaseErrorCode::INVALID_MEMBER;
                break;
            }
        case GamebaseServerErrorCode::MEMBER_INVALID_AUTH:
            {
                if (Response.ApiId.Equals(GamebaseServerInfo::Gateway::Id::TokenLogin) == true)
                {
                    ErrorCode = GamebaseErrorCode::AUTH_TOKEN_LOGIN_INVALID_TOKEN_INFO;
                }
                else if (Response.ApiId.Equals(GamebaseServerInfo::Gateway::Id::IdpLogin) == true)
                {
                    ErrorCode = GamebaseErrorCode::AUTH_IDP_LOGIN_FAILED;
                }
                else if (Response.ApiId.Equals(GamebaseServerInfo::Gateway::Id::AddMapping) == true)
                {
                    ErrorCode = GamebaseErrorCode::AUTH_ADD_MAPPING_FAILED;
                }
                else if (Response.ApiId.Equals(GamebaseServerInfo::Gateway::Id::RemoveMapping) == true)
                {
                    ErrorCode = GamebaseErrorCode::AUTH_REMOVE_MAPPING_FAILED;
                }
                else if (Response.ApiId.Equals(GamebaseServerInfo::Gateway::Id::Withdraw) == true)
                {
                    ErrorCode = GamebaseErrorCode::AUTH_WITHDRAW_FAILED;
                }
                else if (Response.ApiId.Equals(GamebaseServerInfo::Gateway::Id::Logout) == true)
                {
                    ErrorCode = GamebaseErrorCode::AUTH_LOGOUT_FAILED;
                }
                else
                {
                    ErrorCode = GamebaseErrorCode::AUTH_UNKNOWN_ERROR;
                }
                break;
            }
        case GamebaseServerErrorCode::Member_CannotTemporaryWithdraw:
            {
                ErrorCode = GamebaseErrorCode::AUTH_WITHDRAW_ALREADY_TEMPORARY_WITHDRAW;
                break;
            }
        case GamebaseServerErrorCode::Member_NotTemporaryWithdraw:
            {
                ErrorCode = GamebaseErrorCode::AUTH_WITHDRAW_NOT_TEMPORARY_WITHDRAW;
                break;
            }
        case GamebaseServerErrorCode::BANNED_MEMBER_LOGIN:
        case GamebaseServerErrorCode::AUTHKEY_BELONG_TO_BANNED_MEMBER:
            {
                ErrorCode = GamebaseErrorCode::BANNED_MEMBER;
                break;
            }
        case GamebaseServerErrorCode::MEMBER_NOT_EXIST:
            {
                ErrorCode = GamebaseErrorCode::AUTH_NOT_EXIST_MEMBER;
                break;
            }
        case GamebaseServerErrorCode::MEMBER_LAST_MAPPED_IDP:
            {
                ErrorCode = GamebaseErrorCode::AUTH_REMOVE_MAPPING_LAST_MAPPED_IDP;
                break;
            }
        case GamebaseServerErrorCode::MEMBER_TRANSFERACCOUNT_ALREADY_USED:
            {
                ErrorCode = GamebaseErrorCode::AUTH_TRANSFERACCOUNT_ALREADY_USED;
                break;
            }
        case GamebaseServerErrorCode::MEMBER_ALREADY_MAPPED_MEMBER:
            {
                ErrorCode = GamebaseErrorCode::AUTH_ADD_MAPPING_ALREADY_MAPPED_TO_OTHER_MEMBER;
                break;
            }
        case GamebaseServerErrorCode::MEMBER_ALREADY_MAPPED_IDP:
            {
                ErrorCode = GamebaseErrorCode::AUTH_ADD_MAPPING_ALREADY_HAS_SAME_IDP;
                break;
            }
        case GamebaseServerErrorCode::CAN_NOT_ADD_GUEST_IDP:
            {
                ErrorCode = GamebaseErrorCode::AUTH_ADD_MAPPING_CANNOT_ADD_GUEST_IDP;
                break;
            }

        //
        // Add Mapping Forcibly
        //
        case GamebaseServerErrorCode::Member_ForcingMappingTicketNotExist:
            {
                ErrorCode = GamebaseErrorCode::AUTH_ADD_MAPPING_FORCIBLY_NOT_EXIST_KEY;
                break;
            }
        case GamebaseServerErrorCode::Member_ConsumedForcingMappingTicket:
            {
                ErrorCode = GamebaseErrorCode::AUTH_ADD_MAPPING_FORCIBLY_ALREADY_USED_KEY;
                break;
            }
        case GamebaseServerErrorCode::Member_ExpiredForcingMappingTicket:
            {
                ErrorCode = GamebaseErrorCode::AUTH_ADD_MAPPING_FORCIBLY_EXPIRED_KEY;
                break;
            }
        case GamebaseServerErrorCode::Member_IdpMissMatchForForcingMapping:
            {
                ErrorCode = GamebaseErrorCode::AUTH_ADD_MAPPING_FORCIBLY_DIFFERENT_IDP;
                break;
            }
        case GamebaseServerErrorCode::Member_AuthkeyMissMatchForForcingMapping:
            {
                ErrorCode = GamebaseErrorCode::AUTH_ADD_MAPPING_FORCIBLY_DIFFERENT_AUTHKEY;
                break;
            }
        case GamebaseServerErrorCode::GATEWAY_USER_MUST_HAVE_ONLY_GUEST_IDP:
        case GamebaseServerErrorCode::MEMBER_IS_NOT_GUEST:
            {
                ErrorCode = GamebaseErrorCode::NOT_GUEST_OR_HAS_OTHERS;
                break;
            }

        //----------------------------------------
        //  Transfer Account
        //----------------------------------------
        case GamebaseServerErrorCode::MEMBER_TRANSFERACCOUNT_EXPIRED:
            {
                ErrorCode = GamebaseErrorCode::AUTH_TRANSFERACCOUNT_EXPIRED;
                break;
            }
        case GamebaseServerErrorCode::MEMBER_BLOCK_ID:
            {
                ErrorCode = GamebaseErrorCode::AUTH_TRANSFERACCOUNT_BLOCK;
                break;
            }
        case GamebaseServerErrorCode::MEMBER_NEOID_CHECK_PASSWORD_NOT_EXIST_ID:
            {
                ErrorCode = GamebaseErrorCode::AUTH_TRANSFERACCOUNT_INVALID_ID;
                break;
            }
        case GamebaseServerErrorCode::MEMBER_NEOID_CHECK_PASSWORD_INVALID_PASSWORD:
            {
                ErrorCode = GamebaseErrorCode::AUTH_TRANSFERACCOUNT_INVALID_PASSWORD;
                break;
            }
        case GamebaseServerErrorCode::MEMBER_CONSOLE_NO_CONDITION:
            {
                ErrorCode = GamebaseErrorCode::AUTH_TRANSFERACCOUNT_CONSOLE_NO_CONDITION;
                break;
            }
        case GamebaseServerErrorCode::MEMBER_TRANSFERACCOUNT_NOT_EXIST:
            {
                ErrorCode = GamebaseErrorCode::AUTH_TRANSFERACCOUNT_NOT_EXIST;
                break;
            }
        case GamebaseServerErrorCode::MEMBER_NEOID_SIGNUP_ALREADY_EXIST_ID:
            {
                ErrorCode = GamebaseErrorCode::AUTH_TRANSFERACCOUNT_ALREADY_EXIST_ID;
                break;
            }
        case GamebaseServerErrorCode::MEMBER_SAME_REQUESTOR:
            {
                ErrorCode = GamebaseErrorCode::SAME_REQUESTOR;
                break;
            }
        case GamebaseServerErrorCode::MEMBER_ALREADY_WITHDRAWN:
            {
                ErrorCode = GamebaseErrorCode::AUTH_WITHDRAW_FAILED;
                break;
            }
        case GamebaseServerErrorCode::MEMBER_NEOID_SIGNUP_UNKNOWN:
        case GamebaseServerErrorCode::MEMBER_NEOID_CHECK_PASSWORD_UNKNOWN:
        case GamebaseServerErrorCode::MEMBER_NEOID_QUERY:
        case GamebaseServerErrorCode::MEMBER_NEOID_QUERY_INFO:
        case GamebaseServerErrorCode::MEMBER_NEOID_CREATE_PASSWORD:
        case GamebaseServerErrorCode::MEMBER_NEOID_CONNECTION:
            {
                ErrorCode = GamebaseErrorCode::SERVER_INTERNAL_ERROR;
                break;
            }
            
        case GamebaseServerErrorCode::CANNOT_FIND_DEPLOY_TERMS:
            {
                ErrorCode = GamebaseErrorCode::UI_TERMS_NOT_EXIST_IN_CONSOLE;
                break;
            }
        case GamebaseServerErrorCode::CANNOT_FIND_DEFAULT_TERMS:
            {
                ErrorCode = GamebaseErrorCode::UI_TERMS_NOT_EXIST_FOR_DEVICE_COUNTRY;
                break;
            }
        case GamebaseServerErrorCode::ILLEGAL_TERMS_SEQUENCE_NUMBER:
            {
                ErrorCode = GamebaseErrorCode::UI_TERMS_UNREGISTERED_SEQ;
                break;
            }
        case GamebaseServerErrorCode::ILLEGAL_TERMS_CONTENT_SEQUENCE_NUMBER:
            {
                ErrorCode = GamebaseErrorCode::UI_TERMS_UNREGISTERED_SEQ;
                break;
            }

        default:
            {
                ErrorCode = GamebaseErrorCode::UNKNOWN_ERROR;
                break;
            }
    }

    FGamebaseErrorPtr ServerError;
    if (Response.Header.TraceError.IsSet())
    {
        ServerError = MakeShared<FGamebaseError, ESPMode::ThreadSafe>(
            Response.Header.ResultCode,
            Response.Header.ResultMessage,
            Response.Header.TraceError.GetValue().ProductId,
            *CreateGamebaseErrorFromTraceError(Response.Header.TraceError.GetValue())
        );
    }
    else
    {
        ServerError = MakeShared<FGamebaseError, ESPMode::ThreadSafe>(
            Response.Header.ResultCode,
            Response.Header.ResultMessage,
            FString(TEXT("TraceErrorNotExist"))         //TODO: Android에서만 TraceErrorNotExist로 지정하고 있음
        );
    }
    
    FGamebaseErrorPtr ClientError = MakeShared<FGamebaseError, ESPMode::ThreadSafe>(
        ErrorCode,
        GetErrorMessage(ErrorCode),
        ErrorDomain,
        *ServerError,
        Response.Header.TransactionId);
    
    return ClientError;
}

FGamebaseErrorPtr GamebaseErrorUtil::CreateGamebaseErrorFromTraceError(const FGamebaseHeaderResponse::FTraceError& TraceError)
{
    FGamebaseErrorPtr Error = MakeShared<FGamebaseError, ESPMode::ThreadSafe>(
        TraceError.ResultCode,
        TraceError.ResultMessage,
        FString(TraceError.ThrowPoint)
    );

    if (!TraceError.InnerTraceError.IsEmpty())
    {
        FGamebaseHeaderResponse::FTraceError InnerTraceError;
        if (InnerTraceError.FromJson(TraceError.InnerTraceError))
        {
            Error->Error = CreateGamebaseErrorFromTraceError(InnerTraceError);
        }
        else
        {
            Error->Error = MakeShared<FGamebaseError, ESPMode::ThreadSafe>(
                GamebaseErrorCode::INVALID_JSON_FORMAT,
                TEXT("Failed to parse inner trace error (FromJson)"),
                FString(TEXT("JSON_PARSE_ERROR"))
            );
        }
    }

    return Error;
}
