#include "GamebaseStandaloneAuthAdapter.h"

#include "GamebaseAuthConstants.h"
#include "GamebaseAuthProviderCredential.h"
#include "GamebaseStandaloneAuthProviderCredential.h"
#include "GamebaseStandaloneAuthTypes.h"

#include "Auth/GamebaseAuthClient.h"
#include "Auth/GamebaseAuthMemberLoginSystem.h"
#include "Auth/GamebaseAuthMemberLoginWithExternalBrowser.h"
#include "Auth/GamebaseAuthMemberLoginWithWebView.h"
#include "Auth/GamebaseSessionTicketClient.h"

namespace GamebaseAuth
{
    namespace AuthMemberLoginFactory
    {
        FGamebaseLoginSystemPtr MakeSystem(const EGamebaseStandaloneAuthMemberWebType Type)
        {
            switch (Type)
            {
                case EGamebaseStandaloneAuthMemberWebType::WebView:
                    return MakeShared<FGamebaseAuthMemberLoginWithWebView, ESPMode::ThreadSafe>();
                case EGamebaseStandaloneAuthMemberWebType::ExternalBrowser:
                    return MakeShared<FGamebaseAuthMemberLoginWithExternalBrowser, ESPMode::ThreadSafe>();
            }

            checkNoEntry();
            return nullptr;
        }
    }

    namespace IdpInfoKey
    {
        const FName Session("session");
        const FName AuthorizationCode("authorizationCode");
        const FName SubCode("subCode");
        const FName RedirectUri("redirectUri");
    }
    
    constexpr int32 LoginTimeout = 180;
}

void UGamebaseStandaloneAuthAdapter::Initialize(const FGamebaseAuthConfiguration& Configuration)
{
    AuthConfiguration = Configuration;
}

void UGamebaseStandaloneAuthAdapter::CancelLogin(const FCancelLoginCallback& Callback)
{
    Callback(FGamebaseErrorResult(FGamebaseError(
        GamebaseErrorCode::AUTH_LOGIN_CANCEL_FAILED,
        "Login cancel is not supported in this adapter",
        GamebaseAuth::Domain)));
}

void UGamebaseStandaloneAuthMemberWebAdapter::Initialize(const FGamebaseAuthConfiguration& Configuration)
{
    Super::Initialize(Configuration);
}

void UGamebaseStandaloneAuthMemberWebAdapter::Login(const FGamebaseVariantMap& AdditionalInfo, const FLoginCallback& Callback)
{
    using namespace GamebaseAuth;
    
    if (AuthConfiguration.SessionTicketClient.IsValid())
    {
        const auto SessionTicketClient = AuthConfiguration.SessionTicketClient.Pin();
        if (SessionTicketClient.IsValid())
        {
            SessionTicketClient->RequestIssueSessionTicket(LoginTimeout, 
                [this, AdditionalInfo, Callback](const FGamebaseSessionTicketResult& Result)
                {
                    if (Result.IsError())
                    {
                        Callback(FGamebaseAuthCredentialsResult(Result.GetErrorValue()));
                        return;
                    }
                    
                    AuthConfiguration.SessionTicketId = Result.GetOkValue();
                    
                    PerformLogin(AdditionalInfo, Callback);
                });
        }
        else
        {
            PerformLogin(AdditionalInfo, Callback);
        }
    }
    else
    {
        PerformLogin(AdditionalInfo, Callback);
    }
}

void UGamebaseStandaloneAuthMemberWebAdapter::CancelLogin(const FCancelLoginCallback& Callback)
{
    using namespace GamebaseAuth;

    if (LoginSystem.IsValid() == false)
    {
        Callback(FGamebaseErrorResult(FGamebaseError(
            GamebaseErrorCode::AUTH_LOGIN_CANCEL_FAILED,
            "Login system is not initialized.",
            GamebaseAuth::Domain)));
        return;
    }

    LoginSystem->CancelLogin([Callback, this](const FGamebaseErrorResult& Result)
    {
        Callback(Result);
        LoginSystem.Reset();
    });
}

void UGamebaseStandaloneAuthMemberWebAdapter::PerformLogin(const FGamebaseVariantMap& AdditionalInfo, const FLoginCallback& Callback)
{
    using namespace GamebaseAuth;

    LoginSystem = AuthMemberLoginFactory::MakeSystem(GetViewType());

    AuthConfiguration.UserAdditionalInfo = FGamebaseVariantMap();
    AuthConfiguration.UserAdditionalInfo.Append(AdditionalInfo);

    LoginSystem->Login(AuthConfiguration, GetProviderSpecificQueries(), [this, Callback](const FGamebaseAuthLoginResult& Response)
    {
        if (Response.IsError())
        {
            Callback(FGamebaseAuthCredentialsResult(Response.GetErrorValue()));
            LoginSystem.Reset();
            return;
        }

        FGamebaseAuthCredentials Credentials;

        const FGamebaseAuthMemberResponseData& AuthorizationCode = Response.GetOkValue();
        if (AuthorizationCode.Session.IsSet())
        {
            Credentials.Add(GamebaseAuthProviderCredential::MemberOneTimeSession,AuthorizationCode.Session.GetValue());
        }
        if (AuthorizationCode.Token.IsSet())
        {
            Credentials.Add(GamebaseAuthProviderCredential::AccessToken,AuthorizationCode.Token.GetValue());
        }
        if (AuthorizationCode.Code.IsSet())
        {
            Credentials.Add(GamebaseAuthProviderCredential::AuthorizationCode,AuthorizationCode.Code.GetValue());
        }

        Credentials.Append(GetProviderSpecificCredential());

        Callback(FGamebaseAuthCredentialsResult(Credentials));
        LoginSystem.Reset();
    });
}

void UGamebaseStandaloneAuthMemberWebAdapter::Logout(const FLogoutCallback& Callback)
{
    Callback();
}

void UGamebaseStandaloneAuthMemberWebAdapter::Withdraw(const FWithdrawCallback& Callback)
{
    Callback();
}

FString UGamebaseStandaloneAuthMemberWebAdapter::GetProviderName() const
{
    return {};
}

FString UGamebaseStandaloneAuthMemberWebAdapter::GetUserID() const
{
    return {};
}

FString UGamebaseStandaloneAuthMemberWebAdapter::GetAccessToken() const
{
    return {};
}

FGamebaseAuthProviderProfilePtr UGamebaseStandaloneAuthMemberWebAdapter::GetProfile() const
{
    return {};
}

FGamebaseVariantMap UGamebaseStandaloneAuthMemberWebAdapter::GetProviderSpecificQueries() const
{
    return {};
}

FGamebaseVariantMap UGamebaseStandaloneAuthMemberWebAdapter::GetProviderSpecificCredential() const
{
    return {};
}

EGamebaseStandaloneAuthMemberWebType UGamebaseStandaloneAuthMemberWebAdapter::GetViewType() const
{
    return EGamebaseStandaloneAuthMemberWebType::ExternalBrowser;
}
