#include "GamebaseAuthClient.h"

#include "GamebaseAuthExtras.h"
#include "GamebaseErrorExtras.h"
#include "GamebaseErrorUtil.h"
#include "GamebaseStandaloneAuthProviderCredential.h"
#include "GamebaseStandaloneContact.h"
#include "GamebaseStandaloneShortTermTicket.h"
#include "GamebaseSystemUtils.h"
#include "GamebaseWebSocket.h"
#include "Auth/GamebaseAuthConstants.h"
#include "Auth/GamebaseAuthRequest.h"
#include "Auth/GamebaseAuthResponse.h"
#include "Helper/GamebaseSystemPopupHelper.h"
#include "Server/GamebaseServerErrorCode.h"
#include "Server/GamebaseServerInfo.h"
#include "ShortTermTicket/GamebaseShortTermTicketConstants.h"

void FGamebaseAuthClient::IdpLogin(
    const FString& ProviderName,
    const bool bIsAutoLogin,
    const FGamebaseAuthCredentials& Credentials,
    const FAuthTokenResultCallback& Callback)
{
    GamebaseAuth::FLoginParameters Parameters;
    Parameters.AppId = InternalData->GetAppId();
    
    const auto LaunchingInfo = InternalData->GetLaunchingInfo();

    GamebaseAuth::FLoginPayload Payload;
    Payload.IdPInfo.IdPCode = ProviderName;
    Payload.IdPInfo.ClientId = LaunchingInfo->Launching.App.Idp[ProviderName].ClientId;
    
    Credentials.SetOptionalIfExists(GamebaseAuthProviderCredential::SubCode, Payload.IdPInfo.SubCode);
    Credentials.SetOptionalIfExists(GamebaseAuthProviderCredential::AccessToken, Payload.IdPInfo.AccessToken);
    Credentials.SetOptionalIfExists(GamebaseAuthProviderCredential::MemberOneTimeSession, Payload.IdPInfo.Session);
    Credentials.SetOptionalIfExists(GamebaseAuthProviderCredential::RedirectUri, Payload.IdPInfo.RedirectUri);
    Credentials.SetOptionalIfExists(GamebaseAuthProviderCredential::AuthorizationCode, Payload.IdPInfo.AuthorizationCode);
    Credentials.SetOptionalIfExists(GamebaseAuthProviderCredential::AuthorizationProtocol, Payload.IdPInfo.AuthorizationProtocol);
    Credentials.SetOptionalIfExists(GamebaseAuthProviderCredential::ExtraParams, Payload.IdPInfo.ExtraParams);

    if (bIsAutoLogin)
    {
        Payload.IdPInfo.ExtraParams = FGamebaseVariantMap();
        Payload.IdPInfo.ExtraParams->Add(GamebaseAuthExtras::UseIntrospection, false);
    }
    
    Payload.Member.ClientVersion = InternalData->GetAppVersion();
    Payload.Member.SDKVersion = GamebaseSystemUtils::GetPluginVersion();
    Payload.Member.DeviceKey = GamebaseSystemUtils::GetDeviceKey();
    Payload.Member.DeviceModel = GamebaseSystemUtils::GetDeviceModel();
    Payload.Member.OSCode = GamebaseSystemUtils::GetPlatform();
    Payload.Member.OSVersion = GamebaseSystemUtils::GetOSVersion();
    Payload.Member.DeviceLanguage = GamebaseSystemUtils::GetDeviceLanguage();
    Payload.Member.DisplayLanguage = InternalData->GetDisplayLanguageCode();
    Payload.Member.DeviceCountryCode = GamebaseSystemUtils::GetCountryCode();
    Payload.Member.Network = GamebaseSystemUtils::GetNetworkConnectionType();
    Payload.Member.UsimCountryCode = GamebaseSystemUtils::GetSimCode();
    Payload.Member.StoreCode = InternalData->GetStoreCode();
    Payload.Member.UUID = InternalData->GetUuid();
    
    WebSocket->Request(
        GamebaseServerInfo::Gateway::ProductId,
        GamebaseServerInfo::Gateway::Id::IdpLogin,
        GamebaseServerInfo::ApiVersion,
        Parameters,
        Payload,
        [this, Callback](const FGamebaseWebSocketResponseResult& Response)
        {
            if (Response.IsError())
            {
                Callback(FGamebaseAuthTokenResult::Failure(Response.GetErrorValue()));
                return;
            }

            const auto& ResponseSuccessData = Response.GetOkValue();
            if (ResponseSuccessData.Header.bIsSuccessful == false)
            {
                if (ResponseSuccessData.Header.ResultCode == GamebaseServerErrorCode::BANNED_MEMBER_LOGIN ||
                    ResponseSuccessData.Header.ResultCode == GamebaseServerErrorCode::AUTHKEY_BELONG_TO_BANNED_MEMBER)
                {
                    OnLoginFailedDueToBan(ResponseSuccessData, Callback);
                }
                else
                {
                    const FGamebaseErrorPtr Error = GamebaseErrorUtil::NewError(ResponseSuccessData, GamebaseAuth::Domain);
                    Callback(FGamebaseAuthTokenResult::Failure(*Error));
                }
                return;
            }

            FGamebaseAuthToken AuthToken;
            AuthToken.FromJson(ResponseSuccessData.OriginData);
            Callback(FGamebaseAuthTokenResult::Success(AuthToken));
        });
}

void FGamebaseAuthClient::TokenLogin(const FString& ProviderName, const FString& AccessToken, const FGamebaseAuthCredentials& Credentials, const FAuthTokenResultCallback& Callback)
{
    GamebaseAuth::FLoginParameters Parameters;
    Parameters.AppId = InternalData->GetAppId();
    
    GamebaseAuth::FLoginPayload Payload;
    Payload.TokenInfo.IdPCode = ProviderName;
    Payload.TokenInfo.AccessToken = AccessToken;
    Credentials.SetOptionalIfExists(GamebaseAuthProviderCredential::SubCode, Payload.TokenInfo.SubCode);
    Credentials.SetOptionalIfExists(GamebaseAuthProviderCredential::ExtraParams, Payload.TokenInfo.ExtraParams);
    
    Payload.Member.ClientVersion = InternalData->GetAppVersion();
    Payload.Member.SDKVersion = GamebaseSystemUtils::GetPluginVersion();
    Payload.Member.DeviceKey = GamebaseSystemUtils::GetDeviceKey();
    Payload.Member.DeviceModel = GamebaseSystemUtils::GetDeviceModel();
    Payload.Member.OSCode = GamebaseSystemUtils::GetPlatform();
    Payload.Member.OSVersion = GamebaseSystemUtils::GetOSVersion();
    Payload.Member.DeviceLanguage = GamebaseSystemUtils::GetDeviceLanguage();
    Payload.Member.DisplayLanguage = InternalData->GetDisplayLanguageCode();
    Payload.Member.DeviceCountryCode = GamebaseSystemUtils::GetCountryCode();
    Payload.Member.Network = GamebaseSystemUtils::GetNetworkConnectionType();
    Payload.Member.UsimCountryCode = GamebaseSystemUtils::GetSimCode();
    Payload.Member.StoreCode = InternalData->GetStoreCode();
    Payload.Member.UUID = InternalData->GetUuid();
    
    WebSocket->Request(
        GamebaseServerInfo::Gateway::ProductId,
        GamebaseServerInfo::Gateway::Id::TokenLogin,
        GamebaseServerInfo::ApiVersion,
        Parameters,
        Payload,
        [this, Callback](const FGamebaseWebSocketResponseResult& Response)
        {
            if (Response.IsError())
            {
                Callback(FGamebaseAuthTokenResult::Failure(Response.GetErrorValue()));
                return;
            }
                
            const auto& ResponseSuccessData = Response.GetOkValue();
            if (ResponseSuccessData.Header.bIsSuccessful == false)
            {
                if (ResponseSuccessData.Header.ResultCode == GamebaseServerErrorCode::BANNED_MEMBER_LOGIN ||
                    ResponseSuccessData.Header.ResultCode == GamebaseServerErrorCode::AUTHKEY_BELONG_TO_BANNED_MEMBER)
                {
                    OnLoginFailedDueToBan(ResponseSuccessData, Callback);
                }
                else
                {
                    const FGamebaseErrorPtr Error = GamebaseErrorUtil::NewError(ResponseSuccessData, GamebaseAuth::Domain);
                    Callback(FGamebaseAuthTokenResult::Failure(*Error));
                }
                return;
            }

            FGamebaseAuthToken AuthToken;
            AuthToken.FromJson(ResponseSuccessData.OriginData);
            Callback(FGamebaseAuthTokenResult::Success(AuthToken));
        });
}

void FGamebaseAuthClient::BindIdPToken(const FString& ProviderName, const FGamebaseAuthCredentials& Credentials, const FVoidResultCallback& Callback)
{
    const auto AuthToken = InternalData->GetAuthToken();
    
    GamebaseAuth::FBindIdPTokenParameters Parameters;
    Parameters.UserId = InternalData->GetUserId();
    
    GamebaseAuth::FBindIdPTokenPayload Payload;
    Payload.IdPInfo.IdPCode = ProviderName;
    Payload.IdPInfo.Session = Credentials.GetValueOrDefault(GamebaseAuthProviderCredential::MemberOneTimeSession, FString());
    Payload.TokenInfo.AccessToken = AuthToken->Token.AccessToken;

    WebSocket->Request(
        GamebaseServerInfo::Gateway::ProductId,
        GamebaseServerInfo::Gateway::Id::BindIdPToken,
        GamebaseServerInfo::ApiVersion,
        Parameters,
        Payload,
        [this, Callback](const FGamebaseWebSocketResponseResult& Response)
        {
            if (Response.IsError())
            {
                Callback(FGamebaseErrorResult(Response.GetErrorValue()));
                return;
            }
        
            const auto& ResponseSuccessData = Response.GetOkValue();
            if (ResponseSuccessData.Header.bIsSuccessful == false)
            {
                const FGamebaseErrorPtr Error = GamebaseErrorUtil::NewError(ResponseSuccessData, GamebaseAuth::Domain);
                Callback(FGamebaseErrorResult(*Error));
                return;
            }
            
            Callback(FGamebaseErrorResult());
        });
}

void FGamebaseAuthClient::Logout(const FVoidResultCallback& Callback)
{
    GamebaseAuth::FLogoutParameters Parameters;
    Parameters.AppId = InternalData->GetAppId();
    Parameters.UserId = InternalData->GetUserId();
    Parameters.AccessToken = InternalData->GetAccessToken();

    WebSocket->Request(
        GamebaseServerInfo::Gateway::ProductId, 
        GamebaseServerInfo::Gateway::Id::Logout, 
        GamebaseServerInfo::ApiVersion, 
        Parameters,
        [Callback](const FGamebaseWebSocketResponseResult& Response)
        {
            if (Response.IsOk())
            {
                Callback(FGamebaseErrorResult());
            }
            else
            {
                Callback(FGamebaseErrorResult(Response.GetErrorValue()));
            }
        });
}

void FGamebaseAuthClient::Withdraw(const FVoidResultCallback& Callback)
{
    GamebaseAuth::FWithdrawParameters Parameters;
    Parameters.UserId = InternalData->GetUserId();
    
    WebSocket->Request(
        GamebaseServerInfo::Gateway::ProductId, 
        GamebaseServerInfo::Gateway::Id::Withdraw, 
        GamebaseServerInfo::ApiVersion, 
        Parameters,
        [Callback](const FGamebaseWebSocketResponseResult& Response)
        {
            if (Response.IsOk())
            {
                const auto& ResponseSuccessData = Response.GetOkValue();

                if (ResponseSuccessData.Header.bIsSuccessful)
                {
                    Callback(FGamebaseErrorResult());
                }
                else
                {
                    const FGamebaseErrorPtr RecvError = GamebaseErrorUtil::NewError(ResponseSuccessData, GamebaseAuth::Domain);
                    Callback(FGamebaseErrorResult(*RecvError));
                }
            }
            else
            {
                Callback(FGamebaseErrorResult(Response.GetErrorValue()));
            }
        });
}

void FGamebaseAuthClient::TemporaryWithdrawal(const FTemporaryWithdrawalResultCallback& Callback)
{
    GamebaseAuth::FWithdrawParameters Parameters;
    Parameters.UserId = InternalData->GetUserId();
    
    WebSocket->Request(
        GamebaseServerInfo::Gateway::ProductId, 
        GamebaseServerInfo::Gateway::Id::TemporaryWithdrawal, 
        GamebaseServerInfo::ApiVersion, 
        Parameters,
        [Callback](const FGamebaseWebSocketResponseResult& Response)
        {
            if (Response.IsOk())
            {
                const auto& ResponseSuccessData = Response.GetOkValue();

                if (ResponseSuccessData.Header.bIsSuccessful)
                {
                    GamebaseAuth::FTemporaryWithdrawalInfoResponse ResponseVo;
                    ResponseVo.FromJson(ResponseSuccessData.OriginData);
            
                    FGamebaseTemporaryWithdrawalInfo TemporaryWithdrawalInfo;
                    TemporaryWithdrawalInfo.GracePeriodDate = ResponseVo.Member.TemporaryWithdrawal.GracePeriodDate;
                    
                    Callback(FGamebaseTemporaryWithdrawalResult::Success(TemporaryWithdrawalInfo));
                }
                else
                {
                    const FGamebaseErrorPtr RecvError = GamebaseErrorUtil::NewError(ResponseSuccessData, GamebaseAuth::Domain);
                    Callback(FGamebaseTemporaryWithdrawalResult::Failure(*RecvError));
                }
            }
            else
            {
                Callback(FGamebaseTemporaryWithdrawalResult::Failure(Response.GetErrorValue()));
            }
        });
}

void FGamebaseAuthClient::CancelTemporaryWithdrawal(const FVoidResultCallback& Callback)
{
    GamebaseAuth::FWithdrawParameters Parameters;
    Parameters.UserId = InternalData->GetUserId();
    
    WebSocket->Request(
        GamebaseServerInfo::Gateway::ProductId, 
        GamebaseServerInfo::Gateway::Id::CancelTemporaryWithdrawal, 
        GamebaseServerInfo::ApiVersion, 
        Parameters,
        [Callback](const FGamebaseWebSocketResponseResult& Response)
        {
            if (Response.IsOk())
            {
                const auto& ResponseSuccessData = Response.GetOkValue();

                if (ResponseSuccessData.Header.bIsSuccessful)
                {
                    Callback(FGamebaseErrorResult());
                }
                else
                {
                    const FGamebaseErrorPtr RecvError = GamebaseErrorUtil::NewError(ResponseSuccessData, GamebaseAuth::Domain);
                    Callback(FGamebaseErrorResult(*RecvError));
                }
            }
            else
            {
                Callback(FGamebaseErrorResult(Response.GetErrorValue()));
            }
        });
}

void FGamebaseAuthClient::AddMapping(
    const FString& UserId,
    const FString& ProviderName,
    const FGamebaseAuthCredentials& Credentials,
    const FAuthTokenResultCallback& Callback)
{
    GamebaseAuth::FAddMappingParameters Parameters;
    Parameters.UserId = UserId;
    Parameters.bIsForcing = false;

    const auto LaunchingInfo = InternalData->GetLaunchingInfo();

    GamebaseAuth::FLoginPayload Payload;
    Payload.IdPInfo.IdPCode = ProviderName;
    Payload.IdPInfo.ClientId = LaunchingInfo->Launching.App.Idp[ProviderName].ClientId;
    
    Credentials.SetOptionalIfExists(GamebaseAuthProviderCredential::SubCode, Payload.IdPInfo.SubCode);
    Credentials.SetOptionalIfExists(GamebaseAuthProviderCredential::AccessToken, Payload.IdPInfo.AccessToken);
    Credentials.SetOptionalIfExists(GamebaseAuthProviderCredential::MemberOneTimeSession, Payload.IdPInfo.Session);
    Credentials.SetOptionalIfExists(GamebaseAuthProviderCredential::RedirectUri, Payload.IdPInfo.RedirectUri);
    Credentials.SetOptionalIfExists(GamebaseAuthProviderCredential::AuthorizationCode, Payload.IdPInfo.AuthorizationCode);
    Credentials.SetOptionalIfExists(GamebaseAuthProviderCredential::AuthorizationProtocol, Payload.IdPInfo.AuthorizationProtocol);
    Credentials.SetOptionalIfExists(GamebaseAuthProviderCredential::ExtraParams, Payload.IdPInfo.ExtraParams);
    
    WebSocket->Request(
        GamebaseServerInfo::Gateway::ProductId,
        GamebaseServerInfo::Gateway::Id::AddMapping,
        GamebaseServerInfo::ApiVersion,
        Parameters,
        Payload,
        [Callback](const FGamebaseWebSocketResponseResult& Response)
        {
            if (Response.IsError())
            {
                Callback(FGamebaseAuthTokenResult::Failure(Response.GetErrorValue()));
                return;
            }

            const auto& ResponseSuccessData = Response.GetOkValue();
            if (ResponseSuccessData.Header.bIsSuccessful == false)
            {
                const FGamebaseErrorPtr Error = GamebaseErrorUtil::NewError(ResponseSuccessData, GamebaseAuth::Domain);
                if (Error->Code == GamebaseErrorCode::AUTH_ADD_MAPPING_ALREADY_MAPPED_TO_OTHER_MEMBER)
                {
                    GamebaseAuth::FMappingErrorExtras ErrorExtras;
                    ErrorExtras.FromJson(ResponseSuccessData.ErrorExtras);

                    if (ErrorExtras.ForcingMappingTicket.IsEmpty() == false)
                    {
                        Error->AddExtra(GamebaseErrorExtras::ForcingMappingTicket, ErrorExtras.ForcingMappingTicket);
                    }
                }
                Callback(FGamebaseAuthTokenResult::Failure(*Error));
                return;
            }

            FGamebaseAuthToken AuthToken;
            AuthToken.FromJson(ResponseSuccessData.OriginData);
            
            Callback(FGamebaseAuthTokenResult::Success(AuthToken));
        });
}

void FGamebaseAuthClient::AddMappingForcibly(
    const FString& UserId,
    const FGamebaseForcingMappingTicket& ForcingMappingTicket,
    const FAuthTokenResultCallback& Callback)
{
    GamebaseAuth::FAddMappingForciblyParameters Parameters;
    Parameters.UserId = UserId;
    Parameters.ForcingMappingKey = ForcingMappingTicket.ForcingMappingKey;
    Parameters.MappedUserId = ForcingMappingTicket.MappedUserId;

    GamebaseAuth::FAddMappingForciblyPayload Payload;
    Payload.TokenInfo.AccessToken = ForcingMappingTicket.AccessToken;
    Payload.TokenInfo.IdPCode = ForcingMappingTicket.IdPCode;

    WebSocket->Request(
        GamebaseServerInfo::Gateway::ProductId,
        GamebaseServerInfo::Gateway::Id::AddMappingForcibly,
        GamebaseServerInfo::ApiVersion,
        Parameters,
        Payload,
        [Callback](const FGamebaseWebSocketResponseResult& Response)
        {
            if (Response.IsError())
            {
                Callback(FGamebaseAuthTokenResult::Failure(Response.GetErrorValue()));
                return;
            }

            const auto& ResponseSuccessData = Response.GetOkValue();
            if (ResponseSuccessData.Header.bIsSuccessful == false)
            {
                const FGamebaseErrorPtr Error = GamebaseErrorUtil::NewError(ResponseSuccessData, GamebaseAuth::Domain);
                Callback(FGamebaseAuthTokenResult::Failure(*Error));
                return;
            }

            FGamebaseAuthToken AuthToken;
            AuthToken.FromJson(ResponseSuccessData.OriginData);
            Callback(FGamebaseAuthTokenResult::Success(AuthToken));
        });
    
    
}

void FGamebaseAuthClient::RemoveMapping(
    const FString& ProviderName,
    const FString& UserId,
    const FString& AccessToken,
    const FAuthTokenResultCallback& Callback)
{
    GamebaseAuth::FRemoveMappingParameters Parameters;
    Parameters.UserId = UserId;
    Parameters.ProviderName = ProviderName;
    Parameters.AccessToken = AccessToken;

    WebSocket->Request(
        GamebaseServerInfo::Gateway::ProductId,
        GamebaseServerInfo::Gateway::Id::RemoveMapping,
        GamebaseServerInfo::ApiVersion,
        Parameters,
        [Callback](const FGamebaseWebSocketResponseResult& Response)
        {
            if (Response.IsError())
            {
                Callback(FGamebaseAuthTokenResult::Failure(Response.GetErrorValue()));
                return;
            }

            const auto& ResponseSuccessData = Response.GetOkValue();
            if (ResponseSuccessData.Header.bIsSuccessful == false)
            {
                const FGamebaseErrorPtr Error = GamebaseErrorUtil::NewError(ResponseSuccessData, GamebaseAuth::Domain);
                Callback(FGamebaseAuthTokenResult::Failure(*Error));
                return;
            }

            FGamebaseAuthToken AuthToken;
            AuthToken.FromJson(ResponseSuccessData.OriginData);
            Callback(FGamebaseAuthTokenResult::Success(AuthToken));
        });
}


void FGamebaseAuthClient::OnLoginFailedDueToBan(const FGamebaseWebSocketResponse& ResponseSuccessData, const FAuthTokenResultCallback& Callback) const
{
    const FGamebaseErrorPtr Error = GamebaseErrorUtil::NewError(ResponseSuccessData, GamebaseAuth::Domain);
    
    if (ResponseSuccessData.Header.ResultCode == GamebaseServerErrorCode::BANNED_MEMBER_LOGIN ||
        ResponseSuccessData.Header.ResultCode == GamebaseServerErrorCode::AUTHKEY_BELONG_TO_BANNED_MEMBER)
    {
        GamebaseAuth::FErrorExtras ErrorExtras;
        ErrorExtras.FromJson(ResponseSuccessData.ErrorExtras);

        if (ErrorExtras.Ban.IsEmpty())
        {
            Callback(FGamebaseAuthTokenResult::Failure(*Error));
            return;
        }

        const auto LaunchingInfo = InternalData->GetLaunchingInfo();
        if (LaunchingInfo->Launching.App.CustomerService.IsSet() == false)
        {
            Callback(FGamebaseAuthTokenResult::Failure(*Error));
            return;
        }

        FGamebaseBanInfo BanInfo;
        BanInfo.FromJson(ErrorExtras.Ban);
        BanInfo.CsInfo = LaunchingInfo->Launching.App.CustomerService->AccessInfo;
    
        RequestBanContactUrl(BanInfo.UserId, [this, BanInfo, Error, Callback](const FString& CsUrl) mutable
        {
            if (CsUrl.IsEmpty() == false)
            {
                BanInfo.CsUrl = CsUrl;    
            }
            
            Error->AddExtra(GamebaseErrorExtras::BanInfo, BanInfo.ToJson());
                    
            if (InternalData->GetConfiguration().bEnableBanPopup)
            {
                GamebaseSystemPopup::ShowBan(
                    *InternalData->GetDisplayLanguage(),
                    BanInfo,
                    [Callback, Error]
                    {
                        Callback(FGamebaseAuthTokenResult::Failure(*Error));
                    });
            }
            else
            {
                Callback(FGamebaseAuthTokenResult::Failure(*Error));
            }
        });
    }
    else
    {
        Callback(FGamebaseAuthTokenResult::Failure(*Error));
    }
}

void FGamebaseAuthClient::RequestBanContactUrl(const FString& UserId, const TFunction<void(FString)>& Callback) const
{
    if (!Contact.IsValid())
    {
        Callback(TEXT(""));
        return;
    }

    const auto LaunchingInfo = InternalData->GetLaunchingInfo();
    if (LaunchingInfo->Launching.App.CustomerService.IsSet())
    {
        const auto CustomerService = LaunchingInfo->Launching.App.CustomerService.GetValue();

        GamebaseShortTermTicket::FIssueParameters Parameter = GamebaseShortTermTicket::FIssueParameters();
        Parameter.UserId = UserId;
        Parameter.Purpose = GamebaseShortTermTicket::Purpose::OpenContactForBannedUser;
        Parameter.ExpiresIn = GamebaseShortTermTicket::ExpiresIn;
        
        Contact->RequestContactURL(Parameter, TOptional<FGamebaseContactConfiguration>(), FGamebaseContactUrlDelegate::CreateLambda([=](const FString& Url, const FGamebaseError* Error)
        {
            if (Error == nullptr)
            {
                Callback(Url);
            }
            else
            {
                Callback(TEXT(""));
            }          
        }));
    }
    else
    {
        Callback(TEXT(""));
    }
}