#pragma once

#include "GamebaseInternalResult.h"
#include "Async/Future.h"
#include "Templates/Function.h"
#include "Templates/SharedPointer.h"

template <typename T, typename TCallback>
class TGamebaseTaskRunner
    : public TSharedFromThis<TGamebaseTaskRunner<T, TCallback>>
{
public:
    using FResultType = FGamebaseInternalResult<T>;
    using FTaskStep = TFunction<TFuture<FResultType>()>;
    using TCallbackType = TCallback;

    void Run()
    {
        RunStep(0);
    }

protected:
    explicit TGamebaseTaskRunner(TArray<FTaskStep>&& Steps, TCallbackType&& Callback)
        : Steps(MoveTemp(Steps))
        , StepIndex(0)
        , Callback(MoveTemp(Callback))
    {}

    virtual ~TGamebaseTaskRunner() = default;
    
    virtual void OnStepCompleted(const FResultType& Result) = 0;
    
    void RunStep(const int32 Index)
    {
        ensureMsgf(Steps.IsValidIndex(Index), TEXT("Invalid step index encountered in RunStep: %d"), Index);

        Steps[Index]().Next([Self = this->SharedThis(this)](const FResultType& Result)
        {
            Self->OnStepCompleted(Result);
        });
    }
    
protected:
    TArray<FTaskStep> Steps;
    int32 StepIndex;
    TCallbackType Callback;
};


template <typename T>
using TGamebaseStopOnSuccessRunnerCallback = TUniqueFunction<void(const FGamebaseInternalResult<T>&, const TArray<FGamebaseError>&)>;

template <typename T>
class TGamebaseStopOnSuccessRunner final
    : public TGamebaseTaskRunner<T, TGamebaseStopOnSuccessRunnerCallback<T>>
{
public:
    using FBase = TGamebaseTaskRunner<T, TGamebaseStopOnSuccessRunnerCallback<T>>;
    using typename FBase::FResultType;
    using typename FBase::FTaskStep;
    using typename FBase::TCallbackType;

    using FErrorGenerator = TFunction<FGamebaseError(const TArray<FGamebaseError>&)>;

    explicit TGamebaseStopOnSuccessRunner(TArray<FTaskStep> Steps, TCallbackType&& Callback, FErrorGenerator&& ErrorGenerator)
        : FBase(MoveTemp(Steps), MoveTemp(Callback))
        , ErrorGenerator(MoveTemp(ErrorGenerator))
    {}

protected:
    virtual void OnStepCompleted(const FResultType& Result) override
    {
        if (!Result.IsError())
        {
            this->Callback(FResultType(Result), FailedResults);
            return;
        }

        FailedResults.Add(Result.GetErrorValue());

        if (this->Steps.IsValidIndex(++this->StepIndex))
        {
            this->RunStep(this->StepIndex);
        }
        else
        {
            this->Callback(FResultType(ErrorGenerator(FailedResults)), FailedResults);
        }
    }

private:
    FErrorGenerator ErrorGenerator;
    TArray<FGamebaseError> FailedResults;
};

template <typename T>
using TGamebaseAllSuccessRunnerCallback = TUniqueFunction<void(const FGamebaseInternalResult<T>&)>;

template <typename T>
class TGamebaseAllSuccessRunner final
    : public TGamebaseTaskRunner<T, TGamebaseAllSuccessRunnerCallback<T>>
{
public:
    using FBase = TGamebaseTaskRunner<T, TGamebaseAllSuccessRunnerCallback<T>>;
    using typename FBase::FResultType;
    using typename FBase::FTaskStep;
    using typename FBase::TCallbackType;
    
    explicit TGamebaseAllSuccessRunner(TArray<FTaskStep> Steps, TCallbackType&& Callback)
        : FBase(MoveTemp(Steps), MoveTemp(Callback))
    {}

protected:
    virtual void OnStepCompleted(const FResultType& Result) override
    {
        if (Result.IsError())
        {
            this->Callback(Result);
            return;
        }

        if (this->Steps.IsValidIndex(++this->StepIndex))
        {
            this->RunStep(this->StepIndex);
        }
        else
        {
            this->Callback(Result);
        }
    }
};
