﻿#pragma once

#include "CoreMinimal.h"
#include "GamebaseCoreLogger.h"
#include "GamebaseInternalDataPointer.h"
#include "Subsystems/GameInstanceSubsystem.h"
#include "Types/GamebaseAuthTypes.h"
#include "Types/GamebaseBaseTypes.h"
#include "Types/GamebaseLaunchingTypes.h"
#include "GamebaseProvider.generated.h"

struct FGamebaseTransferAccountRenewConfiguration;
struct FGamebaseConfiguration;
class UGpLoggerInstance;
class IGamebaseInternalData;
class UGamebaseConfigSave;
class FGamebaseInternalReporter;
class IGamebaseAnalytics;
class IGamebaseEventHandler;
class IGamebaseSystemInfo;
class IGamebaseAuth;
class IGamebaseTerms;
class IGamebaseTemporaryWithdrawal;
class IGamebaseGameNotice;
class IGamebaseImageNotice;
class IGamebaseWebView;
class IGamebaseUtil;
class IGamebasePush;
class IGamebasePurchase;
class IGamebaseNetwork;
class IGamebaseLogger;
class IGamebaseContact;
class IGamebaseLaunching;
class IGamebaseCommunity;

using FGamebaseTcgbClient = FGamebaseLaunchingInfo::FLaunching::FTcgbClient;

UCLASS(Abstract)
class GAMEBASECORE_API UGamebaseProvider : public UObject
{
    GENERATED_BODY()
    
public:
    static UGamebaseProvider* GetInstance(const FString& InstanceName);
    static TArray<FString> GetInstanceNameList();
    static bool IsMultipleSubsystem();
    
    virtual void Initialize(UGameInstanceSubsystem* Subsystem);
    virtual void Deinitialize();
    
    virtual void InitializeGamebase(const FGamebaseConfiguration& Configuration, const FGamebaseLaunchingInfoDelegate& OnCallback);
    virtual void SetDebugMode(bool bNewDebugMode);

    IGamebaseInternalData* GetInternalData() const { return InternalData.Get(); }
    
    FORCEINLINE UGpLoggerInstance* GetInternalReporter() const { return InternalReporter; }
    FORCEINLINE UGpLoggerInstance* GetStatsReporter() const { return StatsReporter; }
    
    virtual IGamebaseLogger* GetLogger() const { return Cast<IGamebaseLogger>(Logger); }
    
    virtual IGamebaseAnalytics* GetAnalytics() const        PURE_VIRTUAL(UGamebaseProvider::GetAnalytics, return nullptr;);
    virtual IGamebaseAuth* GetAuth() const                  PURE_VIRTUAL(UGamebaseProvider::GetAuth, return nullptr;);
    virtual IGamebaseContact* GetContact() const            PURE_VIRTUAL(UGamebaseProvider::GetContact, return nullptr;);
    virtual IGamebaseEventHandler* GetEventHandler() const  PURE_VIRTUAL(UGamebaseProvider::GetEventHandler, return nullptr;);
    virtual IGamebaseGameNotice* GetGameNotice() const      PURE_VIRTUAL(UGamebaseProvider::GetGameNotice, return nullptr;);
    virtual IGamebaseImageNotice* GetImageNotice() const    PURE_VIRTUAL(UGamebaseProvider::GetImageNotice, return nullptr;);
    virtual IGamebaseLaunching* GetLaunching() const        PURE_VIRTUAL(UGamebaseProvider::GetLaunching, return nullptr;);
    virtual IGamebaseNetwork* GetNetwork() const            PURE_VIRTUAL(UGamebaseProvider::GetNetwork, return nullptr;);
    virtual IGamebasePurchase* GetPurchase() const          PURE_VIRTUAL(UGamebaseProvider::GetPurchase, return nullptr;);
    virtual IGamebasePush* GetPush() const                  PURE_VIRTUAL(UGamebaseProvider::GetPush, return nullptr;);
    virtual IGamebaseSystemInfo* GetSystemInfo() const      PURE_VIRTUAL(UGamebaseProvider::GetSystemInfo, return nullptr;);
    virtual IGamebaseTerms* GetTerms() const                PURE_VIRTUAL(UGamebaseProvider::GetTerms, return nullptr;);
    virtual IGamebaseUtil* GetUtil() const                  PURE_VIRTUAL(UGamebaseProvider::GetUtil, return nullptr;);
    virtual IGamebaseWebView* GetWebView() const            PURE_VIRTUAL(UGamebaseProvider::GetWebView, return nullptr;);
    virtual IGamebaseCommunity* GetCommunity() const        PURE_VIRTUAL(UGamebaseProvider::GetCommunity, return nullptr;);
    
    virtual void Login(const FString& ProviderName, const FGamebaseAuthTokenDelegate& Callback) PURE_VIRTUAL(UGamebaseProvider::Login,);
    virtual void Login(const FGamebaseVariantMap& CredentialInfo, const FGamebaseAuthTokenDelegate& Callback) PURE_VIRTUAL(UGamebaseProvider::Login,);
    virtual void Login(const FString& ProviderName, const FGamebaseVariantMap& AdditionalInfo, const FGamebaseAuthTokenDelegate& Callback) PURE_VIRTUAL(UGamebaseProvider::Login,);
    
    virtual void LoginForLastLoggedInProvider(const FGamebaseAuthTokenDelegate& Callback) PURE_VIRTUAL(UGamebaseProvider::LoginForLastLoggedInProvider,);
    virtual void LoginForLastLoggedInProvider(const FGamebaseVariantMap& AdditionalInfo, const FGamebaseAuthTokenDelegate& Callback) PURE_VIRTUAL(UGamebaseProvider::LoginForLastLoggedInProvider,);

    virtual void CancelLoginWithExternalBrowser(const FGamebaseErrorDelegate& Callback) PURE_VIRTUAL(UGamebaseProvider::CancelLoginWithExternalBrowser,);

    virtual void AddMapping(const FString& ProviderName, const FGamebaseAuthTokenDelegate& Callback) PURE_VIRTUAL(UGamebaseProvider::AddMapping,);
    virtual void AddMapping(const FString& ProviderName, const FGamebaseVariantMap& AdditionalInfo, const FGamebaseAuthTokenDelegate& Callback) PURE_VIRTUAL(UGamebaseProvider::AddMapping,);
    virtual void AddMapping(const FGamebaseVariantMap& CredentialInfo, const FGamebaseAuthTokenDelegate& Callback) PURE_VIRTUAL(UGamebaseProvider::AddMapping,);
    virtual void RemoveMapping(const FString& ProviderName, const FGamebaseErrorDelegate& Callback) PURE_VIRTUAL(UGamebaseProvider::RemoveMapping,);
    
    virtual void AddMappingForcibly(const FGamebaseForcingMappingTicket& ForcingMappingTicket, const FGamebaseAuthTokenDelegate& Callback) PURE_VIRTUAL(UGamebaseProvider::AddMappingForcibly,);
    virtual void AddMappingForcibly(const FString& ProviderName, const FString& ForcingMappingKey, const FGamebaseAuthTokenDelegate& Callback) PURE_VIRTUAL(UGamebaseProvider::AddMappingForcibly,);
    virtual void AddMappingForcibly(const FString& ProviderName, const FString& ForcingMappingKey, const FGamebaseVariantMap& AdditionalInfo, const FGamebaseAuthTokenDelegate& Callback) PURE_VIRTUAL(UGamebaseProvider::AddMappingForcibly,);
    virtual void AddMappingForcibly(const FGamebaseVariantMap& CredentialInfo, const FString& ForcingMappingKey, const FGamebaseAuthTokenDelegate& Callback) PURE_VIRTUAL(UGamebaseProvider::AddMappingForcibly,);
    virtual void ChangeLogin(const FGamebaseForcingMappingTicket& ForcingMappingTicket, const FGamebaseAuthTokenDelegate& Callback) PURE_VIRTUAL(UGamebaseProvider::ChangeLogin,);
    
    virtual void Logout(const FGamebaseErrorDelegate& Callback) PURE_VIRTUAL(UGamebaseProvider::Logout,);
    virtual void Withdraw(const FGamebaseErrorDelegate& Callback) PURE_VIRTUAL(UGamebaseProvider::Withdraw,);
    virtual void WithdrawImmediately(const FGamebaseErrorDelegate& Callback) PURE_VIRTUAL(UGamebaseProvider::WithdrawImmediately,);
    virtual void RequestTemporaryWithdrawal(const FGamebaseTemporaryWithdrawalDelegate& Callback) PURE_VIRTUAL(UGamebaseProvider::RequestTemporaryWithdrawal,);
    virtual void CancelTemporaryWithdrawal(const FGamebaseErrorDelegate& Callback) PURE_VIRTUAL(UGamebaseProvider::CancelTemporaryWithdrawal,);

    virtual void IssueTransferAccount(const FGamebaseTransferAccountDelegate& Callback) PURE_VIRTUAL(UGamebaseProvider::IssueTransferAccount,);
    virtual void QueryTransferAccount(const FGamebaseTransferAccountDelegate& Callback) PURE_VIRTUAL(UGamebaseProvider::QueryTransferAccount,);
    virtual void RenewTransferAccount(const FGamebaseTransferAccountRenewConfiguration& Configuration, const FGamebaseTransferAccountDelegate& Callback) PURE_VIRTUAL(UGamebaseProvider::RenewTransferAccount,);
    virtual void TransferAccountWithIdPLogin(const FString& AccountId, const FString& AccountPassword, const FGamebaseAuthTokenDelegate& Callback) PURE_VIRTUAL(UGamebaseProvider::TransferAccountWithIdPLogin,);

    virtual TArray<FString> GetAuthMappingList() PURE_VIRTUAL(UGamebaseProvider::GetAuthMappingList, return {};);
    virtual FString GetAuthProviderUserID(const FString& ProviderName) PURE_VIRTUAL(UGamebaseProvider::GetAuthProviderUserID, return FString(););
    virtual FString GetAuthProviderAccessToken(const FString& ProviderName) PURE_VIRTUAL(UGamebaseProvider::GetAuthProviderAccessToken, return FString(););
    virtual FGamebaseAuthProviderProfilePtr GetAuthProviderProfile(const FString& ProviderName) PURE_VIRTUAL(UGamebaseProvider::GetAuthProviderProfile, return nullptr;);

    virtual FGamebaseBanInfoPtr GetBanInfo() PURE_VIRTUAL(UGamebaseProvider::GetBanInfo, return nullptr;);

protected:
    virtual void CreateInternalModule();
    virtual void InitializeInternalModule();
    
    virtual bool ShouldCreateModule() const;
    virtual bool IsCreateModule() const;
    
    virtual void OnInitializeGamebaseCompleted(const FGamebaseLaunchingInfo& LaunchingInfo);

private:
    void InitializeInternalReport(const TOptional<FGamebaseTcgbClient>& ClientInfo);

protected:
    bool bIsDebugMode;
    
    UPROPERTY()
    UGamebaseCoreLogger* Logger;
    
    UPROPERTY()
    UGpLoggerInstance* InternalReporter;
    
    UPROPERTY()
    UGpLoggerInstance* StatsReporter;

    FGamebaseInternalDataPtr InternalData;
    FGamebaseDisplayLanguagePtr DisplayLanguage;
    
    static TMap<FString, TWeakObjectPtr<UGamebaseProvider>> ProviderMap;
};
