#include "Utils/GamebaseErrorUtil.h"

#include "GamebaseDebugLogger.h"
#include "GamebaseErrorCode.h"

FGamebaseErrorPtr GamebaseErrorUtil::NewError(const FName& Domain, const int32 Code, const FString& AppendMessage)
{
    FString Message = GetErrorMessage(Code);
    if (AppendMessage.IsEmpty() == false)
    {
        Message += FString::Printf(TEXT(" (%s)"), *AppendMessage);
    }
    
    return MakeShared<FGamebaseError, ESPMode::ThreadSafe>(Code, Message, Domain);
}

FString GamebaseErrorUtil::GetErrorMessage(const int32 Code)
{
    switch (Code)
    {
        case GamebaseErrorCode::SUCCESS:
            {
                checkNoEntry();
                break;
            }

        case GamebaseErrorCode::NOT_INITIALIZED:
            return TEXT("The Gamebase SDK must be initialized");
        
        //----------------------------------------
        //  Common
        //----------------------------------------
        case GamebaseErrorCode::NOT_LOGGED_IN:
            return TEXT("Call this API after log in");

        case GamebaseErrorCode::INVALID_PARAMETER:
            return TEXT("Invalid parameter");
        
        case GamebaseErrorCode::INVALID_JSON_FORMAT:
            return TEXT("Invalid JSON format");
        
        case GamebaseErrorCode::USER_PERMISSION:
            return TEXT("Permission error");
        
        case GamebaseErrorCode::INVALID_MEMBER:
            return TEXT("This user is invalid");
        
        case GamebaseErrorCode::BANNED_MEMBER:
            return TEXT("This user is banned");
            
        case GamebaseErrorCode::SAME_REQUESTOR:
        case GamebaseErrorCode::NOT_GUEST_OR_HAS_OTHERS:
            return TEXT("");
        
        case GamebaseErrorCode::NOT_SUPPORTED:
        case GamebaseErrorCode::NOT_SUPPORTED_UE4_EDITOR:
        case GamebaseErrorCode::NOT_SUPPORTED_UE4_STANDALONE:
            return TEXT("Not supported");

        case GamebaseErrorCode::NOT_SUPPORTED_ANDROID:
        case GamebaseErrorCode::NOT_SUPPORTED_IOS:
        case GamebaseErrorCode::ANDROID_ACTIVITY_DESTROYED:
        case GamebaseErrorCode::ANDROID_ACTIVEAPP_NOT_CALLED:
        case GamebaseErrorCode::IOS_GAMECENTER_DENIED:
            {
                checkNoEntry();
                break;
            }
        
        //----------------------------------------
        //  Network (Socket)
        //----------------------------------------
        case GamebaseErrorCode::SOCKET_RESPONSE_TIMEOUT:
            return TEXT("Socket response timeout");
        
        case GamebaseErrorCode::SOCKET_ERROR:
            return TEXT("Socket error");

        case GamebaseErrorCode::SOCKET_UNKNOWN_ERROR:
            return TEXT("Socket unknown error");

        //----------------------------------------
        //  Launching
        //----------------------------------------
        case GamebaseErrorCode::LAUNCHING_SERVER_ERROR:
            return TEXT("Launching server error");
        
        case GamebaseErrorCode::LAUNCHING_NOT_EXIST_CLIENT_ID:
            return TEXT("The client ID does not exist.");
        
        case GamebaseErrorCode::LAUNCHING_UNREGISTERED_APP:
            return TEXT("This app is not registered.");
        
        case GamebaseErrorCode::LAUNCHING_UNREGISTERED_CLIENT:
            return TEXT("This client is not registered.");

        //----------------------------------------
        //  Auth
        //----------------------------------------
        case GamebaseErrorCode::AUTH_USER_CANCELED:
            return TEXT("User cancel");
        
        case GamebaseErrorCode::AUTH_NOT_SUPPORTED_PROVIDER:
            return TEXT("Not supported provider");
        
        case GamebaseErrorCode::AUTH_NOT_EXIST_MEMBER:
            return TEXT("Not exist member");
        
        case GamebaseErrorCode::AUTH_EXTERNAL_LIBRARY_INITIALIZATION_ERROR:
            return TEXT("Failed to initialize external library.");
        
        case GamebaseErrorCode::AUTH_EXTERNAL_LIBRARY_ERROR:
            return TEXT("External library error");
        
        case GamebaseErrorCode::AUTH_ALREADY_IN_PROGRESS_ERROR:
            return TEXT("Previous authentication process is not finished yet.");
        
        case GamebaseErrorCode::AUTH_INVALID_GAMEBASE_TOKEN:
            return TEXT("You have been logged out because your Gamebase Access Token is not valid. Please try logging in again.");

        case GamebaseErrorCode::AUTH_AUTHENTICATION_SERVER_ERROR:
            return TEXT("An error occurred from the authentication server.");
            
        case GamebaseErrorCode::AUTH_LOGIN_CANCEL_FAILED:
            return TEXT("Cancel login is not available");

        // transferAccount
        case GamebaseErrorCode::AUTH_TRANSFERACCOUNT_EXPIRED:
        case GamebaseErrorCode::AUTH_TRANSFERACCOUNT_BLOCK:
        case GamebaseErrorCode::AUTH_TRANSFERACCOUNT_INVALID_ID:
        case GamebaseErrorCode::AUTH_TRANSFERACCOUNT_INVALID_PASSWORD:
        case GamebaseErrorCode::AUTH_TRANSFERACCOUNT_CONSOLE_NO_CONDITION:
        case GamebaseErrorCode::AUTH_TRANSFERACCOUNT_NOT_EXIST:
        case GamebaseErrorCode::AUTH_TRANSFERACCOUNT_ALREADY_EXIST_ID:
        case GamebaseErrorCode::AUTH_TRANSFERACCOUNT_ALREADY_USED:
            {
                checkNoEntry();
                break;
            }

        // tokenLogin
        case GamebaseErrorCode::AUTH_TOKEN_LOGIN_FAILED:
            return TEXT("Login failed");
        
        case GamebaseErrorCode::AUTH_TOKEN_LOGIN_INVALID_TOKEN_INFO:
            return TEXT("Invalid token info");
        
        case GamebaseErrorCode::AUTH_TOKEN_LOGIN_INVALID_LAST_LOGGED_IN_IDP:
            return TEXT("Invalid last logged in provider");
            
        // idPLogin
        case GamebaseErrorCode::AUTH_IDP_LOGIN_FAILED:
            return TEXT("Login failed");
        
        case GamebaseErrorCode::AUTH_IDP_LOGIN_INVALID_IDP_INFO:
            return TEXT("Invalid IDP information");
        
        case GamebaseErrorCode::AUTH_IDP_LOGIN_EXTERNAL_AUTHENTICATION_REQUIRED:
            return TEXT("");
            
        // Mapping
        case GamebaseErrorCode::AUTH_ADD_MAPPING_FAILED:
            return TEXT("Mapping failed");
            
        case GamebaseErrorCode::AUTH_ADD_MAPPING_ALREADY_MAPPED_TO_OTHER_MEMBER:
            return TEXT("Already mapped to other member");
            
        case GamebaseErrorCode::AUTH_ADD_MAPPING_ALREADY_HAS_SAME_IDP:
            return TEXT("Already has same IDP");
            
        case GamebaseErrorCode::AUTH_ADD_MAPPING_INVALID_IDP_INFO:
            return TEXT("Invalid IDP information");
            
        case GamebaseErrorCode::AUTH_ADD_MAPPING_CANNOT_ADD_GUEST_IDP:
            return TEXT("Mapping with guest IdP is unavailable");
            
        case GamebaseErrorCode::AUTH_ADD_MAPPING_FORCIBLY_NOT_EXIST_KEY:
            return TEXT("The forcingMappingKey does not exist.");
            
        case GamebaseErrorCode::AUTH_ADD_MAPPING_FORCIBLY_ALREADY_USED_KEY:
            return TEXT("This forcingMappingKey was already used.");
            
        case GamebaseErrorCode::AUTH_ADD_MAPPING_FORCIBLY_EXPIRED_KEY:
            return TEXT("This forcingMappingKey has expired.");
            
        case GamebaseErrorCode::AUTH_ADD_MAPPING_FORCIBLY_DIFFERENT_IDP:
            return TEXT("Failed as trying to mapping with different idP.");
            
        case GamebaseErrorCode::AUTH_ADD_MAPPING_FORCIBLY_DIFFERENT_AUTHKEY:
            return TEXT("Failed as trying to mapping with different authKey(account).");
            
        case GamebaseErrorCode::AUTH_REMOVE_MAPPING_FAILED:
            return TEXT("Remove mapping failed");
            
        case GamebaseErrorCode::AUTH_REMOVE_MAPPING_LAST_MAPPED_IDP:
            return TEXT("Remove mapping failed because this provider is last mapped IDP");
            
        case GamebaseErrorCode::AUTH_REMOVE_MAPPING_LOGGED_IN_IDP:
            return TEXT("Currently logged-in provider can not be remove mapping");
            
        // logout
        case GamebaseErrorCode::AUTH_LOGOUT_FAILED:
            return TEXT("Logout failed");
            
        // withdraw
        case GamebaseErrorCode::AUTH_WITHDRAW_FAILED:
            return TEXT("Withdraw failed");
        
        case GamebaseErrorCode::AUTH_WITHDRAW_ALREADY_TEMPORARY_WITHDRAW:
            return TEXT("You are already applying for withdrawal.");
        
        case GamebaseErrorCode::AUTH_WITHDRAW_NOT_TEMPORARY_WITHDRAW:
            return TEXT("You have never asked withdraw.");
        
        // status not playable
        case GamebaseErrorCode::AUTH_NOT_PLAYABLE:
            return TEXT("Not playable");
        
        // unknown
        case GamebaseErrorCode::AUTH_UNKNOWN_ERROR:
            return TEXT("Auth unknown error");

        //----------------------------------------
        //  Purchase
        //----------------------------------------
        case GamebaseErrorCode::PURCHASE_NOT_INITIALIZED:
            return TEXT("Purchase not initialized");
        
        case GamebaseErrorCode::PURCHASE_USER_CANCELED:
            return TEXT("User canceled purchase");
        
        case GamebaseErrorCode::PURCHASE_NOT_FINISHED_PREVIOUS_PURCHASING:
            return TEXT("Not finished previous purchasing yet");
        
        case GamebaseErrorCode::PURCHASE_NOT_ENOUGH_CASH:
            return TEXT("Not enough cash");
        
        case GamebaseErrorCode::PURCHASE_INACTIVE_PRODUCT_ID:
            return TEXT("GamebaseProductID is not activated.");
        
        case GamebaseErrorCode::PURCHASE_NOT_EXIST_PRODUCT_ID:
            return TEXT("GamebaseProductID is not exist.");

        case GamebaseErrorCode::PURCHASE_LIMIT_EXCEEDED:
            return TEXT("Purchase limit exceeded.");
        
        case GamebaseErrorCode::PURCHASE_NOT_SUPPORTED_MARKET:
            return TEXT("Store code is invalid");

        case GamebaseErrorCode::PURCHASE_EXTERNAL_LIBRARY_ERROR:
            return TEXT("External library error");

        case GamebaseErrorCode::PURCHASE_UNKNOWN_ERROR:
            return TEXT("Purchase unknown error");

        //----------------------------------------
        //  Push
        //----------------------------------------
        case GamebaseErrorCode::PUSH_EXTERNAL_LIBRARY_ERROR:
        case GamebaseErrorCode::PUSH_ALREADY_IN_PROGRESS_ERROR:
        case GamebaseErrorCode::PUSH_UNKNOWN_ERROR:
            {
                checkNoEntry();
                break;
            }

        //----------------------------------------
        //  Logger
        //----------------------------------------
        case GamebaseErrorCode::LOGGER_NOT_INITIALIZED:
            return TEXT("Logger not initialized");
        
        case GamebaseErrorCode::LOGGER_EXTERNAL_LIBRARY_ERROR:
            return TEXT("External library error");
        
        case GamebaseErrorCode::LOGGER_UNKNOWN_ERROR:
            return TEXT("Logger unknown error");
            
        //----------------------------------------
        //  UI
        //----------------------------------------
        // image notice
        case GamebaseErrorCode::UI_IMAGE_NOTICE_TIMEOUT:
            return TEXT("Timeout occurred while displaying image notices.");
        
        case GamebaseErrorCode::UI_IMAGE_NOTICE_NOT_SUPPORTED_OS:
            return TEXT("This OS does not support pop-up type image notice.");

        //contact
        case GamebaseErrorCode::UI_CONTACT_FAIL_INVALID_URL:
            return TEXT("Service center URL generation failed.");
        
        case GamebaseErrorCode::UI_CONTACT_FAIL_ISSUE_SHORT_TERM_TICKET:
            return TEXT("Failed to create ShortTermTicket.");

        case GamebaseErrorCode::UI_CONTACT_FAIL_ANDROID_DUPLICATED_VIEW:
            //TODO: Android Only?
            return TEXT("Service center is already open.");
            
        case GamebaseErrorCode::UI_TERMS_NOT_EXIST_IN_CONSOLE:
            return TEXT("The terms and conditions information is not registered on the console.");
        
        case GamebaseErrorCode::UI_TERMS_NOT_EXIST_FOR_DEVICE_COUNTRY:
            return TEXT("The terms and conditions information corresponding to the device country code is not registered on the console.");
        
        case GamebaseErrorCode::UI_TERMS_UNREGISTERED_SEQ:
            return TEXT("The unregistered terms Seq value.");
        
        case GamebaseErrorCode::UI_TERMS_ALREADY_IN_PROGRESS_ERROR:
            return TEXT("Terms API is already in progress");
        
        case GamebaseErrorCode::UI_TERMS_ANDROID_DUPLICATED_VIEW:
            return TEXT("The terms and conditions web view is already open.");
            
        case GamebaseErrorCode::UI_COMMUNITY_FAIL_INVALID_URL:
            return TEXT("Community URL generation failed.");
        
        case GamebaseErrorCode::UI_COMMUNITY_FAIL_ISSUE_SHORT_TERM_TICKET:
            return TEXT("Failed to create ShortTermTicket.");
        
        case GamebaseErrorCode::UI_COMMUNITY_FAIL_ANDROID_DUPLICATED_VIEW:
            return TEXT("Gamebase community is already open.");
            
        case GamebaseErrorCode::UI_UNKNOWN_ERROR:
            return TEXT("UI unknown error");

        //----------------------------------------
        //  Webview
        //----------------------------------------
        case GamebaseErrorCode::WEBVIEW_INVALID_URL:
            return TEXT("The url is invalid. Please check it first.");
        
        case GamebaseErrorCode::WEBVIEW_TIMEOUT:
            return TEXT("Timeout occurred while displaying web view.");
        
        case GamebaseErrorCode::WEBVIEW_HTTP_ERROR:
            return TEXT("Display of the webview failed due to an HTTP error.");
            
        case GamebaseErrorCode::WEBVIEW_OPENED_NEW_BROWSER_BEFORE_CLOSE:
            return TEXT("A new webview is opened before closing the browser type webview.");
        
        case GamebaseErrorCode::WEBVIEW_UNKNOWN_ERROR:
            return TEXT("Web view unknown error.");

        //----------------------------------------
        //  Server
        //----------------------------------------
        case GamebaseErrorCode::SERVER_INTERNAL_ERROR:
            return TEXT("Gamebase Server internal error");
        
        case GamebaseErrorCode::SERVER_REMOTE_SYSTEM_ERROR:
            return TEXT("Gamebase Server remote system error");
        
        case GamebaseErrorCode::SERVER_INVALID_RESPONSE:
            return TEXT("An invalid response was received from the server.");

        case GamebaseErrorCode::SERVER_UNKNOWN_ERROR:
            return TEXT("Gamebase Server unknown error");

        //----------------------------------------
        //  Unknown error
        //----------------------------------------
        case GamebaseErrorCode::UNKNOWN_ERROR:
            return TEXT("Unknown error");

        //----------------------------------------
        //  Platform specified errors (Unreal)
        //----------------------------------------
        case GamebaseErrorCode::UNREAL_GET_ENGINE_INFO_ERROR:
            return TEXT("The info taken from the engine is incorrect.");
            
        case GamebaseErrorCode::NATIVE_METHOD_CALL_EXCEPTION_IN_PLUGIN:
            {
                checkNoEntry();
                break;
            }
            
        default:
            GAMEBASE_LOG_GLOBAL_WARNING("The error code %d does not have a message set.", Code);
            return TEXT("Unknown error");
    }
        
    return {};
}
