﻿#include "GamebaseSystemUtils.h"

#include "GamebaseDebugLogger.h"
#include "GamebaseDeviceInfo.h"
#include "Interfaces/IPluginManager.h"
#include "Internationalization/Culture.h"
#include "Utils/GamebaseMics.h"

namespace GamebaseSystemUtils
{
    const FString PluginName = TEXT("Gamebase");
}

FString GamebaseSystemUtils::GetPluginVersion()
{
    static FString PluginVersion;
    if (PluginVersion.IsEmpty())
    {
        const TSharedPtr<IPlugin> Plugin = IPluginManager::Get().FindPlugin(PluginName);
        if (Plugin.IsValid() == false)
        {
            GAMEBASE_LOG_GLOBAL_DEBUG("Gamebase plugin is not loaded");
        
            static const FString UnknownString = TEXT("UNKNOWN");
            PluginVersion = UnknownString;
        }
    
        PluginVersion = Plugin->GetDescriptor().VersionName;
    }
    
    return PluginVersion;
}

FString GamebaseSystemUtils::GetPlatform()
{
    static const FString PlatformString =
#if PLATFORM_ANDROID
        TEXT("AOS");
#elif PLATFORM_IOS
        TEXT("IOS");
#elif PLATFORM_WINDOWS
        TEXT("WINDOWS");
#elif PLATFORM_MAC
        TEXT("MAC");
#else
        TEXT("UNKNOWN");
#endif

    return PlatformString;
}

FString GamebaseSystemUtils::GetDeviceKey()
{
    return FGamebaseDeviceInfo::Get().GetDeviceKey().Get({});
}

FString GamebaseSystemUtils::GetOSVersion()
{
    return FPlatformMisc::GetOSVersion();
}

FString GamebaseSystemUtils::GetDeviceModel()
{
    return FPlatformMisc::GetDeviceMakeAndModel();
}

FString GamebaseSystemUtils::GetDeviceLanguage()
{
    return FInternationalization::Get().GetDefaultCulture()->GetTwoLetterISOLanguageName();
}

FString GamebaseSystemUtils::GetCountryCode()
{
    return FGamebaseDeviceInfo::Get().GetCountryCode().Get({});
}

FString GamebaseSystemUtils::GetSimCode()
{
    static const FString UnknownString = TEXT("ZZ");
    return UnknownString;
}

FString GamebaseSystemUtils::GetNetworkConnectionType()
{
    return LexToString(FPlatformMisc::GetNetworkConnectionType());
}

FString GamebaseSystemUtils::GetBinariesPath()
{
    const FString PluginDir = IPluginManager::Get().FindPlugin(PluginName)->GetBaseDir();

    return FPaths::Combine(PluginDir, TEXT("Binaries"), FPlatformProcess::GetBinariesSubdirectory());
}

FString GamebaseSystemUtils::GetContentPath()
{
    return IPluginManager::Get().FindPlugin(PluginName)->GetContentDir();
}

bool GamebaseSystemUtils::IsNetworkConnected()
{
    return FGamebaseMics::IsNetworkConnected();
}
