﻿#include "GamebaseProvider.h"

#include "GamebaseDebugLogger.h"
#include "GamebaseDeviceInfo.h"
#include "GamebaseDisplayLanguage.h"
#include "GamebaseInternalData.h"
#include "GamebaseInternalReport.h"
#include "GamebaseSystemUtils.h"
#include "GpLoggerConfiguration.h"
#include "GpLoggerSubsystem.h"

TMap<FString, TWeakObjectPtr<UGamebaseProvider>> UGamebaseProvider::ProviderMap;

UGamebaseProvider* UGamebaseProvider::GetInstance(const FString& InstanceName)
{
    const TWeakObjectPtr<UGamebaseProvider>* InstancePtr = ProviderMap.Find(InstanceName);
    if (InstancePtr && InstancePtr->IsValid())
    {
        return InstancePtr->Get();
    }
    
    return nullptr;
}

TArray<FString> UGamebaseProvider::GetInstanceNameList()
{
    TArray<FString> KeyArray;
    ProviderMap.GenerateKeyArray(KeyArray);
    return KeyArray;
}

bool UGamebaseProvider::IsMultipleSubsystem()
{
    return ProviderMap.Num() > 1;
}

void UGamebaseProvider::Initialize(UGameInstanceSubsystem* Subsystem)
{
    check(Subsystem && Subsystem->GetGameInstance());
    
    FGamebaseDeviceInfo::Get().Initialize();
    
    const FString InstanceName = Subsystem->GetGameInstance()->GetName();
    ProviderMap.Emplace(InstanceName, this);
    
    InternalData = MakeShared<FGamebaseInternalData>(Subsystem->GetGameInstance(), InstanceName, FGamebaseConfiguration());
    
    if (ShouldCreateModule())
    {
        CreateInternalModule();
    }
}

void UGamebaseProvider::Deinitialize()
{
    DisplayLanguage.Reset();
    
    ProviderMap.Remove(InternalData->GetInstanceName());
}

void UGamebaseProvider::InitializeGamebase(const FGamebaseConfiguration& Configuration, const FGamebaseLaunchingInfoDelegate& OnCallback)
{
    GAMEBASE_LOG_DEBUG("Initialize - %s", *Configuration.AppID);
    GAMEBASE_LOG_DEBUG("Gamebase Unreal Plugin started v%s", *GamebaseSystemUtils::GetPluginVersion());
    
    if (InternalData.IsValid())
    {
        InternalData->SetConfiguration(Configuration);
    }
    
    if (ShouldCreateModule())
    {
        CreateInternalModule();
    }
    
    InitializeInternalModule();
}

void UGamebaseProvider::SetDebugMode(const bool bNewDebugMode)
{
    bIsDebugMode = bNewDebugMode;
    FGamebaseDebugLogger::IsDebugLog = bIsDebugMode;
}

void UGamebaseProvider::CreateInternalModule()
{
    DisplayLanguage = MakeShared<FGamebaseDisplayLanguage>(InternalData);
    Logger = NewObject<UGamebaseCoreLogger>(this);
}

void UGamebaseProvider::InitializeInternalModule()
{
    InternalData->SetDisplayLanguage(DisplayLanguage);
}

void UGamebaseProvider::OnInitializeGamebaseCompleted(const FGamebaseLaunchingInfo& LaunchingInfo)
{
    InitializeInternalReport(LaunchingInfo.Launching.TcgbClient);
    GamebaseInternalReport::Init::EngineInformation(*InternalData);
    
    FGamebaseDeviceInfo::Get().InitializeCompleted();
}

bool UGamebaseProvider::ShouldCreateModule() const
{
    return DisplayLanguage.IsValid() == false;
}

bool UGamebaseProvider::IsCreateModule() const
{
    return DisplayLanguage.IsValid();
}

void UGamebaseProvider::InitializeInternalReport(const TOptional<FGamebaseTcgbClient>& ClientInfo)
{
    if (ClientInfo.IsSet() == false)
    {
        GAMEBASE_LOG_DEBUG("The value of launching.tcgbClient is null.")
        return;
    }
    
    if (UGpLoggerSubsystem* LoggerSubsystem = UGameInstance::GetSubsystem<UGpLoggerSubsystem>(GetWorld()->GetGameInstance()))
    {
        auto ConvertZone = [](const FString& Zone)-> EGpLoggerServiceZone
        {
            if (Zone.Equals(TEXT("alpha"), ESearchCase::IgnoreCase))
            {
                return EGpLoggerServiceZone::Alpha;
            }
            if (Zone.Equals(TEXT("beta"), ESearchCase::IgnoreCase))
            {
                return EGpLoggerServiceZone::Beta;
            }

            return EGpLoggerServiceZone::Real;
        };
        
        const EGpLoggerServiceZone LoggerZone = ConvertZone(InternalData->GetZoneType());
    
        const auto& Stability = ClientInfo.GetValue().Stability;
        if (Stability.AppKey.IsEmpty() == false)
        {
            InternalReporter = LoggerSubsystem->NewInstance(Stability.AppKey, LoggerZone);
        }

        const auto& ForceRemoteSettings = ClientInfo.GetValue().ForceRemoteSettings;
        if (ForceRemoteSettings.IsSet())
        {
            if (ForceRemoteSettings.GetValue().Log.AppKeyIndicator.IsEmpty() == false)
            {
                StatsReporter = LoggerSubsystem->NewInstance(ForceRemoteSettings.GetValue().Log.AppKeyIndicator, LoggerZone);
            }
        }
    }
}