#include "GamebaseDisplayLanguage.h"

#include "GamebaseDebugLogger.h"
#include "GamebaseInternalData.h"
#include "GamebaseSystemUtils.h"

namespace GamebaseDisplayLanguage
{
    const FString DefaultCode(TEXT("en"));
    
    const FString DefaultLocalizedJsonFileName(TEXT("defaultlocalizedstring.json"));
    const FString LocalizedJsonFileName(TEXT("localizedstring.json"));
    const FString LocalizedStringValue(TEXT("-"));
}

class FGamebaseDisplayLanguage::FLocalizedStrings
{
public:
    using FLocalizedStringMap = TMap<FLocalizedStringKey, FLocalizedStringValue>;
    
    explicit FLocalizedStrings(
        const FLanguageCode& LanguageCode,
        const FLocalizedStringMap& LoadedStrings)
        : LanguageCode(LanguageCode)
        , LocalizedStringMap(LoadedStrings)
    {
    }

    FLocalizedStringValue GetValue(const FLocalizedStringKey& Key) const
    {
        if (LocalizedStringMap.Contains(Key))
        {
            return LocalizedStringMap[Key];
        }
    
        return FLocalizedStringValue();
    }
    
private:
    FLanguageCode LanguageCode;
    FLocalizedStringMap LocalizedStringMap;
};


FGamebaseDisplayLanguage::FGamebaseDisplayLanguage(const FGamebaseInternalDataPtr& InternalData)
    : InternalData(InternalData)
    , DeviceCode(GamebaseSystemUtils::GetDeviceLanguage())
    , DefaultCode(GamebaseDisplayLanguage::DefaultCode)
    , DisplayCode(DeviceCode)
{
    LoadResources();
}

FGamebaseDisplayLanguage::~FGamebaseDisplayLanguage()
{
}

void FGamebaseDisplayLanguage::SetLanguageCode(
    const FLanguageCode& LanguageCode)
{
    DisplayCode = LanguageCode;
    
    if (LocalizedStringsContainer.Contains(LanguageCode))
    {
        ConfiguredCode = LanguageCode;
    }
    else
    {
        ConfiguredCode.Reset();
        GAMEBASE_LOG_WARNING("Unconfigurable language code. (code: %s)", *LanguageCode);
    }
}

void FGamebaseDisplayLanguage::UpdateLaunchingLanguageCode(
    const FLanguageCode& DeviceLanguage,
    const FLanguageCode& DefaultLanguage)
{
    auto UpdateLanguageCode = [this](FLanguageCode& TargetCode, const FLanguageCode& NewCode, const FString& TypeString) {
        if (!NewCode.IsEmpty())
        {
            GAMEBASE_LOG_DEBUG("Updated %s language based on launching info. (old: %s, new: %s)", *TypeString, *NewCode, *TargetCode);
            TargetCode = NewCode;
        }
        else
        {
            GAMEBASE_LOG_WARNING("%s language info is missing in launching info.", *TypeString);
        }
    };

    UpdateLanguageCode(DeviceCode, DeviceLanguage, TEXT("device"));
    UpdateLanguageCode(DefaultCode, DefaultLanguage, TEXT("default"));
    
    if (!ConfiguredCode.IsSet())
    {
        DisplayCode = [this]() -> FLanguageCode
        {
            const TArray<FLanguageCode> PriorityCodes = { DeviceCode, DefaultCode, GamebaseDisplayLanguage::DefaultCode };
            for (const auto& Code : PriorityCodes)
            {
                if (LocalizedStringsContainer.Contains(Code))
                {
                    return Code;
                }
            }

            GAMEBASE_LOG_WARNING("Default SDK language code '%s' is also unavailable.", *GamebaseDisplayLanguage::DefaultCode);
            return GamebaseDisplayLanguage::DefaultCode;
        }();
    }
}

FString FGamebaseDisplayLanguage::GetDisplayLanguageCode() const
{
    return DisplayCode;
}

FString FGamebaseDisplayLanguage::GetLocalizedString(const FLocalizedStringKey& Key) const
{
    if (LocalizedStringsContainer.Num() == 0)
    {
        return FString();
    }

    FString Value;
    if (LocalizedStringsContainer.Contains(DisplayCode))
    {
        Value = LocalizedStringsContainer[DisplayCode]->GetValue(Key);
        if (!Value.IsEmpty())
        {
            return Value;
        }
    }
    
    if (LocalizedStringsContainer.Contains(GamebaseDisplayLanguage::DefaultCode))
    {
        Value = LocalizedStringsContainer[DisplayCode]->GetValue(Key);
        if (!Value.IsEmpty())
        {
            return Value;
        }
    }
    
    return Key;
}

void FGamebaseDisplayLanguage::LoadResources()
{   
    const FString FilePath = GamebaseSystemUtils::GetContentPath() / GamebaseDisplayLanguage::DefaultLocalizedJsonFileName;
    if (!IFileManager::Get().FileExists(*FilePath))
    {
        GAMEBASE_LOG_WARNING("The LocalizedString file not found");
        return;
    }
    
    FString FileData;
    FFileHelper::LoadFileToString(FileData, *FilePath);
    if (FileData.IsEmpty())
    {
        GAMEBASE_LOG_WARNING("The LocalizedString is empty");
        return;
    }
    
    TSharedPtr<FJsonObject> Object;
    const TSharedRef<TJsonReader<>> Reader = TJsonReaderFactory<>::Create(FileData);
    if (!FJsonSerializer::Deserialize(Reader, Object) || !Object.IsValid())
    {
        GAMEBASE_LOG_WARNING("The LocalizedString is empty");
        return;
    }

    for (const auto& LanguageInfo : Object->Values)
    {
        const FLanguageCode LanguageCode = LanguageInfo.Key;
        const TSharedPtr<FJsonValue> LocalizedStringsJsonValue = LanguageInfo.Value;

        TMap<FLocalizedStringKey, FLocalizedStringValue> LocalizedStrings;
        for (auto LocalizedString : LocalizedStringsJsonValue->AsObject()->Values)
        {
            FString LocalizedStringValue;
            if (LocalizedString.Value->TryGetString(LocalizedStringValue))
            {
                LocalizedStrings.Add(LocalizedString.Key, LocalizedStringValue);
            }
            else
            {
                GAMEBASE_LOG_WARNING("Json parsing error. displayLanguageKey: %s", *LocalizedString.Key);
            }
        }
        
        LocalizedStringsContainer.Add(LanguageCode, MakeShared<FLocalizedStrings>(LanguageCode, LocalizedStrings));
    }
    
    GAMEBASE_LOG_DEBUG("Succeeded to load the LocalizedString");
}
