#include "GamebaseDebugLogger.h"

#include "GamebaseInternalData.h"
#include "GamebaseInternalReport.h"
#include "GamebaseProvider.h"

DEFINE_LOG_CATEGORY(LogGamebase);

bool FGamebaseDebugLogger::IsDebugLog = true;

void FGamebaseDebugLogger::Debug(const FString& Message)
{
    if (IsDebugLog == false)
    {
        return;
    }
    
    UE_LOG(LogGamebase, Display, TEXT("%s"), *Message);
}

void FGamebaseDebugLogger::Warning(const FString& Message)
{
    UE_LOG(LogGamebase, Warning, TEXT("%s"), *Message);

    TArray<FString> Instances = UGamebaseProvider::GetInstanceNameList();
    for (const FString& Instance : Instances)
    {
        if (const auto Provider = UGamebaseProvider::GetInstance(Instance))
        {
            if (Provider == nullptr)
            {
                continue;;
            }
            
            GamebaseInternalReport::Common::WarnMessage(*Provider->GetInternalData(), Message);
        }
    }
}

void FGamebaseDebugLogger::Error(const FString& Message)
{
    UE_LOG(LogGamebase, Error, TEXT("%s"), *Message);

    TArray<FString> Instances = UGamebaseProvider::GetInstanceNameList();
    for (const FString& Instance : Instances)
    {
        if (const auto Provider = UGamebaseProvider::GetInstance(Instance))
        {
            if (Provider == nullptr)
            {
                continue;;
            }
            
            GamebaseInternalReport::Common::ErrorMessage(*Provider->GetInternalData(), Message);
        }
    }
}

void FGamebaseDebugLogger::Debug(const FString& Message, const IGamebaseInternalData& InternalData)
{
    if (IsDebugLog == false)
    {
        return;
    }

    const bool bPrintInstanceName = UGamebaseProvider::IsMultipleSubsystem() && InternalData.GetInstanceName().IsEmpty() == false;
    if (bPrintInstanceName)
    {
        UE_LOG(LogGamebase, Display, TEXT("%s: %s"), *InternalData.GetInstanceName(), *Message);
    }
    else
    {
        UE_LOG(LogGamebase, Display, TEXT("%s"), *Message);
    }

    //TODO: [Gamebase-Client/1543] Debug Level Type 지표가 너무 많이 쌓여 로그 전송 로직 제거
    //GamebaseInternalReport::Common::DebugMessage(Message);
}

void FGamebaseDebugLogger::Warning(const FString& Message, const IGamebaseInternalData& InternalData)
{
    const bool bPrintInstanceName = UGamebaseProvider::IsMultipleSubsystem() && InternalData.GetInstanceName().IsEmpty() == false;
    if (bPrintInstanceName)
    {
        UE_LOG(LogGamebase, Warning, TEXT("%s: %s"), *InternalData.GetInstanceName(), *Message);
    }
    else
    {
        UE_LOG(LogGamebase, Warning, TEXT("%s"), *Message);
    }
    
    GamebaseInternalReport::Common::WarnMessage(InternalData, Message);
}

void FGamebaseDebugLogger::Error(const FString& Message, const IGamebaseInternalData& InternalData)
{
    const bool bPrintInstanceName = UGamebaseProvider::IsMultipleSubsystem() && InternalData.GetInstanceName().IsEmpty() == false;
    if (bPrintInstanceName)
    {
        UE_LOG(LogGamebase, Error, TEXT("%s: %s"), *InternalData.GetInstanceName(), *Message);
    }
    else
    {
        UE_LOG(LogGamebase, Error, TEXT("%s"), *Message);
    }

    GamebaseInternalReport::Common::ErrorMessage(InternalData, Message);
}