﻿#include "GamebaseCoreSubsystem.h"

#include "GamebaseDebugLogger.h"
#include "Interfaces/IGamebaseEventHandler.h"
#include "Interfaces/IGamebaseSystemInfo.h"

class FGamebaseTemporaryWithdrawal final : public IGamebaseTemporaryWithdrawal
{
public:
    explicit FGamebaseTemporaryWithdrawal(const TWeakObjectPtr<UGamebaseProvider>& Provider) : Provider(Provider) {}
    virtual ~FGamebaseTemporaryWithdrawal() override = default;

    virtual void WithdrawImmediately(const FGamebaseErrorDelegate& Callback) override;
    virtual void RequestWithdrawal(const FGamebaseTemporaryWithdrawalDelegate& Callback) override;
    virtual void CancelWithdrawal(const FGamebaseErrorDelegate& Callback) override;

private:
    TWeakObjectPtr<UGamebaseProvider> Provider;
};

void FGamebaseTemporaryWithdrawal::WithdrawImmediately(const FGamebaseErrorDelegate& Callback)
{
    if (Provider.IsValid())
    {
        Provider->WithdrawImmediately(Callback);
    }
}

void FGamebaseTemporaryWithdrawal::RequestWithdrawal(const FGamebaseTemporaryWithdrawalDelegate& Callback)
{
    if (Provider.IsValid())
    {
        Provider->RequestTemporaryWithdrawal(Callback);
    }
}

void FGamebaseTemporaryWithdrawal::CancelWithdrawal(const FGamebaseErrorDelegate& Callback)
{
    if (Provider.IsValid())
    {
        Provider->CancelTemporaryWithdrawal(Callback);
    }
}

void UGamebaseCoreSubsystem::SetDebugMode(const bool bIsDebugMode)
{
    Provider->SetDebugMode(bIsDebugMode);
}

void UGamebaseCoreSubsystem::SetDisplayLanguageCode(const FString& LanguageCode) const
{
    Provider->GetSystemInfo()->SetDisplayLanguageCode(LanguageCode);
}

void UGamebaseCoreSubsystem::Initialize(const FGamebaseConfiguration& Configuration, const FGamebaseLaunchingInfoDelegate& Callback)
{
    Provider->InitializeGamebase(Configuration, Callback);
}

FDelegateHandle UGamebaseCoreSubsystem::AddEventHandler(const FGamebaseEventDelegate::FDelegate& Callback)
{
    return Provider->GetEventHandler()->AddHandler(Callback);
}

void UGamebaseCoreSubsystem::RemoveEventHandler(const FDelegateHandle& Handle)
{
    Provider->GetEventHandler()->RemoveHandler(Handle);
}

void UGamebaseCoreSubsystem::RemoveAllEventHandler()
{
    Provider->GetEventHandler()->RemoveAllHandler();
}

void UGamebaseCoreSubsystem::Login(const FString& ProviderName, const FGamebaseAuthTokenDelegate& Callback)
{
    Provider->Login(ProviderName, Callback);
}

void UGamebaseCoreSubsystem::Login(const FGamebaseVariantMap& CredentialInfo, const FGamebaseAuthTokenDelegate& Callback)
{
    Provider->Login(CredentialInfo, Callback);
}

void UGamebaseCoreSubsystem::Login(const FString& ProviderName, const FGamebaseVariantMap& AdditionalInfo,
    const FGamebaseAuthTokenDelegate& Callback)
{
    Provider->Login(ProviderName, AdditionalInfo, Callback);
}

void UGamebaseCoreSubsystem::LoginForLastLoggedInProvider(const FGamebaseAuthTokenDelegate& Callback)
{
    Provider->LoginForLastLoggedInProvider(Callback);
}

void UGamebaseCoreSubsystem::LoginForLastLoggedInProvider(const FGamebaseVariantMap& AdditionalInfo, const FGamebaseAuthTokenDelegate& Callback)
{
    Provider->LoginForLastLoggedInProvider(AdditionalInfo, Callback);
}

void UGamebaseCoreSubsystem::CancelLoginWithExternalBrowser()
{
    Provider->CancelLoginWithExternalBrowser({});
}

void UGamebaseCoreSubsystem::AddMapping(const FString& ProviderName, const FGamebaseAuthTokenDelegate& Callback)
{
    Provider->AddMapping(ProviderName, Callback);
}

void UGamebaseCoreSubsystem::AddMapping(const FString& ProviderName, const FGamebaseVariantMap& AdditionalInfo,
    const FGamebaseAuthTokenDelegate& Callback)
{
    Provider->AddMapping(ProviderName, AdditionalInfo, Callback);
}

void UGamebaseCoreSubsystem::AddMapping(const FGamebaseVariantMap& CredentialInfo, const FGamebaseAuthTokenDelegate& Callback)
{
    Provider->AddMapping(CredentialInfo, Callback);
}

void UGamebaseCoreSubsystem::ChangeLogin(const FGamebaseForcingMappingTicket& ForcingMappingTicket, const FGamebaseAuthTokenDelegate& Callback)
{
    Provider->ChangeLogin(ForcingMappingTicket, Callback);
}

void UGamebaseCoreSubsystem::AddMappingForcibly(const FGamebaseForcingMappingTicket& ForcingMappingTicket,
    const FGamebaseAuthTokenDelegate& Callback)
{
    Provider->AddMappingForcibly(ForcingMappingTicket, Callback);
}

void UGamebaseCoreSubsystem::AddMappingForcibly(const FString& ProviderName, const FString& ForcingMappingKey,
    const FGamebaseAuthTokenDelegate& Callback)
{
    Provider->AddMappingForcibly(ProviderName, ForcingMappingKey, Callback);
}

void UGamebaseCoreSubsystem::AddMappingForcibly(const FString& ProviderName, const FString& ForcingMappingKey,
    const FGamebaseVariantMap& AdditionalInfo, const FGamebaseAuthTokenDelegate& Callback)
{
    Provider->AddMappingForcibly(ProviderName, ForcingMappingKey, AdditionalInfo, Callback);
}

void UGamebaseCoreSubsystem::AddMappingForcibly(const FGamebaseVariantMap& CredentialInfo, const FString& ForcingMappingKey,
    const FGamebaseAuthTokenDelegate& Callback)
{
    Provider->AddMappingForcibly(CredentialInfo, ForcingMappingKey, Callback);
}

void UGamebaseCoreSubsystem::RemoveMapping(const FString& ProviderName, const FGamebaseErrorDelegate& Callback)
{
    Provider->RemoveMapping(ProviderName, Callback);
}

void UGamebaseCoreSubsystem::Logout(const FGamebaseErrorDelegate& Callback)
{
    Provider->Logout(Callback);
}

void UGamebaseCoreSubsystem::Withdraw(const FGamebaseErrorDelegate& Callback)
{
    Provider->Withdraw(Callback);
}

void UGamebaseCoreSubsystem::IssueTransferAccount(const FGamebaseTransferAccountDelegate& Callback)
{
    Provider->IssueTransferAccount(Callback);
}

void UGamebaseCoreSubsystem::QueryTransferAccount(const FGamebaseTransferAccountDelegate& Callback)
{
    Provider->QueryTransferAccount(Callback);
}

void UGamebaseCoreSubsystem::RenewTransferAccount(const FGamebaseTransferAccountRenewConfiguration& Configuration,
    const FGamebaseTransferAccountDelegate& Callback)
{
    Provider->RenewTransferAccount(Configuration, Callback);
}

void UGamebaseCoreSubsystem::TransferAccountWithIdPLogin(const FString& AccountId, const FString& AccountPassword,
    const FGamebaseAuthTokenDelegate& Callback)
{
    Provider->TransferAccountWithIdPLogin(AccountId, AccountPassword, Callback);
}

void UGamebaseCoreSubsystem::RequestLastLoggedInProvider(const FGamebaseLastLoggedInProviderDelegate& Callback) const
{
    Provider->GetSystemInfo()->RequestLastLoggedInProvider(Callback);
}

FString UGamebaseCoreSubsystem::GetLastLoggedInProvider() const
{
    return Provider->GetSystemInfo()->GetLastLoggedInProvider();
}

FString UGamebaseCoreSubsystem::GetUserID() const
{
    return Provider->GetSystemInfo()->GetUserID();
}

FString UGamebaseCoreSubsystem::GetAccessToken() const
{
    return Provider->GetSystemInfo()->GetAccessToken();
}

TArray<FString> UGamebaseCoreSubsystem::GetAuthMappingList()
{
    return Provider->GetAuthMappingList();
}

FString UGamebaseCoreSubsystem::GetAuthProviderUserID(const FString& ProviderName)
{
    return Provider->GetAuthProviderUserID(ProviderName);
}

FString UGamebaseCoreSubsystem::GetAuthProviderAccessToken(const FString& ProviderName)
{
    return Provider->GetAuthProviderAccessToken(ProviderName);
}

FGamebaseAuthProviderProfilePtr UGamebaseCoreSubsystem::GetAuthProviderProfile(const FString& ProviderName)
{
    return Provider->GetAuthProviderProfile(ProviderName);
}

FGamebaseBanInfoPtr UGamebaseCoreSubsystem::GetBanInfo()
{
    return Provider->GetBanInfo();
}

FString UGamebaseCoreSubsystem::GetSDKVersion() const
{
    return Provider->GetSystemInfo()->GetSDKVersion();
}

bool UGamebaseCoreSubsystem::IsSandbox() const
{
    return Provider->GetSystemInfo()->IsSandbox();
}

FString UGamebaseCoreSubsystem::GetDeviceLanguageCode() const
{
    return Provider->GetSystemInfo()->GetDeviceLanguageCode();
}

FString UGamebaseCoreSubsystem::GetCarrierCode() const
{
    return Provider->GetSystemInfo()->GetCarrierCode();
}

FString UGamebaseCoreSubsystem::GetCarrierName() const
{
    return Provider->GetSystemInfo()->GetCarrierName();
}

FString UGamebaseCoreSubsystem::GetCountryCode() const
{
    return Provider->GetSystemInfo()->GetCountryCode();
}

FString UGamebaseCoreSubsystem::GetCountryCodeOfUSIM() const
{
    return Provider->GetSystemInfo()->GetCountryCodeOfUSIM();
}

FString UGamebaseCoreSubsystem::GetCountryCodeOfDevice() const
{
    return Provider->GetSystemInfo()->GetCountryCodeOfDevice();
}

FString UGamebaseCoreSubsystem::GetDisplayLanguageCode() const
{
    return Provider->GetSystemInfo()->GetDisplayLanguageCode();
}

IGamebaseAnalytics* UGamebaseCoreSubsystem::GetAnalytics()
{
    return Provider->GetAnalytics();
}

IGamebaseContact* UGamebaseCoreSubsystem::GetContact()
{
    return Provider->GetContact();
}

IGamebaseLaunching* UGamebaseCoreSubsystem::GetLaunching()
{
    return Provider->GetLaunching();
}

IGamebaseLogger* UGamebaseCoreSubsystem::GetLogger()
{
    return Provider->GetLogger();
}

IGamebaseNetwork* UGamebaseCoreSubsystem::GetNetwork()
{
    return Provider->GetNetwork();
}

IGamebasePurchase* UGamebaseCoreSubsystem::GetPurchase()
{
    return Provider->GetPurchase();
}

IGamebasePush* UGamebaseCoreSubsystem::GetPush()
{
    return Provider->GetPush();
}

IGamebaseUtil* UGamebaseCoreSubsystem::GetUtil()
{
    return Provider->GetUtil();
}

IGamebaseWebView* UGamebaseCoreSubsystem::GetWebView()
{
    return Provider->GetWebView();
}

IGamebaseGameNotice* UGamebaseCoreSubsystem::GetGameNotice()
{
    return Provider->GetGameNotice();
}

IGamebaseImageNotice* UGamebaseCoreSubsystem::GetImageNotice()
{
    return Provider->GetImageNotice();
}

IGamebaseTemporaryWithdrawal* UGamebaseCoreSubsystem::GetTemporaryWithdrawal()
{
    if (TemporaryWithdrawal.IsValid() == false)
    {
        TemporaryWithdrawal = MakeUnique<FGamebaseTemporaryWithdrawal>(Provider);
    }

    return TemporaryWithdrawal.Get();
}

IGamebaseTerms* UGamebaseCoreSubsystem::GetTerms()
{
    return Provider->GetTerms();
}

IGamebaseCommunity* UGamebaseCoreSubsystem::GetCommunity()
{
    return Provider->GetCommunity();
}

void UGamebaseCoreSubsystem::Initialize(FSubsystemCollectionBase& Collection)
{
    Super::Initialize(Collection);

    Provider = NewObject<UGamebaseProvider>(this, GetProviderClass());
    if (Provider)
    {
        Provider->Initialize(this);
    }
}

void UGamebaseCoreSubsystem::Deinitialize()
{
    Super::Deinitialize();
    
    if (Provider)
    {
        Provider->Deinitialize();
    }
}

void UGamebaseCoreSubsystem::LoadAdapter(const FName& Name)
{
    if (!FModuleManager::Get().IsModuleLoaded(Name))
    {
        FModuleManager::Get().LoadModule(Name);
        GAMEBASE_LOG_GLOBAL_DEBUG("%s has been loaded successfully.", *Name.ToString());
    }
}
