#include "GamebaseAdapterCollection.h"

#include "GamebaseAdapterBase.h"
#include "GamebaseDebugLogger.h"

void FGamebaseAdapterBaseCollection::Initialize(UObject* NewOuter)
{
    if (Outer != nullptr)
    {
        // already initialized
        return;
    }
    
    Outer = NewOuter;
    check(Outer);

    if (ensure(BaseType) && ensureMsgf(AdapterMap.Num() == 0, TEXT("Currently don't support repopulation of Adapter Collections.")))
    {
        check(!bPopulating); //Populating collections on multiple threads?
        
        //non-thread-safe use of Global lists, must be from GameThread:
        check(IsInGameThread());
    
        TArray<UClass*> DerivedClasses;
        GetDerivedClasses(BaseType, DerivedClasses, true);

        for (UClass* AdapterClass : DerivedClasses)
        {
            if (!AdapterClass->HasAnyClassFlags(CLASS_Abstract))
            {
                AddAndInitializeAdapter(AdapterClass);
                GAMEBASE_LOG_GLOBAL_DEBUG("Adapter %s initialized.", *AdapterClass->GetName());
            }
            else
            {
                GAMEBASE_LOG_GLOBAL_DEBUG("Skipping adapter %s because it is abstract.", *AdapterClass->GetName());
            }
        }
    }
}

void FGamebaseAdapterBaseCollection::Deinitialize()
{
}

FGamebaseAdapterBaseCollection::FGamebaseAdapterBaseCollection(const TSubclassOf<UGamebaseAdapterBase> InBaseType)
    : BaseType(InBaseType)
    , Outer(nullptr)
    , bPopulating(false)
{
    check(BaseType);
}

FGamebaseAdapterBaseCollection::FGamebaseAdapterBaseCollection()
    : Outer(nullptr)
    , bPopulating(false)
{
}

UGamebaseAdapterBase* FGamebaseAdapterBaseCollection::GetAdapterInternal(const FString& AdapterKey) const
{
    UGamebaseAdapterBase* Adapter = AdapterKeyMap.FindRef(AdapterKey);
    if (Adapter)
    {
        return Adapter;
    }
    
    return nullptr;
}

UGamebaseAdapterBase* FGamebaseAdapterBaseCollection::GetAdapterInternal(const TSubclassOf<UGamebaseAdapterBase> AdapterClass) const
{
    UGamebaseAdapterBase* Adapter = AdapterMap.FindRef(AdapterClass);
    if (Adapter)
    {
        return Adapter;
    }
    
    const TArray<UGamebaseAdapterBase*>& Adapters = GetAdapterArrayInternal(AdapterClass);
    if (Adapters.Num() > 0)
    {
        return Adapters[0];
    }

    return nullptr;
}

const TArray<UGamebaseAdapterBase*>& FGamebaseAdapterBaseCollection::GetAdapterArrayInternal(const TSubclassOf<UGamebaseAdapterBase> AdapterClass) const
{
    if (!AdapterArrayMap.Contains(AdapterClass))
    {
        TArray<UGamebaseAdapterBase*>& NewList = AdapterArrayMap.Add(AdapterClass);

        for (auto Iter = AdapterMap.CreateConstIterator(); Iter; ++Iter)
        {
            const UClass* KeyClass = Iter.Key();
            if (KeyClass->IsChildOf(AdapterClass))
            {
                NewList.Add(Iter.Value());
            }
        }

        return NewList;
    }

    const TArray<UGamebaseAdapterBase*>& List = AdapterArrayMap.FindChecked(AdapterClass);
    return List;
}

UGamebaseAdapterBase* FGamebaseAdapterBaseCollection::AddAndInitializeAdapter(UClass* AdapterClass)
{
    if (AdapterMap.Contains(AdapterClass))
    {
        return AdapterMap.FindRef(AdapterClass);
    }
    
    if (AdapterClass == nullptr || AdapterClass->HasAllClassFlags(CLASS_Abstract))
    {
        return nullptr;
    }
    
    // Catch any attempt to add a adapter of the wrong type
    checkf(AdapterClass->IsChildOf(BaseType), TEXT("ClassType (%s) must be a subclass of BaseType(%s)."), *AdapterClass->GetName(), *BaseType->GetName());
    
    // Do not create instances of classes aren't authoritative
    if (AdapterClass->GetAuthoritativeClass() != AdapterClass)
    {    
        return nullptr;
    }
    
    // Check if the module containing the adapter is loaded
    const FString ModuleName = FPackageName::GetShortFName(AdapterClass->GetOuterUPackage()->GetFName()).ToString();
    if (!FModuleManager::Get().IsModuleLoaded(*ModuleName))
    {
        GAMEBASE_LOG_GLOBAL_DEBUG("Skipping adapter %s because its module %s is not loaded.", *AdapterClass->GetName(), *ModuleName);
        return nullptr;
    }
    
    UGamebaseAdapterBase* Adapter = NewObject<UGamebaseAdapterBase>(Outer, AdapterClass);
    AdapterMap.Add(AdapterClass, Adapter);

    const FString AdapterKey = Adapter->GetUniqueKey();
    if (AdapterKey.IsEmpty() == false)
    {
        if (AdapterKeyMap.Contains(AdapterKey))
        {
            GAMEBASE_LOG_GLOBAL_WARNING("Adapters with the same key exist in the collection.")
        }
        else
        {
            AdapterKeyMap.Add(AdapterKey, Adapter);
        }
    }
    
    return Adapter;
}

void FGamebaseAdapterBaseCollection::RemoveAndDeinitializeAdapter(const UGamebaseAdapterBase* Adapter)
{
    check(Adapter);
    const UGamebaseAdapterBase* AdapterFound = AdapterMap.FindAndRemoveChecked(Adapter->GetClass());
    check(Adapter == AdapterFound);
}

void FGamebaseAdapterBaseCollection::AddReferencedObjects(FReferenceCollector& Collector)
{
    Collector.AddReferencedObjects(AdapterMap);
}

FString FGamebaseAdapterBaseCollection::GetReferencerName() const
{
    return TEXT("FGamebaseAdapterBaseCollection");
}

bool FGamebaseAdapterBaseCollection::GetReferencerPropertyName(UObject* Object, FString& OutPropertyName) const
{
    return FGCObject::GetReferencerPropertyName(Object, OutPropertyName);
}
