﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text.RegularExpressions;
using UnrealBuildTool;

#if UE_5_0_OR_LATER
using EpicGames.Core;
#else
using Tools.DotNETCommon;
#endif

public class GamebaseCore : ModuleRules
{
    public GamebaseCore(ReadOnlyTargetRules Target) : base(Target)
    {
        PCHUsage = PCHUsageMode.UseExplicitOrSharedPCHs;
        
        PublicIncludePaths.AddRange(
            new[] {
                ModuleDirectory + "/Public",
                ModuleDirectory + "/Public/Adapter",
                ModuleDirectory + "/Public/Interfaces",
                ModuleDirectory + "/Public/Utils",
            }
        );
        
        PublicDependencyModuleNames.AddRange(
            new[]
            {
                "CoreUObject",
                "Core",
                "Engine",
                "EngineSettings",
                "Gamebase",
                "GpLogger",
                "NHNShared",
                "NHNJson",
            }
        );

        PrivateDependencyModuleNames.AddRange(
            new[]
            {
                "HTTP",
                "Json",
                "JsonUtilities",
                "Projects",
            }
        );
        
        RuntimeDependencies.Add(Path.Combine(PluginDirectory, "Content", "..."), StagedFileType.UFS);
    }
}


namespace NHNCloud.Gamebase
{
    public enum SteamworksType
    {
        None,
        Native,
        OnlineSubsystem,
    }
    
    public enum EOSType
    {
        None,
        Native,
        OnlineSubsystem,
    }

    public enum Macro
    {
        WITH_GAMEBASE_EOS_SHARED,
        WITH_GAMEBASE_EOS_OSS,
        WITH_GAMEBASE_EOS_NATIVE_SDK,
        
        WITH_GAMEBASE_STEAMWORKS_SHARED,
        WITH_GAMEBASE_STEAMWORKS_OSS,
        WITH_GAMEBASE_STEAMWORKS_NATIVE_SDK,
        
        WITH_GAMEBASE_STANDALONE_PURCHASE,
        
        WITH_GAMEBASE_WEBVIEW_GAMEBASE_PLUGIN,
        WITH_GAMEBASE_WEBVIEW_ENGINE_PLUGIN
    }
    
    internal class ExternalConstants
    {
        public const int SteamworksVersion = 159;
        public static readonly Version EOSSDKVersion = new Version("1.15.5.0");
    }
    
    internal enum SupportedIdP
    {
        Google,
        GPGSv2,
        GameCenter,
        Facebook,
        Payco,
        Naver,
        Twitter,
        Line,
        AppleId,
        Hangame,
        Weibo,
        Steam,
        EpicGames
    }
    
    internal enum SupportedStore
    {
        GooglePlayStore,
        AppStore,
        GalaxyStore,
        OneStore,
        HuaweiAppGallery,
        EpicGames,
        Steam
    }
    
    public class MacroManager<TEnum>
    {
        private readonly Dictionary<TEnum, bool> Macros = new Dictionary<TEnum, bool>();
        private readonly Dictionary<TEnum, List<TEnum>> Dependencies = new Dictionary<TEnum, List<TEnum>>();
        
        public MacroManager()
        {
            if (!typeof(TEnum).IsEnum)
            {
                throw new InvalidOperationException("MacroManager<TEnum> only supports enum types.");
            }
            
            foreach (TEnum macro in Enum.GetValues(typeof(TEnum)))
            {
                Macros[macro] = false;
            }
        }

        public void Enable(TEnum macro)
        {
            Macros[macro] = true;
        }

        public void Disable(TEnum macro)
        {
            Macros[macro] = false;
        }

        public void AddDependency(TEnum from, TEnum to)
        {
            if (!Dependencies.ContainsKey(from))
            {
                Dependencies[from] = new List<TEnum>();
            }

            if (!Dependencies[from].Contains(to))
            {
                Dependencies[from].Add(to);
            }
        }
        
        public void Finalize(ModuleRules rules)
        {
            ResolveAllDependencies();

            rules.PublicDefinitions.AddRange(
                Macros.Select(kv => string.Format("{0}={1}", kv.Key, kv.Value ? 1 : 0))
            );

            if (Macros.Any(kv => kv.Value))
            {
                Logger.LogInfo("{0}:\n{1}", typeof(TEnum).Name, ToString());
            }
        }
        
        public override string ToString()
        {
            return string.Join("\n", Macros
                .Where(kv => kv.Value)
                .Select(kv => "  - " + kv.Key)
                .OrderBy(macro => macro));
        }
        
        private void ResolveAllDependencies()
        {
            var visited = new HashSet<TEnum>();
            var queue = new Queue<TEnum>(Macros.Where(kv => kv.Value).Select(kv => kv.Key));

            while (queue.Count > 0)
            {
                TEnum current = queue.Dequeue();
                List<TEnum> dependencies;
                if (!Dependencies.TryGetValue(current, out dependencies)) continue;
                
                foreach (TEnum dependent in dependencies.Where(dependent => !visited.Contains(dependent)))
                {
                    Macros[dependent] = true;
                    visited.Add(dependent);
                    queue.Enqueue(dependent);
                }
            }
        }

        public bool GetMacroState(TEnum macro)
        {
            bool State;
            if (Macros.TryGetValue(macro, out State))
            {
                return State;
            }
            throw new ArgumentException("Invalid macro specified.");
        }
    }
    
    public class GamebaseRuntime
    {
        private static GamebaseRuntime _instance;

        private readonly EngineManager _engineManager;

        private readonly MacroManager<Macro> _macros = new MacroManager<Macro>();

        private readonly List<SupportedIdP> _authProviders = new List<SupportedIdP>();
        private readonly List<SupportedStore> _storeProviders = new List<SupportedStore>();

        private readonly EOSType EOSSupportType;
        private readonly SteamworksType SteamworksSupportType;

        public bool IsWindowsIAPEnabled
        {
            get
            {
                if (_engineManager.Target.Platform != UnrealTargetPlatform.Win64 || !_storeProviders.Any())
                {
                    return false;
                }
                
                var storeCheckers = new Dictionary<SupportedStore, Func<bool>>
                {
                    { SupportedStore.EpicGames, () => EOSSupportType != EOSType.None },
                    { SupportedStore.Steam, () => SteamworksSupportType != SteamworksType.None }
                };
                
                return _storeProviders.Any(store =>
                {
                    Func<bool> isAvailable;
                    return !storeCheckers.TryGetValue(store, out isAvailable) || isAvailable();
                });
            }
        }

        public static GamebaseRuntime Instance
        {
            get
            {
                if (_instance == null)
                {
                    throw new BuildException("Instance not initialized! Call Initialize() first.");
                }
                return _instance;
            }
        }

        public static void Initialize(ModuleRules GamebaseModuleRules)
        {
            if (_instance != null)
            {
                return;
            }

            _instance = new GamebaseRuntime(GamebaseModuleRules);
        }
        
        public bool IsMacroEnabled(Macro macro)
        {
            return _macros.GetMacroState(macro);
        }

        private GamebaseRuntime(ModuleRules GamebaseModuleRules)
        {
            _engineManager = new EngineManager(GamebaseModuleRules.Target);

            _macros.AddDependency(Macro.WITH_GAMEBASE_EOS_NATIVE_SDK, Macro.WITH_GAMEBASE_EOS_SHARED);
            _macros.AddDependency(Macro.WITH_GAMEBASE_EOS_OSS, Macro.WITH_GAMEBASE_EOS_SHARED);
            
            _macros.AddDependency(Macro.WITH_GAMEBASE_STEAMWORKS_NATIVE_SDK, Macro.WITH_GAMEBASE_STEAMWORKS_SHARED);
            _macros.AddDependency(Macro.WITH_GAMEBASE_STEAMWORKS_OSS, Macro.WITH_GAMEBASE_STEAMWORKS_SHARED);

            LoadConfigInfo();
            
            if (_authProviders.Any())
            {
                Logger.LogInfo("AuthProviders:\n  - {0}", string.Join("\n  - ", _authProviders.OrderBy(IdP => IdP)));
            }
            
            if (_storeProviders.Any())
            {
                Logger.LogInfo("Stores:\n  - {0}", string.Join("\n  - ", _storeProviders.OrderBy(Store => Store)));
            }
            
            if (IsEOSSDKRequired)
            {
                EOSRuntime EOSRuntime = new EOSRuntime(_engineManager);
                if (EOSRuntime.IsEOSSDKEnabled)
                {
                    EOSSupportType = EOSRuntime.IsOnlineSubsystemEnabled ? EOSType.OnlineSubsystem : EOSType.Native;
                    _macros.Enable(EOSSupportType.Equals(EOSType.OnlineSubsystem)
                        ? Macro.WITH_GAMEBASE_EOS_OSS
                        : Macro.WITH_GAMEBASE_EOS_NATIVE_SDK);
                }
                
                Logger.LogInfo("EOSSDK:\n  - Version: {0}\n  - Type: {1}", EOSRuntime.Version, EOSSupportType);
            }

            if (IsSteamworksRequired)
            {
                SteamworksRuntime SteamworksRuntime = new SteamworksRuntime(_engineManager);
                if (SteamworksRuntime.IsSteamworksEnabled)
                {
                    SteamworksSupportType = SteamworksRuntime.IsOnlineSubsystemEnabled ? SteamworksType.OnlineSubsystem : SteamworksType.Native;
                    _macros.Enable(SteamworksSupportType.Equals(SteamworksType.OnlineSubsystem)
                        ? Macro.WITH_GAMEBASE_STEAMWORKS_OSS
                        : Macro.WITH_GAMEBASE_STEAMWORKS_NATIVE_SDK);
                }
                
                Logger.LogInfo("Steamworks:\n  - Version: {0}\n  - Type: {1}", SteamworksRuntime.Version, SteamworksSupportType);
            }

            if (IsWindowsIAPEnabled)
            {
                _macros.Enable(Macro.WITH_GAMEBASE_STANDALONE_PURCHASE);
            }

            if (GamebaseModuleRules.Target.Platform == UnrealTargetPlatform.Win64)
            {
                bool IsWebBrowserWidgetPlugin = _engineManager.EnablePlugin("WebBrowserWidget");
                bool IsNhnWebViewPlugin = _engineManager.EnablePlugin("NHNWebView");

                if (IsWebBrowserWidgetPlugin && IsNhnWebViewPlugin)
                {
                    throw new BuildException("The WebBrowserWidget plugin and NHNWebView plugin cannot be used together. Please select one or the other.");
                }

                if (IsWebBrowserWidgetPlugin)
                {
                    _macros.Enable(Macro.WITH_GAMEBASE_WEBVIEW_ENGINE_PLUGIN);
                }

                if (IsNhnWebViewPlugin)
                {
                    _macros.Enable(Macro.WITH_GAMEBASE_WEBVIEW_GAMEBASE_PLUGIN);
                }
            }

            _macros.Finalize(GamebaseModuleRules);
        }

        private void LoadConfigInfo()
        {
            ConfigHierarchy Config = ConfigCache.ReadHierarchy(
                ConfigHierarchyType.Engine,
                DirectoryReference.FromFile(_engineManager.Target.ProjectFile),
                _engineManager.Target.Platform);
            
            foreach (SupportedIdP ProviderName in Enum.GetValues(typeof(SupportedIdP)))
            {
                bool IsEnableFlag;
                if ((!Config.GetBool(ConfigSection, string.Format("bEnable{0}", ProviderName), out IsEnableFlag) || !IsEnableFlag) &&
                    (!Config.GetBool(ConfigSection, string.Format("bEnableAuth{0}", ProviderName), out IsEnableFlag) || !IsEnableFlag))
                {
                    continue;
                }
                
                _authProviders.Add(ProviderName);
            }
            
            string SelectedStore;
            if (Config.GetString(ConfigSection, string.Format("{0}Store", PlatformString), out SelectedStore))
            {
                var Store = GetEnumValue<SupportedStore>(SelectedStore);
                if (Store.HasValue)
                {
                    _storeProviders.Add(Store.Value);
                }
            }
            else
            {
                foreach (SupportedStore Store in Enum.GetValues(typeof(SupportedStore)))
                {
                    if (!IsStoreEnabled(Config, ConfigSection, Store))
                    {
                        continue;
                    }
                
                    _storeProviders.Add(Store);
                }
            }
        }
        
        public static TEnum? GetEnumValue<TEnum>(string value) where TEnum : struct
        {
            TEnum result;
            if (Enum.TryParse(value, out result))
            {
                return result;
            }
            return null;
        }

        private static List<string> ExtractConstantsFromHeader(string HeaderFilePath)
        {
            if (string.IsNullOrWhiteSpace(HeaderFilePath))
            {
                throw new ArgumentException("Header file path cannot be null or empty.", "HeaderFilePath");
            }

            if (!File.Exists(HeaderFilePath))
            {
                return new List<string>();
            }

            try
            {
                Regex Regex = new Regex(@"const FString\s+(\w+)\s*\(");

                return File.ReadAllLines(HeaderFilePath)
                    .Select(Line => Regex.Match(Line))
                    .Where(Match => Match.Success)
                    .Select(Match => Match.Groups[1].Value)
                    .ToList();
            }
            catch (IOException Ex)
            {
                Log.TraceWarning(string.Format("Failed to read constants from header file: {0}. Exception: {1}", HeaderFilePath, Ex.Message));
                return new List<string>();
            }
            catch (UnauthorizedAccessException Ex)
            {
                Log.TraceWarning(string.Format("Access denied to header file: {0}. Exception: {1}", HeaderFilePath, Ex.Message));
                return new List<string>();
            }
        }
        
        private bool IsEOSSDKRequired
        {
            get
            {
                return _authProviders.Contains(SupportedIdP.EpicGames) || _storeProviders.Contains(SupportedStore.EpicGames);
            }
        }

        private bool IsSteamworksRequired
        {
            get
            {
                if (_engineManager.Target.Type == TargetType.Editor)
                {
                    Logger.LogInfo("Steamworks is not required in Editor mode.");
                    return false;
                }

                if (_engineManager.Target.Platform == UnrealTargetPlatform.Win64)
                {
                    return _authProviders.Contains(SupportedIdP.Steam) || _storeProviders.Contains(SupportedStore.Steam);
                }

                return false;
            }
        }
        
        private string PlatformString
        {
            get
            {
                if (_engineManager.Target.Platform == UnrealTargetPlatform.Win64)
                {
                    return "Windows";
                }

                if (_engineManager.Target.Platform == UnrealTargetPlatform.Android ||
                    _engineManager.Target.Platform == UnrealTargetPlatform.IOS ||
                    _engineManager.Target.Platform == UnrealTargetPlatform.Mac ||
                    _engineManager.Target.Platform == UnrealTargetPlatform.Linux)
                {
                    return _engineManager.Target.Platform.ToString();
                }

                throw new BuildException("Unsupported platform: " + _engineManager.Target.Platform);
            }
        }
        
        private string ConfigSection
        {
            get
            {
                return string.Format("/Script/GamebaseEditor.Gamebase{0}PlatformSettings", PlatformString); 
            }
        }
        
        private static bool IsStoreEnabled(ConfigHierarchy config, string configSection, SupportedStore store)
        {
            bool isEnableFlag;
            return (config.GetBool(configSection, string.Format("bEnable{0}", store), out isEnableFlag) && isEnableFlag) ||
                   (config.GetBool(configSection, string.Format("bEnablePurchase{0}", store), out isEnableFlag) && isEnableFlag);
        }
    }
    
    internal class EngineManager
    {
        private readonly ReadOnlyTargetRules _target;
        private readonly List<PluginInfo> _cachedPlugins;

        public ReadOnlyTargetRules Target
        {
            get { return _target; }
        }
        
        public EngineManager(ReadOnlyTargetRules Target)
        {
            _target = Target;
            _cachedPlugins = Plugins.ReadAvailablePlugins(new DirectoryReference(Target.RelativeEnginePath), Target.ProjectFile.Directory, null).ToList();
        }
        
        public bool EnablePlugin(string PluginName)
        {
            PluginInfo PluginInfo = _cachedPlugins.Find(Info => Info.Name.Equals(PluginName, StringComparison.InvariantCultureIgnoreCase));
            return PluginInfo != null && Plugins.IsPluginEnabledForTarget(PluginInfo, ProjectDescriptor.FromFile(_target.ProjectFile), _target.Platform, _target.Configuration, _target.Type);
        }
    }
    
    public static class Logger
    {
        public static void LogInfo(string Format, params object[] Args)
        {
            Log.TraceInformation("Gamebase: " + string.Format(Format, Args));
        }
    }
    
    internal class EOSRuntime
    {
        public readonly bool IsOnlineSubsystemEnabled;
        public readonly Version Version;
        private readonly UnrealTargetPlatform _platform;

        public bool IsEOSSDKEnabled
        {
            get
            {
                if (_platform == UnrealTargetPlatform.Win64)
                {
                    return Version != null && Version.CompareTo(ExternalConstants.EOSSDKVersion) > 0;
                }
                
                return _platform == UnrealTargetPlatform.Android || _platform == UnrealTargetPlatform.IOS;
            }
        }
        
        public EOSRuntime(EngineManager Engine)
        {
            _platform = Engine.Target.Platform;
            
            Version = GetEOSVersion(Engine.Target);
            IsOnlineSubsystemEnabled = CheckOnlineSubsystemEnabled(Engine);
        }

        private static bool CheckOnlineSubsystemEnabled(EngineManager engine)
        {
            if (!engine.EnablePlugin("OnlineSubsystemEOS"))
            {
                return false;
            }

            bool IsEnableConfig;
            ConfigHierarchy Config = ConfigCache.ReadHierarchy(ConfigHierarchyType.Engine, DirectoryReference.FromFile(engine.Target.ProjectFile), UnrealTargetPlatform.Win64);
            Config.GetBool("OnlineSubsystemEOS", "bEnabled", out IsEnableConfig);
            return IsEnableConfig;
        }
        
        private static Version GetEOSVersion(ReadOnlyTargetRules Target)
        {
            if (Target.Platform != UnrealTargetPlatform.Win64)
            {
                return null;
            }

            string SDKBinariesDir = Path.Combine(Target.UEThirdPartySourceDirectory, "EOSSDK", "SDK", "Bin"); 
            string RuntimeLibraryFileName = string.Format("EOSSDK-{0}-Shipping.dll", Target.Platform.ToString());

            var Filepath = Path.GetFullPath(Path.Combine(SDKBinariesDir, RuntimeLibraryFileName));
            FileVersionInfo VersionInfo = FileVersionInfo.GetVersionInfo(Filepath);
            Version FileVersion = new Version(VersionInfo.FileMajorPart, VersionInfo.FileMinorPart, VersionInfo.FileBuildPart,
                VersionInfo.FilePrivatePart);
            
            return FileVersion;
        }
    }
    
    internal class SteamworksRuntime
    {
        public readonly bool IsSteamworksEnabled;
        public readonly bool IsOnlineSubsystemEnabled;
        public readonly int Version;

        public SteamworksRuntime(EngineManager engine)
        {
            Version = FindLastSteamworksVersion(engine.Target.UEThirdPartySourceDirectory + "Steamworks");
            IsSteamworksEnabled = Version >= ExternalConstants.SteamworksVersion;
            IsOnlineSubsystemEnabled = CheckOnlineSubsystemEnabled(engine);
        }

        private static bool CheckOnlineSubsystemEnabled(EngineManager engine)
        {
            if (!engine.EnablePlugin("OnlineSubsystemSteam"))
            {
                return false;
            }

            bool IsEnableConfig;
            ConfigHierarchy Config = ConfigCache.ReadHierarchy(ConfigHierarchyType.Engine, DirectoryReference.FromFile(engine.Target.ProjectFile), UnrealTargetPlatform.Win64);
            Config.GetBool("OnlineSubsystemSteam", "bEnabled", out IsEnableConfig);
            return IsEnableConfig;
        }
        
        private static int FindLastSteamworksVersion(string Path)
        {
            if (!Directory.Exists(Path))
            {
                throw new DirectoryNotFoundException(string.Format("Directory does not exist: {0}", Path));
            }
            
            int MaxNumber = int.MinValue;
            Regex NumberPattern = new Regex(@"\d+");

            foreach (var FolderPath in Directory.GetDirectories(Path))
            {
                string FolderName = System.IO.Path.GetFileName(FolderPath);
                var Matches = NumberPattern.Matches(FolderName);

                foreach (Match Match in Matches)
                {
                    int Number;
                    if (int.TryParse(Match.Value, out Number))
                    {
                        MaxNumber = Math.Max(MaxNumber, Number);
                    }
                }
            }

            return MaxNumber == int.MinValue ? 0 : MaxNumber;
        }
    }
}
