#include "GamebaseBridgeWebView.h"
#include "GamebaseCommunicationRequestTypes.h"

namespace GamebaseScheme
{
    const FName OpenWebBrowser(TEXT("gamebase://openWebBrowser"));
    const FName ShowWebView(TEXT("gamebase://showWebView"));
    const FName CloseWebView(TEXT("gamebase://closeWebView"));
    const FName WebViewSchemeEvent(TEXT("gamebase://schemeEvent"));
}

FGamebaseBridgeWebView::FGamebaseBridgeWebView(const TSharedPtr<FGamebaseCommunicatorManager>& Communicator, const FName& ClassName)
    : FGamebaseBridgeBase(Communicator, ClassName)
{
    Communicator->RegisterEvent(GamebaseScheme::WebViewSchemeEvent, [this](const FGamebaseCommunicatorReceiveDataPtr& ReceiveData)
    {
        if (OnSchemeEventMap.Contains(ReceiveData->Handle))
        {
            OnSchemeEventMap[ReceiveData->Handle].ExecuteIfBound(ReceiveData->JsonData, GetGamebaseError(ReceiveData.ToSharedRef()).Get());
        }
    });
}

void FGamebaseBridgeWebView::OpenWebBrowser(const FString& Url)
{
    FGamebaseCommunicatorSendData SendData{ GamebaseScheme::OpenWebBrowser };
    SendData.JsonData = Url;

    Communicator->GetAsync(SendData);
}

void FGamebaseBridgeWebView::ShowWebView(
    const FString& Url,
    const FGamebaseWebViewConfiguration& Configuration,
    const FGamebaseErrorDelegate& CloseCallback,
    const TArray<FString>& SchemeList,
    const FGamebaseSchemeEventDelegate& SchemeEvent)
{
    FWebviewConfiguration ConfigData;
    ConfigData.Url = Url;
    ConfigData.Configuration = Configuration;

    FWebviewSchemeExtraData SchemeData;
    SchemeData.SchemeList = SchemeList;
    SchemeData.SchemeEvent = RegisterSchemeEvent(SchemeEvent);
    
    FGamebaseCommunicatorSendData SendData { GamebaseScheme::ShowWebView };
    SendData.JsonData = ConfigData.ToJson();
    SendData.ExtraData = SchemeData.ToJson();

    Communicator->GetAsync(SendData, CloseCallback, [this](const FString& ExtraData) {
        UnregisterSchemeEvent(ExtraData);
    });
}

void FGamebaseBridgeWebView::CloseWebView()
{
    const FGamebaseCommunicatorSendData SendData{ GamebaseScheme::CloseWebView };

    Communicator->GetAsync(SendData);
}

int32 FGamebaseBridgeWebView::RegisterSchemeEvent(const FGamebaseSchemeEventDelegate& SchemeEvent)
{
    static FThreadSafeCounter SchemeHandleCounter;
    
    const int32 SchemeHandle = SchemeHandleCounter.Increment();
    OnSchemeEventMap.Emplace(SchemeHandle, SchemeEvent);
    
    return SchemeHandle;
}

void FGamebaseBridgeWebView::UnregisterSchemeEvent(const FString& ExtraData)
{
    const int32 SchemeHandle = FCString::Atoi(*ExtraData);
    OnSchemeEventMap.Remove(SchemeHandle);
}
