﻿#include "GamebaseBridgeProvider.h"
#include "GamebasePlatformDefines.h"

// Mobile platforms don't account for multiple game instances
UGamebaseBridgeProvider* CachedProvider = nullptr;

UGamebaseBridgeProvider* UGamebaseBridgeProvider::GetInstance()
{
    return CachedProvider;
}

void UGamebaseBridgeProvider::Initialize(UGameInstanceSubsystem* Subsystem)
{
    CachedProvider = this;

    Super::Initialize(Subsystem);
}

void UGamebaseBridgeProvider::Deinitialize()
{
    Super::Deinitialize();
    
#define GAMEBASE_MODULE_RESET(Module)      Module.Reset();
    GAMEBASE_MODULE_RESET(Core);
    GAMEBASE_MODULE_RESET(Analytics);
    GAMEBASE_MODULE_RESET(Auth);
    GAMEBASE_MODULE_RESET(Contact);
    GAMEBASE_MODULE_RESET(GameNotice);
    GAMEBASE_MODULE_RESET(EventHandler);
    GAMEBASE_MODULE_RESET(ImageNotice);
    GAMEBASE_MODULE_RESET(Launching);
    GAMEBASE_MODULE_RESET(Network);
    GAMEBASE_MODULE_RESET(Purchase);
    GAMEBASE_MODULE_RESET(Push);
    GAMEBASE_MODULE_RESET(SystemInfo);
    GAMEBASE_MODULE_RESET(Terms);
    GAMEBASE_MODULE_RESET(Util);
    GAMEBASE_MODULE_RESET(WebView);
    GAMEBASE_MODULE_RESET(Community);
    
    CachedProvider = nullptr;
}

void UGamebaseBridgeProvider::InitializeGamebase(const FGamebaseConfiguration& Configuration, const FGamebaseLaunchingInfoDelegate& Callback)
{
    Super::InitializeGamebase(Configuration, Callback);
    
    Core->SetDebugMode(bIsDebugMode);
    Core->InitializeGamebase(Configuration, FGamebaseLaunchingInfoDelegate::CreateLambda(
        [&, Callback](const FGamebaseLaunchingInfo* LaunchingInfo, const FGamebaseError* Error)
        {
            if (LaunchingInfo != nullptr)
            {
                OnInitializeGamebaseCompleted(*LaunchingInfo);
            }

            Callback.ExecuteIfBound(LaunchingInfo, Error);
        }));
}

void UGamebaseBridgeProvider::SetDebugMode(const bool bDebugMode)
{
    Super::SetDebugMode(bDebugMode);
    
    if (IsCreateModule())
    {
        Core->SetDebugMode(bDebugMode);
    }
}

void UGamebaseBridgeProvider::Login(const FString& ProviderName, const FGamebaseAuthTokenDelegate& Callback)
{
    Auth->Login(ProviderName, Callback);
}

void UGamebaseBridgeProvider::Login(const FGamebaseVariantMap& CredentialInfo, const FGamebaseAuthTokenDelegate& Callback)
{
    Auth->Login(CredentialInfo, Callback);
}

void UGamebaseBridgeProvider::Login(const FString& ProviderName, const FGamebaseVariantMap& AdditionalInfo,
    const FGamebaseAuthTokenDelegate& Callback)
{
    Auth->Login(ProviderName, AdditionalInfo, Callback);
}

void UGamebaseBridgeProvider::LoginForLastLoggedInProvider(const FGamebaseAuthTokenDelegate& Callback)
{
    Auth->LoginForLastLoggedInProvider(Callback);
}

void UGamebaseBridgeProvider::LoginForLastLoggedInProvider(const FGamebaseVariantMap& AdditionalInfo, const FGamebaseAuthTokenDelegate& Callback)
{
    Auth->LoginForLastLoggedInProvider(AdditionalInfo, Callback);
}

void UGamebaseBridgeProvider::CancelLoginWithExternalBrowser(const FGamebaseErrorDelegate& Callback)
{
    GAMEBASE_NOT_SUPPORT_API();
}

void UGamebaseBridgeProvider::AddMapping(const FString& ProviderName, const FGamebaseAuthTokenDelegate& Callback)
{
    Auth->AddMapping(ProviderName, Callback);
}

void UGamebaseBridgeProvider::AddMapping(const FString& ProviderName, const FGamebaseVariantMap& AdditionalInfo,
    const FGamebaseAuthTokenDelegate& Callback)
{
    Auth->AddMapping(ProviderName, AdditionalInfo, Callback);
}

void UGamebaseBridgeProvider::AddMapping(const FGamebaseVariantMap& CredentialInfo, const FGamebaseAuthTokenDelegate& Callback)
{
    Auth->AddMapping(CredentialInfo, Callback);
}

void UGamebaseBridgeProvider::RemoveMapping(const FString& ProviderName, const FGamebaseErrorDelegate& Callback)
{
    Auth->RemoveMapping(ProviderName, Callback);
}

void UGamebaseBridgeProvider::AddMappingForcibly(const FGamebaseForcingMappingTicket& ForcingMappingTicket,
    const FGamebaseAuthTokenDelegate& Callback)
{
    Auth->AddMappingForcibly(ForcingMappingTicket, Callback);
}

void UGamebaseBridgeProvider::AddMappingForcibly(const FString& ProviderName, const FString& ForcingMappingKey,
    const FGamebaseAuthTokenDelegate& Callback)
{
    Auth->AddMappingForcibly(ProviderName, ForcingMappingKey, Callback);
}

void UGamebaseBridgeProvider::AddMappingForcibly(const FString& ProviderName, const FString& ForcingMappingKey,
    const FGamebaseVariantMap& AdditionalInfo, const FGamebaseAuthTokenDelegate& Callback)
{
    Auth->AddMappingForcibly(ProviderName, ForcingMappingKey, AdditionalInfo, Callback);
}

void UGamebaseBridgeProvider::AddMappingForcibly(const FGamebaseVariantMap& CredentialInfo, const FString& ForcingMappingKey,
    const FGamebaseAuthTokenDelegate& Callback)
{
    Auth->AddMappingForcibly(CredentialInfo, ForcingMappingKey, Callback);
}

void UGamebaseBridgeProvider::ChangeLogin(const FGamebaseForcingMappingTicket& ForcingMappingTicket, const FGamebaseAuthTokenDelegate& Callback)
{
    Auth->ChangeLogin(ForcingMappingTicket, Callback);
}

void UGamebaseBridgeProvider::Logout(const FGamebaseErrorDelegate& Callback)
{
    Auth->Logout(Callback);
}

void UGamebaseBridgeProvider::Withdraw(const FGamebaseErrorDelegate& Callback)
{
    Auth->Withdraw(Callback);
}

void UGamebaseBridgeProvider::WithdrawImmediately(const FGamebaseErrorDelegate& Callback)
{
    Auth->WithdrawImmediately(Callback);
}

void UGamebaseBridgeProvider::RequestTemporaryWithdrawal(const FGamebaseTemporaryWithdrawalDelegate& Callback)
{
    Auth->RequestTemporaryWithdrawal(Callback);
}

void UGamebaseBridgeProvider::CancelTemporaryWithdrawal(const FGamebaseErrorDelegate& Callback)
{
    Auth->CancelTemporaryWithdrawal(Callback);
}

void UGamebaseBridgeProvider::IssueTransferAccount(const FGamebaseTransferAccountDelegate& Callback)
{
    Auth->IssueTransferAccount(Callback);
}

void UGamebaseBridgeProvider::QueryTransferAccount(const FGamebaseTransferAccountDelegate& Callback)
{
    Auth->QueryTransferAccount(Callback);
}

void UGamebaseBridgeProvider::RenewTransferAccount(const FGamebaseTransferAccountRenewConfiguration& Configuration,
    const FGamebaseTransferAccountDelegate& Callback)
{
    Auth->RenewTransferAccount(Configuration, Callback);
}

void UGamebaseBridgeProvider::TransferAccountWithIdPLogin(const FString& AccountId, const FString& AccountPassword,
    const FGamebaseAuthTokenDelegate& Callback)
{
    Auth->TransferAccountWithIdPLogin(AccountId, AccountPassword, Callback);
}

TArray<FString> UGamebaseBridgeProvider::GetAuthMappingList()
{
    return Auth->GetAuthMappingList();
}

FString UGamebaseBridgeProvider::GetAuthProviderUserID(const FString& ProviderName)
{
    return Auth->GetAuthProviderUserID(ProviderName);
}

FString UGamebaseBridgeProvider::GetAuthProviderAccessToken(const FString& ProviderName)
{
    return Auth->GetAuthProviderAccessToken(ProviderName);
}

FGamebaseAuthProviderProfilePtr UGamebaseBridgeProvider::GetAuthProviderProfile(const FString& ProviderName)
{
    return Auth->GetAuthProviderProfile(ProviderName);
}

FGamebaseBanInfoPtr UGamebaseBridgeProvider::GetBanInfo()
{
    return Auth->GetBanInfo();
}

void UGamebaseBridgeProvider::OnInitializeGamebaseCompleted(const FGamebaseLaunchingInfo& LaunchingInfo)
{
    UGamebaseProvider::OnInitializeGamebaseCompleted(LaunchingInfo);
}

void UGamebaseBridgeProvider::CreateInternalModule()
{
    CommunicatorManager = MakeShared<FGamebaseCommunicatorManager>();

    UGamebaseProvider::CreateInternalModule();
    
#define GAMEBASE_MODULE_CREATE(Module)      Module = MakeShared<FGamebase##Module>(CommunicatorManager);
    GAMEBASE_MODULE_CREATE(Core);
    GAMEBASE_MODULE_CREATE(Analytics);
    GAMEBASE_MODULE_CREATE(Auth);
    GAMEBASE_MODULE_CREATE(Contact);
    GAMEBASE_MODULE_CREATE(GameNotice);
    GAMEBASE_MODULE_CREATE(EventHandler);
    GAMEBASE_MODULE_CREATE(ImageNotice);
    GAMEBASE_MODULE_CREATE(Launching);
    GAMEBASE_MODULE_CREATE(Network);
    GAMEBASE_MODULE_CREATE(Purchase);
    GAMEBASE_MODULE_CREATE(Push);
    GAMEBASE_MODULE_CREATE(SystemInfo);
    GAMEBASE_MODULE_CREATE(Terms);
    GAMEBASE_MODULE_CREATE(Util);
    GAMEBASE_MODULE_CREATE(WebView);
    GAMEBASE_MODULE_CREATE(Community);
}

bool UGamebaseBridgeProvider::ShouldCreateModule() const
{
    return Core.IsValid() == false;
}

bool UGamebaseBridgeProvider::IsCreateModule() const
{
    return Core.IsValid();
}
