#include "GamebaseBridgeImageNotice.h"
#include "GamebaseCommunicationRequestTypes.h"

namespace GamebaseScheme
{
    const FName ShowImageNotices(TEXT("gamebase://showImageNotices"));
    const FName CloseImageNotices(TEXT("gamebase://closeImageNotices"));
    const FName NoticeSchemeEvent(TEXT("gamebase://schemeEventImageNotices"));
}

FGamebaseBridgeImageNotice::FGamebaseBridgeImageNotice(const TSharedPtr<FGamebaseCommunicatorManager>& Communicator, const FName& ClassName)
    : FGamebaseBridgeBase(Communicator, ClassName)
{
    Communicator->RegisterEvent(GamebaseScheme::NoticeSchemeEvent, [this](const FGamebaseCommunicatorReceiveDataPtr& ReceiveData)
    {
        if (OnSchemeEventMap.Contains(ReceiveData->Handle))
        {
            OnSchemeEventMap[ReceiveData->Handle].ExecuteIfBound(ReceiveData->JsonData, GetGamebaseError(ReceiveData.ToSharedRef()).Get());
        }
    });
}

void FGamebaseBridgeImageNotice::ShowImageNotices(const FGamebaseImageNoticeConfiguration& Configuration, const FGamebaseErrorDelegate& CloseCallback, const FGamebaseImageNoticeEventDelegate& EventCallback)
{
    FImageNoticeSchemeExtraData ExtraData;
    ExtraData.SchemeEvent = RegisterSchemeEvent(EventCallback);

    FGamebaseCommunicatorSendData SendData{ GamebaseScheme::ShowImageNotices };
    SendData.JsonData = Configuration.ToJson();
    SendData.ExtraData = ExtraData.ToJson(false);
    
    Communicator->GetAsync(SendData, CloseCallback, [this](const FString& ExtraData) {
        UnregisterSchemeEvent(ExtraData);
    });
}

void FGamebaseBridgeImageNotice::CloseImageNotices()
{
    const FGamebaseCommunicatorSendData SendData{ GamebaseScheme::CloseImageNotices };

    Communicator->GetAsync(SendData);
}

int32 FGamebaseBridgeImageNotice::RegisterSchemeEvent(const FGamebaseImageNoticeEventDelegate& SchemeEvent)
{
    static FThreadSafeCounter SchemeHandleCounter;
    
    const int32 SchemeHandle = SchemeHandleCounter.Increment();
    OnSchemeEventMap.Emplace(SchemeHandle, SchemeEvent);
    
    return SchemeHandle;
}

void FGamebaseBridgeImageNotice::UnregisterSchemeEvent(const FString& ExtraData)
{
    const int32 SchemeHandle = FCString::Atoi(*ExtraData);
    OnSchemeEventMap.Remove(SchemeHandle);
}
