#include "GamebaseBridgeAuth.h"

#include "GamebaseBridgeAuthAdapter.h"
#include "GamebaseBridgeSubsystem.h"
#include "GamebaseCommunicationRequestTypes.h"
#include "GamebaseCommunicatorManager.h"
#include "GamebaseDebugLogger.h"

namespace GamebaseScheme
{
    const FName Login(TEXT("gamebase://login"));
    const FName LoginWithAdditionalInfo(TEXT("gamebase://loginWithAdditionalInfo"));
    const FName LoginWithCredentialInfo(TEXT("gamebase://loginWithCredentialInfo"));
    const FName LoginForLastLoggedInProvider(TEXT("gamebase://loginForLastLoggedInProvider"));
    const FName LoginForLastLoggedInProviderWithAdditionalInfo(TEXT("gamebase://loginForLastLoggedInProviderWithAdditionalInfo"));
    const FName Logout(TEXT("gamebase://logout"));

    const FName Withdraw(TEXT("gamebase://withdraw"));
    const FName WithdrawImmediately(TEXT("gamebase://withdrawImmediately"));
    const FName RequestTemporaryWithdrawal(TEXT("gamebase://requestTemporaryWithdrawal"));
    const FName CancelTemporaryWithdrawal(TEXT("gamebase://cancelTemporaryWithdrawal"));

    const FName AddMapping(TEXT("gamebase://addMapping"));
    const FName AddMappingWithCredentialInfo(TEXT("gamebase://addMappingWithCredentialInfo"));
    const FName AddMappingWithAdditionalInfo(TEXT("gamebase://addMappingWithAdditionalInfo"));

    const FName ChangeLogin(TEXT("gamebase://changeLoginWithForcingMappingTicket"));
    const FName AddMappingForciblyWithForcingMappingTicket(TEXT("gamebase://addMappingForciblyWithForcingMappingTicket"));
    
    const FName AddMappingForcibly(TEXT("gamebase://addMappingForcibly"));
    const FName AddMappingForciblyWithCredentialInfo(TEXT("gamebase://addMappingForciblyWithCredentialInfo"));
    const FName AddMappingForciblyWithAdditionalInfo(TEXT("gamebase://addMappingForciblyWithAdditionalInfo"));
    const FName RemoveMapping(TEXT("gamebase://removeMapping"));

    const FName IssueTransferAccount(TEXT("gamebase://issueTransferAccount"));
    const FName QueryTransferAccount(TEXT("gamebase://queryTransferAccount"));
    const FName RenewTransferAccount(TEXT("gamebase://renewTransferAccount"));
    const FName TransferAccountWithIdPLogin(TEXT("gamebase://transferAccountWithIdPLogin"));

    const FName GetAuthMappingList(TEXT("gamebase://getAuthMappingList"));
    const FName GetAuthProviderUserID(TEXT("gamebase://getAuthProviderUserID"));
    const FName GetAuthProviderAccessToken(TEXT("gamebase://getAuthProviderAccessToken"));
    const FName GetAuthProviderProfile(TEXT("gamebase://getAuthProviderProfile"));
    const FName GetBanInfo(TEXT("gamebase://getBanInfo"));
}

FGamebaseBridgeAuth::FGamebaseBridgeAuth(const TSharedPtr<FGamebaseCommunicatorManager>& Communicator, const FName& ClassName)
    : FGamebaseBridgeBase(Communicator, ClassName)
{
    if (UGamebaseBridgeSubsystem* Subsystem = GetSubsystem())
    {
        LoadedAdapters.Initialize(Subsystem);

        LoadedAdapters.ForEach([](UGamebaseBridgeAuthAdapter* Adapter)
        {
            if (Adapter)
            {
                Adapter->Initialize();
            }
        });

        GAMEBASE_LOG_GLOBAL_WARNING("Number of loaded adapters: %d", LoadedAdapters.GetAdapterArray<UGamebaseBridgeAuthAdapter>(UGamebaseBridgeAuthAdapter::StaticClass()).Num());
    }
    else
    {
        GAMEBASE_LOG_GLOBAL_WARNING("Failed to initialize FGamebaseBridgeAuth: Subsystem is invalid.");
    }
}

void FGamebaseBridgeAuth::Login(const FString& ProviderName, const FGamebaseAuthTokenDelegate& Callback)
{
    if (const auto Adapter = LoadedAdapters.GetAdapter(ProviderName))
    {
        Adapter->Login([this, Callback](const FGamebaseAuthCredentialsResult& Result)
        {
            if (Result.IsError())
            {
                Callback.ExecuteIfBound(nullptr, &Result.GetErrorValue());
                return;
            }
            
            Login(Result.GetOkValue(), Callback);
        });
        
        return;
    }
    
    FAuthProviderInfo LoginData;
    LoginData.ProviderName = ProviderName;

    FGamebaseCommunicatorSendData SendData(GamebaseScheme::Login);
    SendData.JsonData = LoginData.ToJson(false);

    Communicator->GetAsync(SendData, Callback);
}

void FGamebaseBridgeAuth::Login(const FGamebaseVariantMap& CredentialInfo, const FGamebaseAuthTokenDelegate& Callback)
{    
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::LoginWithCredentialInfo);
    SendData.JsonData = NHN::Json::SerializeVariantMapToJsonString(CredentialInfo);

    Communicator->GetAsync(SendData, Callback);
}

void FGamebaseBridgeAuth::Login(const FString& ProviderName, const FGamebaseVariantMap& AdditionalInfo, const FGamebaseAuthTokenDelegate& Callback)
{
    if (const auto Adapter = LoadedAdapters.GetAdapter(ProviderName))
    {
        Adapter->Login(AdditionalInfo, [this, Callback](const FGamebaseAuthCredentialsResult& Result)
        {
            if (Result.IsError())
            {
                Callback.ExecuteIfBound(nullptr, &Result.GetErrorValue());
                return;
            }
            Login(Result.GetOkValue(), Callback);
        });
        return;
    }
    
    FGamebaseVariantMap JsonData;
    JsonData.Add(TEXT("providerName"), ProviderName);
    JsonData.Add(TEXT("additionalInfo"), AdditionalInfo);
    
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::LoginWithAdditionalInfo);
    SendData.JsonData = NHN::Json::SerializeVariantMapToJsonString(JsonData);
    
    Communicator->GetAsync(SendData, Callback);
}

void FGamebaseBridgeAuth::LoginForLastLoggedInProvider(const FGamebaseAuthTokenDelegate& Callback)
{
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::LoginForLastLoggedInProvider);
    
    Communicator->GetAsync(SendData, Callback);
}

void FGamebaseBridgeAuth::LoginForLastLoggedInProvider(const FGamebaseVariantMap& AdditionalInfo, const FGamebaseAuthTokenDelegate& Callback)
{
    FGamebaseVariantMap JsonData;
    JsonData.Add(TEXT("additionalInfo"), AdditionalInfo);
    
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::LoginForLastLoggedInProviderWithAdditionalInfo);
    SendData.JsonData = NHN::Json::SerializeVariantMapToJsonString(JsonData);

    Communicator->GetAsync(SendData, Callback);
}

void FGamebaseBridgeAuth::AddMapping(const FString& ProviderName, const FGamebaseAuthTokenDelegate& Callback)
{
    if (const auto Adapter = LoadedAdapters.GetAdapter(ProviderName))
    {
        Adapter->Login([this, Callback](const FGamebaseAuthCredentialsResult& Result)
        {
            if (Result.IsError())
            {
                Callback.ExecuteIfBound(nullptr, &Result.GetErrorValue());
                return;
            }
            
            AddMapping(Result.GetOkValue(), Callback);
        });
        
        return;
    }
    
    FAuthProviderInfo MappingData;
    MappingData.ProviderName = ProviderName;

    FGamebaseCommunicatorSendData SendData(GamebaseScheme::AddMapping);
    SendData.JsonData = MappingData.ToJson(false);

    Communicator->GetAsync(SendData, Callback);
}

void FGamebaseBridgeAuth::AddMapping(const FString& ProviderName, const FGamebaseVariantMap& AdditionalInfo, const FGamebaseAuthTokenDelegate& Callback)
{
    if (const auto Adapter = LoadedAdapters.GetAdapter(ProviderName))
    {
        Adapter->Login(AdditionalInfo, [this, Callback](const FGamebaseAuthCredentialsResult& Result)
        {
            if (Result.IsError())
            {
                Callback.ExecuteIfBound(nullptr, &Result.GetErrorValue());
                return;
            }
            AddMapping(Result.GetOkValue(), Callback);
        });
        return;
    }

    FGamebaseVariantMap JsonData;
    JsonData.Add(TEXT("providerName"), ProviderName);
    JsonData.Add(TEXT("additionalInfo"), AdditionalInfo);
    
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::AddMappingWithAdditionalInfo);
    SendData.JsonData = NHN::Json::SerializeVariantMapToJsonString(JsonData);

    Communicator->GetAsync(SendData, Callback);
}

void FGamebaseBridgeAuth::AddMapping(const FGamebaseVariantMap& CredentialInfo, const FGamebaseAuthTokenDelegate& Callback)
{
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::AddMappingWithCredentialInfo);
    SendData.JsonData = NHN::Json::SerializeVariantMapToJsonString(CredentialInfo);
    
    Communicator->GetAsync(SendData, Callback);
}

void FGamebaseBridgeAuth::ChangeLogin(const FGamebaseForcingMappingTicket& ForcingMappingTicket, const FGamebaseAuthTokenDelegate& Callback)
{
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::ChangeLogin);
    SendData.JsonData = ForcingMappingTicket.ToJson();

    Communicator->GetAsync(SendData, Callback);
}

void FGamebaseBridgeAuth::AddMappingForcibly(const FGamebaseForcingMappingTicket& ForcingMappingTicket, const FGamebaseAuthTokenDelegate& Callback)
{
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::AddMappingForciblyWithForcingMappingTicket);
    SendData.JsonData = ForcingMappingTicket.ToJson();

    Communicator->GetAsync(SendData, Callback);
}

void FGamebaseBridgeAuth::AddMappingForcibly(const FString& ProviderName, const FString& ForcingMappingKey, const FGamebaseAuthTokenDelegate& Callback)
{
    FAuthAddMappingForcibly MappingData;
    MappingData.ProviderName = ProviderName;
    MappingData.ForcingMappingKey = ForcingMappingKey;
    
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::AddMappingForcibly);
    SendData.JsonData = MappingData.ToJson(false);
    
    Communicator->GetAsync(SendData, Callback);
}

void FGamebaseBridgeAuth::AddMappingForcibly(const FString& ProviderName, const FString& ForcingMappingKey, const FGamebaseVariantMap& AdditionalInfo, const FGamebaseAuthTokenDelegate& Callback)
{
    FGamebaseVariantMap JsonData;
    JsonData.Add(TEXT("providerName"), ProviderName);
    JsonData.Add(TEXT("forcingMappingKey"), ForcingMappingKey);
    JsonData.Add(TEXT("additionalInfo"), AdditionalInfo);
    
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::AddMappingForciblyWithAdditionalInfo);
    SendData.JsonData = NHN::Json::SerializeVariantMapToJsonString(JsonData);
    
    Communicator->GetAsync(SendData, Callback);
}

void FGamebaseBridgeAuth::AddMappingForcibly(const FGamebaseVariantMap& CredentialInfo, const FString& ForcingMappingKey, const FGamebaseAuthTokenDelegate& Callback)
{
    FGamebaseVariantMap JsonData;
    JsonData.Add(TEXT("forcingMappingKey"), ForcingMappingKey);
    JsonData.Add(TEXT("additionalInfo"), CredentialInfo);
    
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::AddMappingForciblyWithCredentialInfo);
    SendData.JsonData = NHN::Json::SerializeVariantMapToJsonString(JsonData);
    
    Communicator->GetAsync(SendData, Callback);
}

void FGamebaseBridgeAuth::RemoveMapping(const FString& ProviderName, const FGamebaseErrorDelegate& Callback)
{
    FAuthProviderInfo MappingData;
    MappingData.ProviderName = ProviderName;

    FGamebaseCommunicatorSendData SendData(GamebaseScheme::RemoveMapping);
    SendData.JsonData = MappingData.ToJson(false);

    Communicator->GetAsync(SendData, Callback);
}

void FGamebaseBridgeAuth::Logout(const FGamebaseErrorDelegate& Callback)
{
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::Logout);

    Communicator->GetAsync(SendData, Callback);
}

void FGamebaseBridgeAuth::Withdraw(const FGamebaseErrorDelegate& Callback)
{
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::Withdraw);

    Communicator->GetAsync(SendData, Callback);
}

void FGamebaseBridgeAuth::WithdrawImmediately(const FGamebaseErrorDelegate& Callback)
{
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::WithdrawImmediately);

    Communicator->GetAsync(SendData, Callback);
}

void FGamebaseBridgeAuth::RequestTemporaryWithdrawal(const FGamebaseTemporaryWithdrawalDelegate& Callback)
{
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::RequestTemporaryWithdrawal);

    Communicator->GetAsync(SendData, Callback);
}

void FGamebaseBridgeAuth::CancelTemporaryWithdrawal(const FGamebaseErrorDelegate& Callback)
{
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::CancelTemporaryWithdrawal);

    Communicator->GetAsync(SendData, Callback);
}

void FGamebaseBridgeAuth::IssueTransferAccount(const FGamebaseTransferAccountDelegate& Callback)
{
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::IssueTransferAccount);

    Communicator->GetAsync(SendData, Callback);
}

void FGamebaseBridgeAuth::QueryTransferAccount(const FGamebaseTransferAccountDelegate& Callback)
{
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::QueryTransferAccount);

    Communicator->GetAsync(SendData, Callback);
}

void FGamebaseBridgeAuth::RenewTransferAccount(const FGamebaseTransferAccountRenewConfiguration& Configuration, const FGamebaseTransferAccountDelegate& Callback)
{
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::RenewTransferAccount);
    SendData.JsonData = Configuration.ToJson();
    
    Communicator->GetAsync(SendData, Callback);
}

void FGamebaseBridgeAuth::TransferAccountWithIdPLogin(const FString& AccountId, const FString& AccountPassword, const FGamebaseAuthTokenDelegate& Callback)
{
    FAuthTransferAccount AccountData;
    AccountData.AccountId = AccountId;
    AccountData.AccountPassword = AccountPassword;
    
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::TransferAccountWithIdPLogin);
    SendData.JsonData = AccountData.ToJson(false);

    Communicator->GetAsync(SendData, Callback);
}

TArray<FString> FGamebaseBridgeAuth::GetAuthMappingList()
{
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::GetAuthMappingList);

    TArray<FString> MappingList;

    const FString GetJsonString = Communicator->GetSync(SendData);
    if (GetJsonString.IsEmpty() == false)
    {
        const auto JsonReader = TJsonReaderFactory<>::Create(GetJsonString);
        TArray<TSharedPtr<FJsonValue>> JsonValues;

        if (FJsonSerializer::Deserialize(JsonReader, JsonValues) == true)
        {
            for (const auto& JsonValue : JsonValues)
            {
                MappingList.Add(JsonValue->AsString());
            }
        }
    }

    return MappingList;
}

FString FGamebaseBridgeAuth::GetAuthProviderUserID(const FString& ProviderName)
{
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::GetAuthProviderUserID);
    SendData.JsonData = ProviderName;

    return Communicator->GetSync(SendData);
}

FString FGamebaseBridgeAuth::GetAuthProviderAccessToken(const FString& ProviderName)
{
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::GetAuthProviderAccessToken);
    SendData.JsonData = ProviderName;

    return Communicator->GetSync(SendData);
}

const FGamebaseAuthProviderProfilePtr FGamebaseBridgeAuth::GetAuthProviderProfile(const FString& ProviderName)
{
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::GetAuthProviderProfile);
    SendData.JsonData = ProviderName;

    const FString GetJsonString = Communicator->GetSync(SendData);
    if (GetJsonString.IsEmpty())
    {
        return nullptr;
    }
    
    const FGamebaseAuthProviderProfilePtr Profile = MakeShared<FGamebaseAuthProviderProfile, ESPMode::ThreadSafe>();
    Profile->FromJson(GetJsonString);

    return Profile;
}

const FGamebaseBanInfoPtr FGamebaseBridgeAuth::GetBanInfo()
{
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::GetBanInfo);

    const FString GetJsonString = Communicator->GetSync(SendData);
    if (GetJsonString.IsEmpty())
    {
        return nullptr;
    }
    
    const FGamebaseBanInfoPtr BanInfo = MakeShared<FGamebaseBanInfo, ESPMode::ThreadSafe>();
    BanInfo->FromJson(GetJsonString);
    
    return BanInfo;
}