#pragma once

#include "CoreMinimal.h"
#include "NhnJsonSerializeMacros.h"
#include "Types/GamebaseError.h"

struct FGamebaseCommunicatorSendData final : FNhnJsonSerializable
{
    FString Scheme;
    int32 Handle = 0;
    FString ObjectName;      // for Unity
    FString JsonData;
    FString ExtraData;

    explicit FGamebaseCommunicatorSendData(const FName& Scheme) : Scheme(Scheme.ToString()) {}

    BEGIN_NHN_JSON_SERIALIZER
        NHN_JSON_SERIALIZE("scheme", Scheme);
        NHN_JSON_SERIALIZE("handle", Handle);
        NHN_JSON_SERIALIZE("gameObjectName", ObjectName);
        NHN_JSON_SERIALIZE("jsonData", JsonData);
        NHN_JSON_SERIALIZE("extraData", ExtraData);
    END_NHN_JSON_SERIALIZER
};

struct FGamebaseCommunicatorReceiveData final : FNhnJsonSerializable
{
    FString Scheme;
    int32 Handle;
    FString JsonData;
    FString ExtraData;
    FString GamebaseError;
    
    BEGIN_NHN_JSON_SERIALIZER
        NHN_JSON_SERIALIZE("scheme", Scheme);
        NHN_JSON_SERIALIZE("handle", Handle);
        NHN_JSON_SERIALIZE("jsonData", JsonData);
        NHN_JSON_SERIALIZE("extraData", ExtraData);
        NHN_JSON_SERIALIZE("gamebaseError", GamebaseError);
    END_NHN_JSON_SERIALIZER
};

using FGamebaseCommunicatorReceiveDataPtr = TSharedPtr<FGamebaseCommunicatorReceiveData, ESPMode::ThreadSafe>;
using FGamebaseCommunicatorReceiveDataRef = TSharedRef<FGamebaseCommunicatorReceiveData, ESPMode::ThreadSafe>;

using FGamebaseCommunicatorReceiveCallback = TFunction<void(FGamebaseCommunicatorReceiveDataRef)>;
using FGamebaseCommunicatorExtraDataCallback = TFunction<void(const FString&)>;

static FGamebaseErrorPtr GetGamebaseError(const FGamebaseCommunicatorReceiveDataRef& ReceiveData)
{
    if (ReceiveData->GamebaseError.IsEmpty())
    {
        return nullptr;
    }
    
    const FGamebaseErrorPtr Error = MakeShared<FGamebaseError, ESPMode::ThreadSafe>();
    Error->FromJson(ReceiveData->GamebaseError);
    
    return Error;
}