﻿#include "GamebaseCommunicatorManager.h"

#include "GamebasePlatformDefines.h"
#include "Async/TaskGraphInterfaces.h"

FGamebaseCommunicatorManager::FGamebaseCommunicatorManager()
    : Communicator(MakeUnique<FGamebaseCommunicator>())
{
}

FGamebaseCommunicatorManager::~FGamebaseCommunicatorManager()
{
}

void FGamebaseCommunicatorManager::Initialize(const FName& ClassName)
{
    Communicator->Initialize(ClassName);
}

void FGamebaseCommunicatorManager::RegisterEvent(const FName& Scheme, const FGamebaseCommunicatorReceiveCallback& Callback)
{
    FScopeLock Lock(&CriticalSection);
    SchemeCallbacks.FindOrAdd(Scheme, Callback);
}

void FGamebaseCommunicatorManager::ReceiveEvent(FGamebaseCommunicatorReceiveDataPtr Result)
{
    check(Result.IsValid());

    FFunctionGraphTask::CreateAndDispatchWhenReady([this, Result]()
    {
        FScopeLock Lock(&CriticalSection);
        
        if (const FGamebaseCommunicatorReceiveCallback* SchemeCallback = SchemeCallbacks.Find(FName(Result->Scheme)))
        {
            (*SchemeCallback)(Result.ToSharedRef());
        }
        
        if (const FGamebaseCommunicatorReceiveCallback* HandleCallback = HandleCallbacks.Find(Result->Handle))
        {
            (*HandleCallback)(Result.ToSharedRef());
            UnregisterHandleCallback(Result->Handle);
        }
    }
    , TStatId(), nullptr, ENamedThreads::GameThread);
}

FString FGamebaseCommunicatorManager::GetSync(const FGamebaseCommunicatorSendData& SendData)
{
    return Communicator->GetSync(SendData.ToJson(false));
}

void FGamebaseCommunicatorManager::RegisterAsyncEvent(FGamebaseCommunicatorSendData Data, const FGamebaseCommunicatorReceiveCallback& Callback)
{
    static FThreadSafeCounter HandleCounter;
    Data.Handle = HandleCounter.Increment();
    RegisterHandleCallback(Data.Handle, Callback);

    const FString JsonString = Data.ToJson(false);
    Communicator->GetAsync(JsonString);
}

void FGamebaseCommunicatorManager::RegisterHandleCallback(const int32 Handle, const FGamebaseCommunicatorReceiveCallback& Callback)
{
    FScopeLock Lock(&CriticalSection);
    HandleCallbacks.FindOrAdd(Handle, Callback);
}

void FGamebaseCommunicatorManager::UnregisterHandleCallback(const int32 Handle)
{
    FScopeLock Lock(&CriticalSection);
    HandleCallbacks.Remove(Handle);
}