#pragma once

#include "CoreMinimal.h"
#include "GamebaseCommunicatorConverter.h"
#include "GamebaseCommunicatorTypes.h"

namespace Gamebase
{
    using FVoidDelegate = TDelegate<void()>;
    using FVoidWithErrorDelegate = TDelegate<void(const FGamebaseError*)>;
    
    template<typename T>
    using TSingleValueDelegate = TDelegate<void(T)>;
    
    template<typename T>
    using TSingleValueWithErrorDelegate = TDelegate<void(T, const FGamebaseError*)>;

    namespace DelegateTraits
    {
        template<typename T> struct TDelegateTraits;

        template<>
        struct TDelegateTraits<FVoidDelegate>
        {
            static void Execute(const FVoidDelegate& Delegate, const FGamebaseCommunicatorReceiveDataRef& Data)
            {
                Delegate.ExecuteIfBound();
            }
        };
        
        template<>
        struct TDelegateTraits<FVoidWithErrorDelegate>
        {
            static void Execute(const FVoidWithErrorDelegate& Delegate, const FGamebaseCommunicatorReceiveDataRef& Data)
            {
                Delegate.ExecuteIfBound(GetGamebaseError(Data).Get());
            }
        };
        
        template<typename T>
        struct TDelegateTraits<TSingleValueDelegate<T>>
        {
            static void Execute(const TSingleValueDelegate<T>& Delegate, const FGamebaseCommunicatorReceiveDataRef& Data)
            {
                T Value = Communicator::Convert<T>(Data);
                Delegate.ExecuteIfBound(Value);
            }
        };
        
        template<typename T>
        struct TDelegateTraits<TSingleValueWithErrorDelegate<T>>
        {
            static void Execute(const TSingleValueWithErrorDelegate<T>& Delegate, const FGamebaseCommunicatorReceiveDataRef& Data)
            {
                T Value = Communicator::Convert<T>(Data);
                Delegate.ExecuteIfBound(Value, GetGamebaseError(Data).Get());
            }
        };
    }
    
    /**
     * Creates a communicator callback from a response delegate and optional extra data callback.
     */
    template<typename DelegateType>
    FGamebaseCommunicatorReceiveCallback MakeCallback(
        const DelegateType& ResponseCallback,
        const FGamebaseCommunicatorExtraDataCallback& ExtraDataCallback = nullptr)
    {
        return [ResponseCallback, ExtraDataCallback](const FGamebaseCommunicatorReceiveDataRef& Data)
        {
            DelegateTraits::TDelegateTraits<DelegateType>::Execute(ResponseCallback, Data);
            
            if (ExtraDataCallback)
            {
                ExtraDataCallback(Data->ExtraData);
            }
        };
    }
}
