#pragma once

#include "CoreMinimal.h"
#include "GamebaseValueObject.h"
#include "GamebaseLoggerTypes.generated.h"

UENUM()
enum class EGamebaseLogLevel : uint8
{
    Debug = 0,
    Info = 1,
    Warn = 2,
    Error = 3,
    Fatal = 4,
    None = 5
};

UENUM()
enum class EGamebaseLogType : uint8
{
    Error = 0,
    Assert = 1,
    Warning = 2,
    Log = 3,
    Exception = 4
};

struct GAMEBASE_API FGamebaseLoggerConfiguration final : FGamebaseValueObject
{
    FString AppKey;
    bool bEnableCrashReporter;
    FString ZoneType = TEXT("REAL");

    FGamebaseLoggerConfiguration()
        : bEnableCrashReporter(false)
    {
    }

    FGamebaseLoggerConfiguration(const FString& AppKey, const bool bInEnableCrashReporter)
        : AppKey(AppKey)
        , bEnableCrashReporter(bInEnableCrashReporter)
    {
    }

    virtual void Serialize(FJsonSerializerBase& Serializer, bool bFlatObject) override;
};


struct GAMEBASE_API FGamebaseLogEntry final : FGamebaseValueObject
{
    FString LogType;
    EGamebaseLogLevel LogLevel;
    FString Message;
    FString TransactionId;
    int64 CreateTime;
    TMap<FString, FString> UserFields;

    FGamebaseLogEntry() = default;
    FGamebaseLogEntry(const FString& Type, const EGamebaseLogLevel Level, const FString& Message, const TMap<FString, FString>& UserFields)
        : LogType(Type)
        , LogLevel(Level)
        , Message(Message)
        , CreateTime(0)
        , UserFields(UserFields)
    {
    }

    virtual void Serialize(FJsonSerializerBase& Serializer, bool bFlatObject) override;
};

struct GAMEBASE_API FGamebaseLogFilter final : FGamebaseValueObject
{
    FString Name;
    
    virtual void Serialize(FJsonSerializerBase& Serializer, bool bFlatObject) override;
};

DECLARE_DELEGATE_OneParam(FGamebaseLoggerSuccessDelegate, const FGamebaseLogEntry&);
DECLARE_DELEGATE_OneParam(FGamebaseLoggerSaveDelegate, const FGamebaseLogEntry&);
DECLARE_DELEGATE_TwoParams(FGamebaseLoggerFilterDelegate, const FGamebaseLogEntry&, const FGamebaseLogFilter&);
DECLARE_DELEGATE_TwoParams(FGamebaseLoggerErrorDelegate, const FGamebaseLogEntry&, const FString&);

struct GAMEBASE_API FGamebaseLoggerListener
{
    FGamebaseLoggerSuccessDelegate OnSuccessDelegate;
    FGamebaseLoggerSaveDelegate OnSaveDelegate;
    FGamebaseLoggerFilterDelegate OnFilterDelegate;
    FGamebaseLoggerErrorDelegate OnErrorDelegate;
};

struct GAMEBASE_API FGamebaseLoggerEngineCrashData final : FGamebaseValueObject
{
    EGamebaseLogLevel LogLevel;
    FString Condition;
    FString StackTrace;

    virtual void Serialize(FJsonSerializerBase& Serializer, bool bFlatObject) override;
};

DECLARE_DELEGATE_RetVal_OneParam(bool, FGamebaseLoggerCrashFilterDelegate, const FGamebaseLoggerEngineCrashData&);