#pragma once

#include "CoreMinimal.h"
#include "Types/GamebaseWebViewTypes.h"

class GAMEBASE_API IGamebaseWebView
{
public:
    virtual ~IGamebaseWebView() = default;

    /**
     * This method to create the web browser and display it on screen.
     *
     * @param Url       The url of the resource to load.
     *
     * Example Usage:
     * @code
     *  void OpenWebBrowserSample(const FString& url)
     *  {
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->GetWebView()->OpenWebBrowser(url);
     *  }
     * @endcode
     */
    virtual void OpenWebBrowser(const FString& Url) = 0;

    /**
     * This method to create the webview and display it on screen.
     *
     * @param Url               The url of the resource to load.
     * @param Configuration     The initSettings of webview.
     * @param CloseCallback     called when the browser closed.
     * @param SchemeList        for receiving events.
     * @param SchemeEvent       from Scheme list.
     *
     * Example Usage:
     * @code
     *  void ShowWebViewSample(const FString& Url, const FString& scheme)
     *  {
     *      FGamebaseWebViewConfiguration Configuration;
     *      Configuration.title = TEXT("ShowWebViewWithConfiguration");
     *      Configuration.isNavigationBarVisible = true;
     *      Configuration.barHeight = 60;
     *      Configuration.isBackButtonVisible = true;
     *      Configuration.contentMode = GamebaseWebViewContentMode::Recommended;
     *      Configuration.enableFixedFontSize = false;
     *      Configuration.renderOutsideSafeArea = false;
     *
     *      TArray<FString> SchemeList { scheme };
     *
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->GetWebView()->ShowWebView(Url, Configuration,
     *          FGamebaseErrorDelegate::CreateLambda([](const FGamebaseError* error) {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("WebView closed."));
     *          }),
     *          SchemeList,
     *          FGamebaseSchemeEventDelegate::CreateLambda([](const FString& scheme, const FGamebaseError* error) {
     *              if (Gamebase::IsSuccess(error))
     *              {
     *                  UE_LOG(GamebaseTestResults, Display, TEXT("Scheme event called. (scheme= %s)"), *scheme);
     *              }
     *              else
     *              {
     *                  UE_LOG(GamebaseTestResults, Display, TEXT("Scheme event failed. (error: %d)"), error->code);
     *              }
     *          }));
     *  }
     * @endcode
     */
    virtual void ShowWebView(const FString& Url,
        const FGamebaseWebViewConfiguration& Configuration,
        const FGamebaseErrorDelegate& CloseCallback = FGamebaseErrorDelegate(),
        const TArray<FString>& SchemeList = TArray<FString>(),
        const FGamebaseSchemeEventDelegate& SchemeEvent = FGamebaseSchemeEventDelegate()) = 0;

    /**
     * This method to close the webview in display on a screen.
     *
     * Example Usage:
     * @code
     *  void CloseWebViewSample(const FString& url)
     *  {
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->GetWebView()->CloseWebView();
     *  }
     * @endcode
     */
    virtual void CloseWebView() = 0;
};