#pragma once

#include "CoreMinimal.h"
#include "Types/GamebaseUtilTypes.h"

class GAMEBASE_API IGamebaseUtil
{
public:
    virtual ~IGamebaseUtil() = default;

    /**
     * This method to create the dialog and display it on screen.
     *
     * @param Title     The title displayed in the dialog.
     * @param Message   The message to display.
     *
     * Example Usage:
     * @code
     *  void ShowAlertSample(const FString& title, const FString& message)
     *  {
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->GetUtil()->ShowAlert(title, message);
     *  }
     * @endcode
     */
    virtual void ShowAlert(const FString& Title, const FString& Message) = 0;

    /**
     * This method to create the dialog and display it on screen.
     *
     * @param Title         The title displayed in the dialog.
     * @param Message       The message to display.
     * @param CloseCallback The clickListener to be invoked when the button of the dialog is pressed.
     *
     * Example Usage:
     * @code
     *  void ShowAlertSample(const FString& title, const FString& message)
     *  {
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->GetUtil()->ShowAlert(title, message, FGamebaseAlertCloseDelegate::CreateLambda([]()
     *      {
     *          UE_LOG(GamebaseTestResults, Display, TEXT("The button was clicked."));
     *      }));
     *  }
     * @endcode
     */
    virtual void ShowAlert(const FString& Title, const FString& Message, const FGamebaseAlertCloseDelegate& CloseCallback) = 0;

    /**
     * Show a standard toast that just contains a text view.
     *
     * @param Message           The message to show.
     * @param ExposureTimeType  The time interval to be exposed. GamebaseUIToastType.TOAST_LENGTH_LONG (3.5 seconds), GamebaseUIToastType.TOAST_LENGTH_SHORT (2 seconds)
     *
     * Example Usage:
     * @code
     *  void ShowToastSample(const FString& message)
     *  {
     *      // EGamebaseToastExposureTime::Short or 0 : 2 seconds
     *      // EGamebaseToastExposureTime::Long or 1 : 3.5 seconds
     *      // All the rest values : 2 seconds
     *
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->GetUtil()->ShowToast(message, EGamebaseToastExposureTime::Long);
     *  }
     * @endcode
     */
    virtual void ShowToast(const FString& Message, EGamebaseToastExposureTime ExposureTimeType) = 0;

    virtual void ShowAlert(TMap<FString, FString> Parameters, const FGamebaseAlertDelegate& Callback) = 0;
};