#pragma once

#include "CoreMinimal.h"
#include "Types/GamebaseAuthTypes.h"

/**
 * This class provides wrapping of function execution related to temporary withdrawal.
 */
class GAMEBASE_API IGamebaseTemporaryWithdrawal
{
public:
    virtual ~IGamebaseTemporaryWithdrawal() = default;

    /**
     * This function ignores the withdrawal grace period and proceeds withdrawal immediately.<br>
     * Internally, this function works exactly the same as the GamebaseSubsystem->Withdraw().
     *
     * @param Callback  Callbacks the result of withdraw.
     *
     * Example Usage:
     * @code
     *  void RequestWithdrawal()
     *  {
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->GetTemporaryWithdrawal()->WithdrawImmediately(FGamebaseErrorDelegate::CreateLambda([=](const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("WithdrawImmediately succeeded."));
     *          }
     *          else
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("WithdrawImmediately failed. (errorCode: %d, errorMessage: %s)"), Error->code, *Error->message);
     *          }
     *      }));
     *  }
     * @endcode
     */
    virtual void WithdrawImmediately(const FGamebaseErrorDelegate& Callback) = 0;
     
    /**
     * If the user has requested to withdraw, cancel it.<br>
     * If the user has never requested to leave, the function call will fail.
     *
     * @param Callback    Callbacks the result of withdraw.
     *
     * Example Usage:
     * @code
     *  void RequestWithdrawal()
     *  {
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->GetTemporaryWithdrawal()->RequestWithdrawal(FGamebaseTemporaryWithdrawalDelegate::CreateLambda([=](const FGamebaseTemporaryWithdrawalInfo* info, const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("RequestWithdrawal succeeded. The date when you can withdraw your withdrawal is %d"), info->gracePeriodDate);
     *          }
     *          else
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("RequestWithdrawal failed. (errorCode: %d, errorMessage: %s)"), Error->code, *Error->message);
     *          }
     *      }));
     *  }
     * @endcode
     */
    virtual void RequestWithdrawal(const FGamebaseTemporaryWithdrawalDelegate& Callback) = 0;
    
     /**
      * If the user has requested to withdraw, cancel it.<br>
      * If the user has never requested to leave, the function call will fail.
      *
      * @param Callback    Callbacks the result of withdraw.
      *
      * Example Usage:
      * @code
      *  void CancelTemporaryWithdrawal()
      *  {
      *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
      *      GamebaseSubsystem->GetTemporaryWithdrawal()->CancelTemporaryWithdrawal(FGamebaseErrorDelegate::CreateLambda([=](const FGamebaseError* Error)
      *      {
      *          if (Gamebase::IsSuccess(Error))
      *          {
      *              UE_LOG(GamebaseTestResults, Display, TEXT("CancelTemporaryWithdrawal succeeded."));
      *          }
      *          else
      *          {
      *              UE_LOG(GamebaseTestResults, Display, TEXT("CancelTemporaryWithdrawal failed. (errorCode: %d, errorMessage: %s)"), Error->code, *Error->message);
      *          }
      *      }));
      *  }
      * @endcode
      */
    virtual void CancelWithdrawal(const FGamebaseErrorDelegate& Callback) = 0;
};