#pragma once

#include "CoreMinimal.h"
#include "Types/GamebasePushTypes.h"

class GAMEBASE_API IGamebasePush
{
public:
    virtual ~IGamebasePush() = default;
 
    /**
     * Register push information to the push server.
     *
     * @param Configuration   Settings of the notification.
     * @param Callback        Callback pass to API result.
     *
     * Example Usage:
     * @code
     *  void RegisterPushSample(bool bPushEnabled, bool bAdAgreement, bool bAdAgreementNight,
     *      bool bRequestNotificationPermission, bool bAlwaysAllowTokenRegistration)
     *  {
     *      FGamebasePushConfiguration Configuration;
     *      Configuration.bPushEnabled = bPushEnabled;
     *      Configuration.bADAgreement = bADAgreement;
     *      Configuration.bADAgreementNight = bADAgreementNight;
     *      Configuration.bRequestNotificationPermission = bRequestNotificationPermission;
     *      Configuration.bAlwaysAllowTokenRegistration = bAlwaysAllowTokenRegistration;
     *
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->GetPush()->RegisterPush(configuration,
     *          FGamebaseErrorDelegate::CreateLambda([](const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("RegisterPush succeeded."));
     *          }
     *          else
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("RegisterPush failed"));
     *          }
     *      }));
     *  }
     * @endcode
     */
    virtual void RegisterPush(const FGamebasePushConfiguration& Configuration, const FGamebaseErrorDelegate& Callback) = 0;

    /**
     * Register push information to the push server.
     *
     * @param Configuration         Settings of the push from server.
     * @param NotificationOptions   Settings of the local notification.
     * @param Callback              Callback pass to API result.
     *
     * Example Usage:
     * @code
     *  void RegisterPushSample(bool bPushEnabled, bool bAdAgreement, bool bAdAgreementNight,
     *      bool bRequestNotificationPermission, bool bAlwaysAllowTokenRegistration, const FString& DisplayLanguage,
     *      bool bForegroundEnabled, bool bBadgeEnabled, bool bSoundEnabled, int32 Priority, const FString& SmallIconName, const FString& SoundFileName)
     *  {
     *      FGamebasePushConfiguration Configuration;
     *      Configuration.bPushEnabled = bPushEnabled;
     *      Configuration.bADAgreement = bADAgreement;
     *      Configuration.bADAgreementNight = bADAgreementNight;
     *      Configuration.bRequestNotificationPermission = bRequestNotificationPermission;
     *      Configuration.bAlwaysAllowTokenRegistration = bAlwaysAllowTokenRegistration;
     *      
     *      FGamebaseNotificationOptions NotificationOptions;
     *      Options.bForegroundEnabled = bForegroundEnabled;
     *      Options.bBadgeEnabled = bBadgeEnabled;
     *      Options.bSoundEnabled = bSoundEnabled;
     *      Options.Priority = Priority;
     *      Options.SmallIconName = SmallIconName;
     *      Options.SoundFileName = SoundFileName;
     *      
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->GetPush()->RegisterPush(Configuration, NotificationOptions,
     *          FGamebaseErrorDelegate::CreateLambda([](const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("RegisterPush succeeded."));
     *          }
     *          else
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("RegisterPush failed"));
     *          }
     *      }));
     *  }
     * @endcode
     */
    virtual void RegisterPush(const FGamebasePushConfiguration& Configuration, const FGamebaseNotificationOptions& NotificationOptions, const FGamebaseErrorDelegate& Callback) = 0;

    /**
     * Get push settings from the the push server.
     *
     * @param Callback Callback pass to API result.
     *
     * Example Usage:
     * @code
     *  void QueryTokenInfoSample()
     *  {
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->GetPush()->QueryTokenInfo(
     *         FGamebasePushTokenInfoDelegate::CreateLambda([=](const FGamebasePushTokenInfo* tokenInfo, const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("RegisterPush succeeded. (pushEnabled= %s, adAgreement= %s, adAgreementNight= %s, displayLanguageCode= %s)"),
     *                  pushAdvertisements->pushEnabled ? TEXT("true") : TEXT("false"),
     *                  pushAdvertisements->adAgreement ? TEXT("true") : TEXT("false"),
     *                  pushAdvertisements->adAgreementNight ? TEXT("true") : TEXT("false"),
     *                  *pushAdvertisements->displayLanguageCode.ToString());
     *          }
     *          else
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("RegisterPush failed"));
     *          }
     *      }));
     *  }
     * @endcode
     */
    virtual void QueryTokenInfo(const FGamebasePushTokenInfoDelegate& Callback) = 0;

    /**
     * Get push settings from the the push server.
     *
     * @param Callback Callback pass to API result.
     *
     * Example Usage:
     * @code
     *  void QueryPushSample()
     *  {
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->GetPush()->QueryPush(
     *          FGamebasePushConfigurationDelegate::CreateLambda([](const FGamebasePushConfiguration* pushAdvertisements, const FGamebaseError* Error)
     *          {
     *              if (Gamebase::IsSuccess(Error))
     *              {
     *                  UE_LOG(GamebaseTestResults, Display, TEXT("RegisterPush succeeded. (pushEnabled= %s, adAgreement= %s, adAgreementNight= %s, displayLanguageCode= %s)"),
     *                      pushAdvertisements->pushEnabled ? TEXT("true") : TEXT("false"),
     *                      pushAdvertisements->adAgreement ? TEXT("true") : TEXT("false"),
     *                      pushAdvertisements->adAgreementNight ? TEXT("true") : TEXT("false"),
     *                      *pushAdvertisements->displayLanguageCode.ToString());
     *              }
     *              else
     *              {
     *                  UE_LOG(GamebaseTestResults, Display, TEXT("RegisterPush failed"));
     *              }
     *          })
     *      );
     *  }
     * @endcode
     */
    virtual void QueryPush(const FGamebasePushConfigurationDelegate& Callback) = 0;

    /**
     * Set SandboxMode.(iOS Only)
     *
     * @param bIsSandbox "true" if application is on the sandbox mode.
     *
     * Example Usage:
     * @code
     *  void SetSandboxModeSample(bool isSandbox)
     *  {
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->GetPush()->SetSandboxMode(isSandbox);
     *  }
     * @endcode
     */
    virtual void SetSandboxMode(bool bIsSandbox) = 0;
 
    /**
     * Get notification options in device.
     *
     * @return The option settings of device notification.
     */
    virtual FGamebaseNotificationOptionsPtr GetNotificationOptions() = 0;

    /**
     * This API tells that the user has allowed the device to display notification.
     *
     * @param Callback Callback pass to API result.
     */
    virtual void QueryNotificationAllowed(const FGamebaseQueryNotificationAllowedDelegate& Callback) = 0;
};