#pragma once

#include "CoreMinimal.h"
#include "GamebaseTypes.h"
#include "Types/GamebasePurchaseTypes.h"

class GAMEBASE_API IGamebasePurchase
{
public:
    virtual ~IGamebasePurchase() = default;
 
    /**
     * Request Purchasing Item registered in the console with the gamebaseProductId.
     *
     * @param GamebaseProductId     registered in the Gamebase console.
     * @param Callback              callback pass to API result.
     *
     * Example Usage:
     * @code
     *  void RequestPurchaseSample(const FString& GamebaseProductId)
     *  {
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->GetPurchase()->RequestPurchase(GamebaseProductId, FGamebasePurchasableReceiptDelegate::CreateLambda(
     *          [](const FGamebasePurchasableReceipt* purchasableReceipt, const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("RequestPurchase succeeded. (itemSeq= %ld, price= %f, currency= %s, paymentSeq= %s, purchaseToken= %s)"),
     *                  purchasableReceipt->itemSeq, purchasableReceipt->price, *purchasableReceipt->currency,
     *                  *purchasableReceipt->paymentSeq, *purchasableReceipt->purchaseToken);
     *          }
     *          else
     *          {
     *              if (Error->code == GamebaseErrorCode::PURCHASE_USER_CANCELED)
     *              {
     *                  UE_LOG(GamebaseTestResults, Display, TEXT("User canceled purchase."));
     *              }
     *              else
     *              {
     *                  // Check the error code and handle the error appropriately.
     *                  UE_LOG(GamebaseTestResults, Display, TEXT("RequestPurchase failed. (Error: %d)"), Error->code);
     *              }
     *
     *          }
     *      }));
     *  }
     * @endcode
     */
    virtual void RequestPurchase(const FString& GamebaseProductId, const FGamebasePurchasableReceiptDelegate& Callback) = 0;

    /**
     * Request Purchasing Item registered in the console with the gamebaseProductId.<br/>
     * The value entered in the payload can be checked again in the PurchasableReceipt.payload,
     *     so it can be used as a method of classifying item payment according to the game server channel and character.
     *
     * @param GamebaseProductId     registered in the Gamebase console.
     * @param Payload               used to enter the information you want to check again on the receipt.
     * @param Callback              callback pass to API result.
     *
     * Example Usage:
     * @code
     *  void RequestPurchaseSample(const FString& GamebaseProductId, const FString& Payload)
     *  {
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->GetPurchase()->RequestPurchase(GamebaseProductId, Payload, FGamebasePurchasableReceiptDelegate::CreateLambda(
     *          [](const FGamebasePurchasableReceipt* purchasableReceipt, const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("RequestPurchase succeeded. (itemSeq= %ld, price= %f, currency= %s, paymentSeq= %s, purchaseToken= %s)"),
     *                  purchasableReceipt->itemSeq, purchasableReceipt->price, *purchasableReceipt->currency,
     *                  *purchasableReceipt->paymentSeq, *purchasableReceipt->purchaseToken);
     *          }
     *          else
     *          {
     *              if (Error->code == GamebaseErrorCode::PURCHASE_USER_CANCELED)
     *              {
     *                  UE_LOG(GamebaseTestResults, Display, TEXT("User canceled purchase."));
     *              }
     *              else
     *              {
     *                  // Check the error code and handle the error appropriately.
     *                  UE_LOG(GamebaseTestResults, Display, TEXT("RequestPurchase failed. (Error: %d)"), Error->code);
     *              }
     *
     *          }
     *      }));
     *  }
     * @endcode
     */
    virtual void RequestPurchase(const FString& GamebaseProductId, const FString& Payload, const FGamebasePurchasableReceiptDelegate& Callback) = 0;
    
    /**
     * This API is legacy and no longer maintained.
     * Please refer to the URL below for migrating to the new API:
     * https://docs.nhncloud.com/ko/Game/Gamebase/ko/unreal-purchase/#purchase-item
     * @note The legacy API may be deprecated in the future. Transitioning is recommended.
    */
    virtual void RequestPurchase(int64 ItemSeq, const FGamebasePurchasableReceiptDelegate& Callback) = 0;

    /**
     * Request a Item List which is not consumed.
     * You should deliver this itemReceipt info to your game-server to consume it
     * or request consumeAPI to ToastCloud IAP Server
     *
     * @param Callback      callback pass to API result.
     *
     * Example Usage:
     * @code
     *  void RequestItemListOfNotConsumedSample()
     *  {
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->GetPurchase()->RequestItemListOfNotConsumed(FGamebasePurchasableReceiptListDelegate::CreateLambda(
     *          [](const TArray<FGamebasePurchasableReceipt>* purchasableReceiptList, const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("RequestItemListOfNotConsumed succeeded."));
     *
     *              for (const FGamebasePurchasableReceipt& purchasableReceipt : *purchasableReceiptList)
     *              {
     *                  GAMEBASE_RESULT_LOG(" - itemSeq= %ld, price= %f, currency= %s, paymentSeq= %s, purchaseToken= %s",
     *                      purchasableReceipt.itemSeq, purchasableReceipt.price, *purchasableReceipt.currency,
     *                      *purchasableReceipt.paymentSeq, *purchasableReceipt.purchaseToken);
     *              }
     *          }
     *          else
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("RequestItemListOfNotConsumed failed."));
     *          }
     *      }));
     * }
     * @endcode
     */
    GAMEBASE_DEPRECATED(2.49.0, "Use the new API that takes configuration as a parameter.")
    virtual void RequestItemListOfNotConsumed(const FGamebasePurchasableReceiptListDelegate& Callback) = 0;

    /**
     * Request a Item List which is not consumed.
     * You should deliver this itemReceipt info to your game-server to consume it
     * or request consumeAPI to ToastCloud IAP Server
     *
     * @param Configuration Sets additional conditions for retrieving results.
     * @param Callback      Callback pass to API result.
     *
     * Example Usage:
     * @code
     *  void RequestItemListOfNotConsumedSample(bool bAllStores, bool bIncludeExpiredSubscriptions)
     *  {
     *      FGamebasePurchasableConfiguration Configuration;
     *      Configuration.bAllStores = bAllStores;
     *      Configuration.bIncludeExpiredSubscriptions = bIncludeExpiredSubscriptions;
     *  
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->GetPurchase()->RequestItemListOfNotConsumed(Configuration, FGamebasePurchasableReceiptDelegate::CreateLambda(
     *          [](const TArray<FGamebasePurchasableReceipt>* purchasableReceiptList, const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("RequestItemListOfNotConsumed succeeded."));
     *
     *              for (const FGamebasePurchasableReceipt& purchasableReceipt : *purchasableReceiptList)
     *              {
     *                  GAMEBASE_RESULT_LOG(" - itemSeq= %ld, price= %f, currency= %s, paymentSeq= %s, purchaseToken= %s",
     *                      purchasableReceipt.itemSeq, purchasableReceipt.price, *purchasableReceipt.currency,
     *                      *purchasableReceipt.paymentSeq, *purchasableReceipt.purchaseToken);
     *              }
     *          }
     *          else
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("RequestItemListOfNotConsumed failed."));
     *          }
     *      }));
     *  }
     * @endcode
     */
    virtual void RequestItemListOfNotConsumed(const FGamebasePurchasableConfiguration& Configuration, const FGamebasePurchasableReceiptListDelegate& Callback) = 0;
    
    /**
     * Request a item list which is purchasable.
     * This list has items which are registered in both Market(AppStore)
     * and ToastCloud IAP Console
     *
     * @param Callback      callback pass to API result.
     *
     * Example Usage:
     * @code
     *  void RequestItemListOfNotConsumedSample()
     *  {
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->GetPurchase()->RequestItemListPurchasable(FGamebasePurchasableItemListDelegate::CreateLambda(
     *          [](const TArray<FGamebasePurchasableItem>* purchasableItemList, const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("RequestItemListPurchasable succeeded."));
     *
     *              for (const FGamebasePurchasableItem& purchasableItem : *purchasableItemList)
     *              {
     *                  GAMEBASE_RESULT_LOG(" - itemSeq= %ld, price= %f, itemName= %s, itemName= %s, marketItemId= %s",
     *                      purchasableItem.itemSeq, purchasableItem.price, *purchasableItem.currency, *purchasableItem.itemName, *purchasableItem.marketItemId);
     *              }
     *          }
     *          else
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("RequestItemListPurchasable failed."));
     *          }
     *      }));
     *  }
     * @endcode
     */
    virtual void RequestItemListPurchasable(const FGamebasePurchasableItemListDelegate& Callback) = 0;


    /**
     * Request a item list which is purchasable.
     * This list has items which are only registered in ToastCloud IAP Console,
     * not Market(Google, OneStore)
     *
     * @param Callback      callback pass to API result.
     *
     * @code
     *  void RequestItemListOfNotConsumedSample()
     *  {
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->GetPurchase()->RequestItemListPurchasable(FGamebasePurchasableItemListDelegate::CreateLambda(
     *          [](const TArray<FGamebasePurchasableItem>* purchasableItemList, const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("RequestItemListPurchasable succeeded."));
     *
     *              for (const FGamebasePurchasableItem& purchasableItem : *purchasableItemList)
     *              {
     *                  UE_LOG(GamebaseTestResults, Display, TEXT(" - (itemSeq= %ld, price= %f, itemName= %s, itemName= %s, marketItemId= %s)"),
     *                      purchasableItem.itemSeq, purchasableItem.price, *purchasableItem.currency, *purchasableItem.itemName, *purchasableItem.marketItemId);
     *              }
     *          }
     *          else
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("RequestItemListPurchasable failed."));
     *          }
     *      }));
     *  }
     * @endcode
     */
    virtual void RequestItemListAtIAPConsole(const FGamebasePurchasableItemListDelegate& Callback) = 0;

    /**
     * If you want to deliver some items that have been purchased to user, you should implement this handler.
     * You can receive a GamebaseResponse.Purchase.PurchasableReceipt object and receive whether error or success by callback.
     * This API return same information with 'RequestPurchase'
     * You should call this method after `logged in`, and you should call this before an other APIs in `Gamebase.Purchase`.
     * This API can work on iOS 11 or later.
     *
     * @param Callback      Callback may return the result of purchase by App Store Promotion.
     *
     * Example Usage:
     * @code
     *  void SetPromotionIAPHandlerSample()
     *  {
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->GetPurchase()->SetPromotionIAPHandler(FGamebasePurchasableReceiptDelegate::CreateLambda(
     *          [](const FGamebasePurchasableReceipt* purchasableReceipt, const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("SetPromotionIAPHandler succeeded. (itemSeq= %ld, price= %f, currency= %s, paymentSeq= %s, purchaseToken= %s)"),
     *                  purchasableReceipt->itemSeq, purchasableReceipt->price, *purchasableReceipt->currency, *purchasableReceipt->paymentSeq, *purchasableReceipt->purchaseToken);
     *          }
     *          else
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("SetPromotionIAPHandler failed."));
     *          }
     *      }));
     *  }
     * @endcode
     */
    virtual void SetPromotionIAPHandler(const FGamebasePurchasableReceiptDelegate& Callback) = 0;

    /**
     * Request a list of payment products that are currently subscribed.
     * If there is a subscription purchased from another platform (such as iOS),
     * the itemSeq value is returned as -1.
     *
     * @param Callback      List of payment products that are currently subscribed.
     *
     * Example Usage:
     * @code
     *  void RequestActivatedPurchasesSample()
     *  {
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->GetPurchase()->RequestActivatedPurchases(FGamebasePurchasableReceiptListDelegate::CreateLambda(
     *          [](const TArray<FGamebasePurchasableReceipt>* purchasableReceiptList, const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("RequestActivatedPurchases succeeded."));
     *
     *              for (const FGamebasePurchasableReceipt& purchasableReceipt : *purchasableReceiptList)
     *              {
     *                  UE_LOG(GamebaseTestResults, Display, TEXT(" - itemSeq= %ld, price= %f, currency= %s, paymentSeq= %s, purchaseToken= %s"),
     *                      purchasableReceipt.itemSeq, purchasableReceipt.price, *purchasableReceipt.currency, *purchasableReceipt.paymentSeq, *purchasableReceipt.purchaseToken);
     *              }
     *          }
     *          else
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("RequestActivatedPurchases failed."));
     *          }
     *      }));
     *  }
     * @endcode
     */
    GAMEBASE_DEPRECATED(2.49.0, "Use the new API that takes configuration as a parameter.")
    virtual void RequestActivatedPurchases(const FGamebasePurchasableReceiptListDelegate& Callback) = 0;
    
    /**
     * Request a list of payment products that are currently subscribed.
     * If there is a subscription purchased from another platform (such as iOS),
     * the itemSeq value is returned as -1.
     *
     * @param Configuration Sets additional conditions for retrieving results.
     * @param Callback      List of payment products that are currently subscribed.
     *
     * Example Usage:
     * @code
     *  void RequestActivatedPurchasesSample(bool bAllStores, bool bIncludeExpiredSubscriptions)
     *  {
     *      FGamebasePurchasableConfiguration Configuration;
     *      Configuration.bAllStores = bAllStores;
     *      Configuration.bIncludeExpiredSubscriptions = bIncludeExpiredSubscriptions;
     *  
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->GetPurchase()->RequestActivatedPurchases(Configuration, FGamebasePurchasableReceiptListDelegate::CreateLambda(
     *          [](const TArray<FGamebasePurchasableReceipt>* purchasableReceiptList, const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("RequestActivatedPurchases succeeded."));
     *
     *              for (const FGamebasePurchasableReceipt& purchasableReceipt : *purchasableReceiptList)
     *              {
     *                  GAMEBASE_RESULT_LOG(" - itemSeq= %ld, price= %f, currency= %s, paymentSeq= %s, purchaseToken= %s",
     *                      purchasableReceipt.itemSeq, purchasableReceipt.price, *purchasableReceipt.currency, *purchasableReceipt.paymentSeq, *purchasableReceipt.purchaseToken);
     *              }
     *          }
     *          else
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("RequestActivatedPurchases failed."));
     *          }
     *      }));
     *  }
     * @endcode
     */
    virtual void RequestActivatedPurchases(const FGamebasePurchasableConfiguration& Configuration, const FGamebasePurchasableReceiptListDelegate& Callback) = 0;

    /**
     * Request a list of purchased subscription status.
     * This API will work with Google store. (Android Only)
     *
     * @param Configuration Sets additional conditions for retrieving results.
     * @param Callback      Callback pass to API result.
     *
     * Example Usage:
     * @code
     *  void RequestSubscriptionsStatusSample(bool bIncludeExpiredSubscriptions)
     *  {
     *      FGamebasePurchasableConfiguration Configuration;
     *      Configuration.bIncludeExpiredSubscriptions = bIncludeExpiredSubscriptions;
     *      
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->GetPurchase()->RequestSubscriptionsStatus(Configuration, FGamebasePurchasableSubscriptionStatusDelegate::CreateLambda(
     *          [](const TArray<FGamebasePurchasableSubscriptionStatus>* statusList, const FGamebaseError* Error)
     *      {
     *          if (Gamebase::IsSuccess(Error))
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("RequestSubscriptionsStatus succeeded."));
     *              for (const FGamebasePurchasableSubscriptionStatus& statusInfo : *statusList)
     *              {
     *                  GAMEBASE_RESULT_LOG(" - itemSeq= %ld, price= %f, currency= %s, paymentSeq= %s, purchaseToken= %s",
     *                      statusInfo.itemSeq, statusInfo.price, *statusInfo.currency, *statusInfo.paymentSeq, *statusInfo.purchaseToken);
     *              }
     *          }
     *          else
     *          {
     *              UE_LOG(GamebaseTestResults, Display, TEXT("RequestActivatedPurchases failed."));
     *          }
     *      }));
     *  }
     * @endcode
     */
    virtual void RequestSubscriptionsStatus(const FGamebasePurchasableConfiguration& Configuration, const FGamebasePurchasableSubscriptionStatusDelegate& Callback) = 0;
};