#pragma once

#include "CoreMinimal.h"
#include "UObject/Interface.h"
#include "Types/GamebaseLoggerTypes.h"
#include "IGamebaseLogger.generated.h"

UINTERFACE()
class GAMEBASE_API UGamebaseLogger : public UInterface
{
    GENERATED_BODY()
};

class GAMEBASE_API IGamebaseLogger
{
    GENERATED_BODY()
 
public:
    /**
     * Initialize Logger SDK<br>
     * Send logs to a NHN Cloud Log & Crash server for data analysis.
     *
     * @param Configuration  The appKey of the Log & Crash project must be set in this parameter.
     */
    virtual void Initialize(const FGamebaseLoggerConfiguration& Configuration) = 0;

     /**
      * Send a log to Log & Crash Server for analyzing the string message with debug level.<br>
      * The levels are as follows: debug < info < warn < error < fatal<br>
      * Log & Crash Search allows fast and patterned search of any logs you want among a large volume.
      *
      * @param Message       Log messages to send to the Log & Crash server.
      * @param UserFields    Additional information you want to send Log & Crash server together.
      */
    virtual void Debug(const FString& Message, const TMap<FString, FString>& UserFields = TMap<FString, FString>()) = 0;

    /**
     * Send a log to Log & Crash Server for analyzing the string message with info level.<br>
     * The levels are as follows: debug < info < warn < error < fatal<br>
     * Log & Crash Search allows fast and patterned search of any logs you want among a large volume.
     *
     * @param Message       Log messages to send to the Log & Crash server.
     * @param UserFields    Additional information you want to send Log & Crash server together.
     */
    virtual void Info(const FString& Message, const TMap<FString, FString>& UserFields = TMap<FString, FString>()) = 0;

    /**
     * Send a log to Log & Crash Server for analyzing the string message with warn level.<br>
     * The levels are as follows: debug < info < warn < error < fatal<br>
     * Log & Crash Search allows fast and patterned search of any logs you want among a large volume.
     *
     * @param Message       Log messages to send to the Log & Crash server.
     * @param UserFields    Additional information you want to send Log & Crash server together.
     */
    virtual void Warn(const FString& Message, const TMap<FString, FString>& UserFields = TMap<FString, FString>()) = 0;

    /**
     * Send a log to Log & Crash Server for analyzing the string message with error level.<br>
     * The levels are as follows: debug < info < warn < error < fatal<br>
     * Log & Crash Search allows fast and patterned search of any logs you want among a large volume.
     *
     * @param Message       Log messages to send to the Log & Crash server.
     * @param UserFields    Additional information you want to send Log & Crash server together.
     */
    virtual void Error(const FString& Message, const TMap<FString, FString>& UserFields = TMap<FString, FString>()) = 0;

    /**
     * Send a log to Log & Crash Server for analyzing the string message with fatal level.<br>
     * The levels are as follows: debug < info < warn < error < fatal<br>
     * Log & Crash Search allows fast and patterned search of any logs you want among a large volume.
     *
     * @param Message       Log messages to send to the Log & Crash server.
     * @param UserFields    Additional information you want to send Log & Crash server together.
     */
    virtual void Fatal(const FString& Message, const TMap<FString, FString>& UserFields = TMap<FString, FString>()) = 0;

    /**
     * Used to predefine additional information.<br>
     * The value set here is sent together without being set every time.
     *
     * @param Key    The key of the additional information.
     * @param Value  The value of the additional information.
     */
    virtual void SetUserField(const FString& Key, const FString& Value) = 0;
    
    /**
     * Register as an event listener for logger
     */
    virtual void SetListener(const FGamebaseLoggerListener& Listener) = 0;
 
    /**
     * Register as a crash filter
     */
    virtual void SetCrashFilter(const FGamebaseLoggerCrashFilterDelegate& FilterDelegate) = 0;
};